"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.vpcBootstrap = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const context_1 = require("./context");
/**
 * @internal
 * Create a VPC with the provided CIDR and attach to it an Amazon S3 Gateway Vpc Endpoint
 * @param {Construct} scope The scope of the stack where the VPC will be created
 * @param {string} vpcCidr The cidr for vpc
 * @param {Key} logKmsKey The KMS key used to encrypt the VPC flow log
 * @param {RemovalPolicy} vpcFlowlogRemovalPolicy The removal policy for vpc flowlog in cloudwatch log
 * @param {string} eksClusterName The name used to tag the subnet and vpc
 * @param {string} emrAppName The name used to tag the subnet and vpc
 */
function vpcBootstrap(scope, vpcCidr, logKmsKey, vpcFlowlogRemovalPolicy, eksClusterName, emrAppName, vpcFlowLogRole) {
    const vpcMask = parseInt(vpcCidr.split('/')[1]);
    const smallestVpcCidr = 28;
    const removalPolicy = context_1.Context.revertRemovalPolicy(scope, vpcFlowlogRemovalPolicy);
    if (vpcMask > smallestVpcCidr) {
        throw new Error(`The VPC netmask should be at least 28, netmask provided is ${vpcMask}`);
    }
    // Calculate subnet masks based on VPC's mask
    const publicSubnetMask = vpcMask + 4;
    const privateSubnetMask = publicSubnetMask + 2; // twice as large as public subnet
    const vpc = new aws_ec2_1.Vpc(scope, 'DsfVpc', {
        ipAddresses: aws_ec2_1.IpAddresses.cidr(vpcCidr),
        maxAzs: 3,
        natGateways: 3,
        subnetConfiguration: [
            {
                cidrMask: publicSubnetMask,
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                cidrMask: privateSubnetMask,
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
        ],
    });
    //Create a loggroup name based on the purpose of the VPC, either used by emr on eks or emr serverless app
    const logGroupName = eksClusterName ? `/aws/emr-eks-vpc-flow/${eksClusterName}` : `/aws/emr-serverless-vpc/${aws_cdk_lib_1.Names.nodeUniqueId(scope.node)}`;
    const logGroupResourceId = eksClusterName ? 'EmrEksVpcFlowLog' : 'EmrServerlessVpcFlowLog';
    //Create VPC flow log for the VPC
    let vpcFlowLogLogGroup = new aws_logs_1.LogGroup(scope, `${logGroupResourceId}Group`, {
        logGroupName: logGroupName,
        encryptionKey: logKmsKey,
        retention: aws_logs_1.RetentionDays.ONE_WEEK,
        removalPolicy: removalPolicy,
    });
    //Allow vpc flowlog to access KMS key to encrypt logs
    logKmsKey.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        principals: [new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(scope).region}.amazonaws.com`)],
        actions: [
            'kms:Encrypt*',
            'kms:Decrypt*',
            'kms:ReEncrypt*',
            'kms:GenerateDataKey*',
            'kms:Describe*',
        ],
        conditions: {
            ArnLike: {
                'kms:EncryptionContext:aws:logs:arn': `arn:aws:logs:${aws_cdk_lib_1.Stack.of(scope).region}:${aws_cdk_lib_1.Stack.of(scope).account}:*`,
            },
        },
        resources: ['*'],
    }));
    //Setup the VPC flow logs
    const iamFlowLogRole = vpcFlowLogRole || new aws_iam_1.Role(scope, 'FlowLogRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
    });
    vpc.addFlowLog(`${logGroupResourceId}`, {
        destination: aws_ec2_1.FlowLogDestination.toCloudWatchLogs(vpcFlowLogLogGroup, iamFlowLogRole),
    });
    // Create a gateway endpoint for S3
    const s3GatewayVpcEndpoint = vpc.addGatewayEndpoint('DsfS3VpcEndpoint', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    if (eksClusterName) {
        // Add tags to subnets
        for (let subnet of [...vpc.publicSubnets, ...vpc.privateSubnets]) {
            aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName);
        }
        // Add tags to vpc
        aws_cdk_lib_1.Tags.of(vpc).add('karpenter.sh/discovery', eksClusterName);
    }
    if (emrAppName) {
        // Add tags to subnets
        for (let subnet of [...vpc.publicSubnets, ...vpc.privateSubnets]) {
            aws_cdk_lib_1.Tags.of(subnet).add('use-by', 'emr-serverless');
        }
        // Add tags to vpc
        aws_cdk_lib_1.Tags.of(vpc).add('use-by', 'emr-serverless');
    }
    const networkConfiguration = {
        vpc,
        s3GatewayVpcEndpoint,
        vpcFlowLogLogGroup,
        iamFlowLogRole,
    };
    return networkConfiguration;
}
exports.vpcBootstrap = vpcBootstrap;
//# sourceMappingURL=data:application/json;base64,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