import datetime as dt
from typing import List

import attr


@attr.s(frozen=True, slots=True)
class CommercialCard:
    name: str = attr.ib()  # HAPPY_CARD, YOUNG_PASS, ...
    number: str = attr.ib()


@attr.s(frozen=True, slots=True)
class FidelityCard:
    """
    Carte voyageur.
    """

    number: str = attr.ib()


@attr.s(frozen=True, slots=True)
class Passenger:
    """
    A passenger.
    """

    name: str = attr.ib()
    birthday: dt.date = attr.ib()
    commercial_card_type: str = attr.ib(default="")
    commercial_card_number: str = attr.ib(default="")
    fidelity_card_number: str = attr.ib(default="")
    # commercial_card: Optional[CommercialCard] = attr.ib(default=None)
    # fidelity_card: Optional[FidelityCard] = attr.ib(default=None)

    # mypy annotation
    __attrs_attrs__: List[attr.Attribute]

    @classmethod
    def dummy(cls) -> "Passenger":
        """
        Get a dummy passenger.
        """
        birthday = dt.date.today() - dt.timedelta(days=26 * 365)
        return cls(birthday=birthday, name="Default Passenger")

    @property
    def age(self) -> int:
        """
        Returns the age of the passenger on the current date, in years.
        """
        return ((dt.date.today() - self.birthday) / 365).days


# Valid commercial cards
# [
#   "ABONNEMENT_FORFAIT_1ERE",
#   "ABONNEMENT_FORFAIT_2NDE",
#   "ACCOMPAGNANT_ENFANT_PLUS",
#   "ACCOMPAGNANT_WEEK_END",
#   "ADULT_PASS_COMP",
#   "BAHN_CARD_25_1ERE",
#   "BAHN_CARD_25_2EME",
#   "BAHN_CARD_50_1ERE",
#   "BAHN_CARD_50_2EME",
#   "CFF_ABO_GENERAL_1ERE_CLASSE",
#   "CFF_ABO_GENERAL_2EME_CLASSE",
#   "CFF_AG_1_F1",
#   "CFF_AG_1_Q1",
#   "CFF_AG_2_F2",
#   "CFF_AG_2_Q2",
#   "CFF_DEMI_F1",
#   "CFF_DEMI_F2",
#   "CFF_DEMI_Q1",
#   "CFF_DEMI_Q2",
#   "CFF_DEMI_TARIF",
#   "CHILD_ON_KNEES",
#   "CHILD_PASS_COMP",
#   "ENFANT_FAMILLE",
#   "ENFANT_PLUS",
#   "ESCAPADES",
#   "FAMILLE_MILITAIRE",
#   "FAMILLE_NOMBREUSE_30PC",
#   "FAMILLE_NOMBREUSE_40PC",
#   "FAMILLE_NOMBREUSE_50PC",
#   "FAMILLE_NOMBREUSE_75PC",
#   "FAMILY_PASS",
#   "FORFAIT_BAMBIN",
#   "FREQUENCE_1",
#   "FREQUENCE_2",
#   "FREQUENCE_INTERMEDIAIRE_1",
#   "FREQUENCE_INTERMEDIAIRE_2",
#   "GU_RPG_FREE",
#   "GU_RP_75",
#   "HAPPY_CARD",
#   "JEUNE",
#   "LIBERTE_PASS_COMP",
#   "LIBERTE_PASS_FE",
#   "MILITAIRE_1ERE",
#   "MILITAIRE_2NDE",
#   "NO_CARD",
#   "ODS_ABONNEMENT_BOURGOGNE_FREQUENCE_MENSUELANNUEL_26",
#   "ODS_ABONNEMENT_BOURGOGNE_FREQUENCE_MENSUELANNUEL_TOUT_PUBLIC",
#   "ODS_ABONNEMENT_INTERNE_ETUDIANT",
#   "ODS_ABONNEMENT_IZY_PASS_EN_COURS_DE_VALIDITE",
#   "ODS_ABONNEMENT_MODALIS_EN_COURS_DE_VALIDITE",
#   "ODS_ABONNEMENT_PASS_MOBILITE_EN_COURS_DE_VALIDITE",
#   "ODS_ABONNEMENT_PRATIK",
#   "ODS_ABONNEMENT_PRATIK26_ANS",
#   "ODS_ABONNEMENT_SCOLAIRE_EN_COURS_DE_VALIDITE",
#   "ODS_ABONNEMENT_TER_PC_EN_COURS_DE_VALIDITE",
#   "ODS_ABONNE_TER_LORRAINE",
#   "ODS_ACCOMPAGNANT_CARTE_26",
#   "ODS_ACCOMPAGNANT_CARTE_ASTUCES",
#   "ODS_ACCOMPAGNANT_CARTE_AVANTAGES",
#   "ODS_ACCOMPAGNANT_CARTE_AVANTAGES_EMPLOI_TER_POITOU_CH",
#   "ODS_ACCOMPAGNANT_CARTE_FIFTI_50",
#   "ODS_ACCOMPAGNANT_CARTE_IZY_AIR",
#   "ODS_ACCOMPAGNANT_CARTE_MEZZO",
#   "ODS_ACCOMPAGNANT_CARTE_MEZZO_26",
#   "ODS_ACCOMPAGNANT_CARTE_MOBITER",
#   "ODS_ACCOMPAGNANT_CARTE_MOINS26_ANS",
#   "ODS_ACCOMPAGNANT_CARTE_PLUS26_ANS",
#   "ODS_ACCOMPAGNANT_CARTE_PRESTO_GRAND_EST",
#   "ODS_ACCOMPAGNANT_CARTE_TIVA_50",
#   "ODS_ACCOMPAGNANT_FORFAIT_TUTTI",
#   "ODS_ACCOMPAGNANT_FORFAIT_TUTTI_MOINS26",
#   "ODS_ACCOMPAGNANT_PRATIK",
#   "ODS_ACCOMPAGNANT_PRATIK_26_ANS",
#   "ODS_ACCOMPAGNATEUR_CARTE_ZOU_MOINS_DE_26_ANS",
#   "ODS_ACCOMPAGNATEUR_CARTE_ZOU_PLUS_DE_26_ANS",
#   "ODS_CARTASTUCES",
#   "ODS_CARTE_26",
#   "ODS_CARTE_ACTI",
#   "ODS_CARTE_ACTI_75",
#   "ODS_CARTE_ACTUEL",
#   "ODS_CARTE_ASTUCES",
#   "ODS_CARTE_AVANTAGES",
#   "ODS_CARTE_AVANTAGES_EMPLOI_TER_POITOU_CH",
#   "ODS_CARTE_AVANTAGES_TER_POITOU_CHARENTES",
#   "ODS_CARTE_BOOSTER",
#   "ODS_CARTE_BOURGOGNE_LIBERTE_26",
#   "ODS_CARTE_BOURGOGNE_LIBERTE_TOUT_PUBLIC",
#   "ODS_CARTE_CIJA_AQUITAINE_ETUDIANTS",
#   "ODS_CARTE_FIFTI_50",
#   "ODS_CARTE_HNGO_JEUNES",
#   "ODS_CARTE_ILLICO_LIBERTE",
#   "ODS_CARTE_ILLICO_LIBERTE_JEUNES",
#   "ODS_CARTE_IZY_AIR",
#   "ODS_CARTE_IZY_AIR_AQUITAINE",
#   "ODS_CARTE_JEUNES_TER_POITOU_CHARENTES",
#   "ODS_CARTE_JEUNES_TER_ROUEN_CAEN",
#   "ODS_CARTE_JEUNE_TER_CAEN_ROUEN",
#   "ODS_CARTE_KARTATOO",
#   "ODS_CARTE_KARTATOO_ETUDES",
#   "ODS_CARTE_LIBERTER",
#   "ODS_CARTE_LIBERTIO",
#   "ODS_CARTE_LIBERTIO_JEUNES",
#   "ODS_CARTE_LOISIRYS",
#   "ODS_CARTE_METROLOR",
#   "ODS_CARTE_METROLOR_JEUNE",
#   "ODS_CARTE_METROLOR_PLUS",
#   "ODS_CARTE_MEZZO",
#   "ODS_CARTE_MEZZO_26",
#   "ODS_CARTE_MOBI",
#   "ODS_CARTE_MOBITER",
#   "ODS_CARTE_MOINS26_ANS",
#   "ODS_CARTE_MULTIPASS",
#   "ODS_CARTE_PASS_EVASION",
#   "ODS_CARTE_PLUS",
#   "ODS_CARTE_PLUS26_ANS",
#   "ODS_CARTE_PLUS_ACCOMPAGNANT",
#   "ODS_CARTE_PRESTO_GRAND_EST",
#   "ODS_CARTE_PRIMO_GRAND_EST",
#   "ODS_CARTE_RECIPROCITE_BOURGOGNE_MOBITER",
#   "ODS_CARTE_RECIPROCITE_RHONE_ALPES_MOBITER",
#   "ODS_CARTE_REFLEXE",
#   "ODS_CARTE_TER_APPRENTI",
#   "ODS_CARTE_TER_BAC",
#   "ODS_CARTE_TER_HAUTS_DE_FRANCE",
#   "ODS_CARTE_TER_HAUTS_DE_FRANCE_MOINS_26_ANS",
#   "ODS_CARTE_TIVA_50",
#   "ODS_CARTE_TONUS",
#   "ODS_CARTE_TREMPLIN",
#   "ODS_CARTE_VIA_ETUDES",
#   "ODS_CARTE_VIA_LIBERTE",
#   "ODS_CARTE_VIA_PRO",
#   "ODS_CARTE_VISITER",
#   "ODS_CARTE_ZOU_50_75_MOINS_DE_26_ANS",
#   "ODS_CARTE_ZOU_50_75_PLUS_DE_26_ANS",
#   "ODS_CARTE_ZOU_COUPON_DOMICILE_ETUDES",
#   "ODS_CARTE_ZOU_ETUDES",
#   "ODS_COUPON_DOMICILE_ETUDE",
#   "ODS_FORFAIT_TUTTI",
#   "ODS_FORFAIT_TUTTI_MOINS26",
#   "ODS_HNGO_LIBERTE",
#   "ODS_PACK_ILLIMITE",
#   "ODS_PACK_ILLIMITE_JEUNES",
#   "ODS_PACK_LIBERTE",
#   "ODS_PACK_LIBERTE_JEUNES",
#   "OTHER",
#   "RAIL_PLUS_SENIOR",
#   "RAIL_PLUS_YOUNG",
#   "RPG_BLUE",
#   "RPG_BLUE_DOUBLE",
#   "RPG_RED",
#   "SENIOR",
#   "SENIOR_PASS",
#   "SENIOR_PLUS",
#   "THEPASS_BUSINESS",
#   "THEPASS_PREMIUM",
#   "THEPASS_WEEKEND",
#   "WEEKEND_PASS",
#   "WEEK_END",
#   "YOUNG_PASS",
#   "_12_25"
# ]
