import { Dialog, showDialog, showErrorMessage } from '@jupyterlab/apputils';
import { PathExt } from '@jupyterlab/coreutils';
import { Menu } from '@lumino/widgets';
import * as React from 'react';
import { hiddenButtonStyle } from '../style/ActionButtonStyle';
import { fileListWrapperClass } from '../style/FileListStyle';
import { addIcon, diffIcon, discardIcon, openIcon, removeIcon } from '../style/icons';
import { openListedFile } from '../utils';
import { ActionButton } from './ActionButton';
import { isDiffSupported } from './diff/Diff';
import { openDiffView } from './diff/DiffWidget';
import { FileItem } from './FileItem';
import { GitStage } from './GitStage';
export var CommandIDs;
(function (CommandIDs) {
    CommandIDs.gitFileOpen = 'git:context-open';
    CommandIDs.gitFileUnstage = 'git:context-unstage';
    CommandIDs.gitFileStage = 'git:context-stage';
    CommandIDs.gitFileTrack = 'git:context-track';
    CommandIDs.gitFileDiscard = 'git:context-discard';
    CommandIDs.gitFileDiffWorking = 'git:context-diffWorking';
    CommandIDs.gitFileDiffIndex = 'git:context-diffIndex';
    CommandIDs.gitIgnore = 'git:context-ignore';
    CommandIDs.gitIgnoreExtension = 'git:context-ignoreExtension';
})(CommandIDs || (CommandIDs = {}));
export class FileList extends React.Component {
    constructor(props) {
        super(props);
        /** Handle right-click on a staged file */
        this.contextMenuStaged = (event) => {
            event.preventDefault();
            this._contextMenuStaged.open(event.clientX, event.clientY);
        };
        /** Handle right-click on an unstaged file */
        this.contextMenuUnstaged = (event) => {
            event.preventDefault();
            this._contextMenuUnstaged.open(event.clientX, event.clientY);
        };
        /** Handle right-click on an untracked file */
        this.contextMenuUntracked = (event) => {
            event.preventDefault();
            const extension = PathExt.extname(this.state.selectedFile.to);
            if (extension.length > 0) {
                this._contextMenuUntracked.open(event.clientX, event.clientY);
            }
            else {
                this._contextMenuUntrackedMin.open(event.clientX, event.clientY);
            }
        };
        /** Handle right-click on an untracked file in Simple mode*/
        this.contextMenuSimpleUntracked = (event) => {
            event.preventDefault();
            const extension = PathExt.extname(this.state.selectedFile.to);
            if (extension.length > 0) {
                this._contextMenuSimpleUntracked.open(event.clientX, event.clientY);
            }
            else {
                this._contextMenuSimpleUntrackedMin.open(event.clientX, event.clientY);
            }
        };
        /** Handle right-click on an tracked file in Simple mode*/
        this.contextMenuSimpleTracked = (event) => {
            event.preventDefault();
            this._contextMenuSimpleTracked.open(event.clientX, event.clientY);
        };
        /** Reset all staged files */
        this.resetAllStagedFiles = async () => {
            await this.props.model.reset();
        };
        /** Reset a specific staged file */
        this.resetStagedFile = async (file) => {
            await this.props.model.reset(file);
        };
        /** Add all unstaged files */
        this.addAllUnstagedFiles = async () => {
            await this.props.model.addAllUnstaged();
        };
        /** Discard changes in all unstaged files */
        this.discardAllUnstagedFiles = async () => {
            const result = await showDialog({
                title: 'Discard all changes',
                body: 'Are you sure you want to permanently discard changes to all files? This action cannot be undone.',
                buttons: [Dialog.cancelButton(), Dialog.warnButton({ label: 'Discard' })]
            });
            if (result.button.accept) {
                try {
                    await this.props.model.checkout();
                }
                catch (reason) {
                    showErrorMessage('Discard all unstaged changes failed.', reason);
                }
            }
        };
        /** Discard changes in all unstaged and staged files */
        this.discardAllChanges = async () => {
            const result = await showDialog({
                title: 'Discard all changes',
                body: 'Are you sure you want to permanently discard changes to all files? This action cannot be undone.',
                buttons: [Dialog.cancelButton(), Dialog.warnButton({ label: 'Discard' })]
            });
            if (result.button.accept) {
                try {
                    await this.props.model.resetToCommit();
                }
                catch (reason) {
                    showErrorMessage('Discard all changes failed.', reason);
                }
            }
        };
        /** Add a specific unstaged file */
        this.addFile = async (...file) => {
            await this.props.model.add(...file);
        };
        /** Discard changes in a specific unstaged or staged file */
        this.discardChanges = async (file) => {
            const result = await showDialog({
                title: 'Discard changes',
                body: (React.createElement("span", null,
                    "Are you sure you want to permanently discard changes to",
                    ' ',
                    React.createElement("b", null, file.to),
                    "? This action cannot be undone.")),
                buttons: [Dialog.cancelButton(), Dialog.warnButton({ label: 'Discard' })]
            });
            if (result.button.accept) {
                try {
                    if (file.status === 'staged' || file.status === 'partially-staged') {
                        await this.props.model.reset(file.to);
                    }
                    if (file.status === 'unstaged' ||
                        (file.status === 'partially-staged' && file.x !== 'A')) {
                        // resetting an added file moves it to untracked category => checkout will fail
                        await this.props.model.checkout({ filename: file.to });
                    }
                }
                catch (reason) {
                    showErrorMessage(`Discard changes for ${file.to} failed.`, reason, [
                        Dialog.warnButton({ label: 'DISMISS' })
                    ]);
                }
            }
        };
        /** Add all untracked files */
        this.addAllUntrackedFiles = async () => {
            await this.props.model.addAllUntracked();
        };
        this.addAllMarkedFiles = async () => {
            await this.addFile(...this.markedFiles.map(file => file.to));
        };
        this.updateSelectedFile = (file) => {
            this.setState({ selectedFile: file });
        };
        const commands = this.props.model.commands;
        this._contextMenuStaged = new Menu({ commands });
        this._contextMenuUnstaged = new Menu({ commands });
        this._contextMenuUntracked = new Menu({ commands });
        this._contextMenuUntrackedMin = new Menu({ commands });
        this._contextMenuSimpleUntracked = new Menu({ commands });
        this._contextMenuSimpleUntrackedMin = new Menu({ commands });
        this._contextMenuSimpleTracked = new Menu({ commands });
        this.state = {
            selectedFile: null
        };
        if (!commands.hasCommand(CommandIDs.gitFileOpen)) {
            commands.addCommand(CommandIDs.gitFileOpen, {
                label: 'Open',
                caption: 'Open selected file',
                execute: async () => {
                    await openListedFile(this.state.selectedFile, this.props.model);
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitFileDiffWorking)) {
            commands.addCommand(CommandIDs.gitFileDiffWorking, {
                label: 'Diff',
                caption: 'Diff selected file',
                execute: async () => {
                    await openDiffView(this.state.selectedFile.to, this.props.model, {
                        currentRef: { specialRef: 'WORKING' },
                        previousRef: { gitRef: 'HEAD' }
                    }, this.props.renderMime, !this.state.selectedFile.is_binary);
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitFileDiffIndex)) {
            commands.addCommand(CommandIDs.gitFileDiffIndex, {
                label: 'Diff',
                caption: 'Diff selected file',
                execute: async () => {
                    await openDiffView(this.state.selectedFile.to, this.props.model, {
                        currentRef: { specialRef: 'INDEX' },
                        previousRef: { gitRef: 'HEAD' }
                    }, this.props.renderMime, !this.state.selectedFile.is_binary);
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitFileStage)) {
            commands.addCommand(CommandIDs.gitFileStage, {
                label: 'Stage',
                caption: 'Stage the changes of selected file',
                execute: () => {
                    this.addFile(this.state.selectedFile.to);
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitFileTrack)) {
            commands.addCommand(CommandIDs.gitFileTrack, {
                label: 'Track',
                caption: 'Start tracking selected file',
                execute: () => {
                    this.addFile(this.state.selectedFile.to);
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitFileUnstage)) {
            commands.addCommand(CommandIDs.gitFileUnstage, {
                label: 'Unstage',
                caption: 'Unstage the changes of selected file',
                execute: () => {
                    if (this.state.selectedFile.x !== 'D') {
                        this.resetStagedFile(this.state.selectedFile.to);
                    }
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitFileDiscard)) {
            commands.addCommand(CommandIDs.gitFileDiscard, {
                label: 'Discard',
                caption: 'Discard recent changes of selected file',
                execute: () => {
                    this.discardChanges(this.state.selectedFile);
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitIgnore)) {
            commands.addCommand(CommandIDs.gitIgnore, {
                label: () => 'Ignore this file (add to .gitignore)',
                caption: () => 'Ignore this file (add to .gitignore)',
                execute: async () => {
                    if (this.state.selectedFile) {
                        await this.props.model.ignore(this.state.selectedFile.to, false);
                    }
                }
            });
        }
        if (!commands.hasCommand(CommandIDs.gitIgnoreExtension)) {
            commands.addCommand(CommandIDs.gitIgnoreExtension, {
                label: 'Ignore this file extension (add to .gitignore)',
                caption: 'Ignore this file extension (add to .gitignore)',
                execute: async () => {
                    if (this.state.selectedFile) {
                        const extension = PathExt.extname(this.state.selectedFile.to);
                        if (extension.length > 0) {
                            const result = await showDialog({
                                title: 'Ignore file extension',
                                body: `Are you sure you want to ignore all ${extension} files within this git repository?`,
                                buttons: [
                                    Dialog.cancelButton(),
                                    Dialog.okButton({ label: 'Ignore' })
                                ]
                            });
                            if (result.button.label === 'Ignore') {
                                await this.props.model.ignore(this.state.selectedFile.to, true);
                            }
                        }
                    }
                }
            });
        }
        [
            CommandIDs.gitFileOpen,
            CommandIDs.gitFileUnstage,
            CommandIDs.gitFileDiffIndex
        ].forEach(command => {
            this._contextMenuStaged.addItem({ command });
        });
        [
            CommandIDs.gitFileOpen,
            CommandIDs.gitFileStage,
            CommandIDs.gitFileDiscard,
            CommandIDs.gitFileDiffWorking
        ].forEach(command => {
            this._contextMenuUnstaged.addItem({ command });
        });
        [
            CommandIDs.gitFileOpen,
            CommandIDs.gitFileTrack,
            CommandIDs.gitIgnore,
            CommandIDs.gitIgnoreExtension
        ].forEach(command => {
            this._contextMenuUntracked.addItem({ command });
        });
        [
            CommandIDs.gitFileOpen,
            CommandIDs.gitFileTrack,
            CommandIDs.gitIgnore
        ].forEach(command => {
            this._contextMenuUntrackedMin.addItem({ command });
        });
        [
            CommandIDs.gitFileOpen,
            CommandIDs.gitFileDiscard,
            CommandIDs.gitFileDiffWorking
        ].forEach(command => {
            this._contextMenuSimpleTracked.addItem({ command });
        });
        [
            CommandIDs.gitFileOpen,
            CommandIDs.gitIgnore,
            CommandIDs.gitIgnoreExtension
        ].forEach(command => {
            this._contextMenuSimpleUntracked.addItem({ command });
        });
        [CommandIDs.gitFileOpen, CommandIDs.gitIgnore].forEach(command => {
            this._contextMenuSimpleUntrackedMin.addItem({ command });
        });
    }
    get markedFiles() {
        return this.props.files.filter(file => this.props.model.getMark(file.to));
    }
    render() {
        if (this.props.settings.composite['simpleStaging']) {
            return (React.createElement("div", { className: fileListWrapperClass }, this._renderSimpleStage(this.props.files)));
        }
        else {
            const stagedFiles = [];
            const unstagedFiles = [];
            const untrackedFiles = [];
            this.props.files.forEach(file => {
                switch (file.status) {
                    case 'staged':
                        stagedFiles.push(file);
                        break;
                    case 'unstaged':
                        unstagedFiles.push(file);
                        break;
                    case 'untracked':
                        untrackedFiles.push(file);
                        break;
                    case 'partially-staged':
                        stagedFiles.push(Object.assign(Object.assign({}, file), { status: 'staged' }));
                        unstagedFiles.push(Object.assign(Object.assign({}, file), { status: 'unstaged' }));
                        break;
                    default:
                        break;
                }
            });
            return (React.createElement("div", { className: fileListWrapperClass, onContextMenu: event => event.preventDefault() },
                this._renderStaged(stagedFiles),
                this._renderChanged(unstagedFiles),
                this._renderUntracked(untrackedFiles)));
        }
    }
    _isSelectedFile(candidate) {
        if (this.state.selectedFile === null) {
            return false;
        }
        return (this.state.selectedFile.x === candidate.x &&
            this.state.selectedFile.y === candidate.y &&
            this.state.selectedFile.from === candidate.from &&
            this.state.selectedFile.to === candidate.to &&
            this.state.selectedFile.status === candidate.status);
    }
    _renderStaged(files) {
        const doubleClickDiff = this.props.settings.get('doubleClickDiff')
            .composite;
        return (React.createElement(GitStage, { actions: React.createElement(ActionButton, { className: hiddenButtonStyle, disabled: files.length === 0, icon: removeIcon, title: 'Unstage all changes', onClick: this.resetAllStagedFiles }), collapsible: true, heading: 'Staged', nFiles: files.length }, files.map((file) => {
            const openFile = () => {
                openListedFile(file, this.props.model);
            };
            const diffButton = this._createDiffButton(file, 'INDEX');
            return (React.createElement(FileItem, { key: file.to, actions: React.createElement(React.Fragment, null,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: openIcon, title: 'Open this file', onClick: openFile }),
                    diffButton,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: removeIcon, title: 'Unstage this change', onClick: () => {
                            this.resetStagedFile(file.to);
                        } })), file: file, contextMenu: this.contextMenuStaged, model: this.props.model, selected: this._isSelectedFile(file), selectFile: this.updateSelectedFile, onDoubleClick: doubleClickDiff
                    ? diffButton
                        ? () => this._openDiffView(file, 'INDEX')
                        : () => undefined
                    : openFile }));
        })));
    }
    _renderChanged(files) {
        const doubleClickDiff = this.props.settings.get('doubleClickDiff')
            .composite;
        const disabled = files.length === 0;
        return (React.createElement(GitStage, { actions: React.createElement(React.Fragment, null,
                React.createElement(ActionButton, { className: hiddenButtonStyle, disabled: disabled, icon: discardIcon, title: 'Discard All Changes', onClick: this.discardAllUnstagedFiles }),
                React.createElement(ActionButton, { className: hiddenButtonStyle, disabled: disabled, icon: addIcon, title: 'Stage all changes', onClick: this.addAllUnstagedFiles })), collapsible: true, heading: 'Changed', nFiles: files.length }, files.map((file) => {
            const openFile = () => {
                openListedFile(file, this.props.model);
            };
            const diffButton = this._createDiffButton(file, 'WORKING');
            return (React.createElement(FileItem, { key: file.to, actions: React.createElement(React.Fragment, null,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: openIcon, title: 'Open this file', onClick: openFile }),
                    diffButton,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: discardIcon, title: 'Discard changes', onClick: () => {
                            this.discardChanges(file);
                        } }),
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: addIcon, title: 'Stage this change', onClick: () => {
                            this.addFile(file.to);
                        } })), file: file, contextMenu: this.contextMenuUnstaged, model: this.props.model, selected: this._isSelectedFile(file), selectFile: this.updateSelectedFile, onDoubleClick: doubleClickDiff
                    ? diffButton
                        ? () => this._openDiffView(file, 'WORKING')
                        : () => undefined
                    : openFile }));
        })));
    }
    _renderUntracked(files) {
        const doubleClickDiff = this.props.settings.get('doubleClickDiff')
            .composite;
        return (React.createElement(GitStage, { actions: React.createElement(ActionButton, { className: hiddenButtonStyle, disabled: files.length === 0, icon: addIcon, title: 'Track all untracked files', onClick: this.addAllUntrackedFiles }), collapsible: true, heading: 'Untracked', nFiles: files.length }, files.map((file) => {
            return (React.createElement(FileItem, { key: file.to, actions: React.createElement(React.Fragment, null,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: openIcon, title: 'Open this file', onClick: () => {
                            openListedFile(file, this.props.model);
                        } }),
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: addIcon, title: 'Track this file', onClick: () => {
                            this.addFile(file.to);
                        } })), file: file, contextMenu: this.contextMenuUntracked, model: this.props.model, onDoubleClick: () => {
                    if (!doubleClickDiff) {
                        openListedFile(file, this.props.model);
                    }
                }, selected: this._isSelectedFile(file), selectFile: this.updateSelectedFile }));
        })));
    }
    _renderSimpleStage(files) {
        const doubleClickDiff = this.props.settings.get('doubleClickDiff')
            .composite;
        return (React.createElement(GitStage, { actions: React.createElement(ActionButton, { className: hiddenButtonStyle, disabled: files.length === 0, icon: discardIcon, title: 'Discard All Changes', onClick: this.discardAllChanges }), heading: 'Changed', nFiles: files.length }, files.map((file) => {
            const openFile = () => {
                openListedFile(file, this.props.model);
            };
            // Default value for actions and double click
            let actions = (React.createElement(ActionButton, { className: hiddenButtonStyle, icon: openIcon, title: 'Open this file', onClick: openFile }));
            let onDoubleClick = doubleClickDiff
                ? () => undefined
                : openFile;
            let contextMenu = this.contextMenuSimpleUntracked;
            if (file.status === 'unstaged' ||
                file.status === 'partially-staged') {
                const diffButton = this._createDiffButton(file, 'WORKING');
                actions = (React.createElement(React.Fragment, null,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: openIcon, title: 'Open this file', onClick: openFile }),
                    diffButton,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: discardIcon, title: 'Discard changes', onClick: () => {
                            this.discardChanges(file);
                        } })));
                onDoubleClick = doubleClickDiff
                    ? diffButton
                        ? () => this._openDiffView(file, 'WORKING')
                        : () => undefined
                    : openFile;
                contextMenu = this.contextMenuSimpleTracked;
            }
            else if (file.status === 'staged') {
                const diffButton = this._createDiffButton(file, 'INDEX');
                actions = (React.createElement(React.Fragment, null,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: openIcon, title: 'Open this file', onClick: openFile }),
                    diffButton,
                    React.createElement(ActionButton, { className: hiddenButtonStyle, icon: discardIcon, title: 'Discard changes', onClick: () => {
                            this.discardChanges(file);
                        } })));
                onDoubleClick = doubleClickDiff
                    ? diffButton
                        ? () => this._openDiffView(file, 'INDEX')
                        : () => undefined
                    : openFile;
                contextMenu = this.contextMenuSimpleTracked;
            }
            return (React.createElement(FileItem, { key: file.to, actions: actions, file: file, markBox: true, model: this.props.model, onDoubleClick: onDoubleClick, contextMenu: contextMenu, selectFile: this.updateSelectedFile }));
        })));
    }
    /**
     * Creates a button element which, depending on the settings, is used
     * to either request a diff of the file, or open the file
     *
     * @param path File path of interest
     * @param currentRef the ref to diff against the git 'HEAD' ref
     */
    _createDiffButton(file, currentRef) {
        return ((isDiffSupported(file.to) || !file.is_binary) && (React.createElement(ActionButton, { className: hiddenButtonStyle, icon: diffIcon, title: 'Diff this file', onClick: () => this._openDiffView(file, currentRef) })));
    }
    /**
     * Returns a callback which opens a diff of the file
     *
     * @param file File to open diff for
     * @param currentRef the ref to diff against the git 'HEAD' ref
     */
    async _openDiffView(file, currentRef) {
        try {
            await openDiffView(file.to, this.props.model, {
                previousRef: { gitRef: 'HEAD' },
                currentRef: { specialRef: currentRef }
            }, this.props.renderMime, !file.is_binary);
        }
        catch (reason) {
            console.error(`Failed to open diff view for ${file.to}.\n${reason}`);
        }
    }
}
//# sourceMappingURL=FileList.js.map