import { showDialog } from '@jupyterlab/apputils';
import { PathExt } from '@jupyterlab/coreutils';
import { caretDownIcon, caretUpIcon, refreshIcon } from '@jupyterlab/ui-components';
import * as React from 'react';
import { classes } from 'typestyle';
import { CommandIDs } from '../commandsAndMenu';
import { branchIcon, desktopIcon, pullIcon, pushIcon, tagIcon } from '../style/icons';
import { spacer, toolbarButtonClass, toolbarClass, toolbarMenuButtonClass, toolbarMenuButtonEnabledClass, toolbarMenuButtonIconClass, toolbarMenuButtonSubtitleClass, toolbarMenuButtonTitleClass, toolbarMenuButtonTitleWrapperClass, toolbarMenuWrapperClass, toolbarNavClass } from '../style/Toolbar';
import { sleep } from '../utils';
import { GitTagDialog } from '../widgets/TagList';
import { ActionButton } from './ActionButton';
import { Alert } from './Alert';
import { BranchMenu } from './BranchMenu';
import { SuspendModal } from './SuspendModal';
/**
 * React component for rendering a panel toolbar.
 */
export class Toolbar extends React.Component {
    /**
     * Returns a React component for rendering a panel toolbar.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props) {
        super(props);
        /**
         * Callback invoked upon clicking a button to pull the latest changes.
         *
         * @param event - event object
         * @returns a promise which resolves upon pulling the latest changes
         */
        this._onPullClick = () => {
            this._suspend(true);
            const commands = this.props.model.commands;
            if (commands) {
                commands.execute(CommandIDs.gitPull);
            }
            this._suspend(false);
        };
        /**
         * Callback invoked upon clicking a button to push the latest changes.
         *
         * @param event - event object
         * @returns a promise which resolves upon pushing the latest changes
         */
        this._onPushClick = () => {
            this._suspend(true);
            const commands = this.props.model.commands;
            if (commands) {
                commands.execute(CommandIDs.gitPush);
            }
            this._suspend(false);
        };
        /**
         * Callback invoked upon clicking a button to change the current repository.
         *
         * @param event - event object
         */
        this._onRepositoryClick = () => {
            // Toggle the repository menu:
            this.setState({
                repoMenu: !this.state.repoMenu
            });
        };
        /**
         * Callback invoked upon clicking a button to change the current branch.
         *
         * @param event - event object
         */
        this._onBranchClick = () => {
            // Toggle the branch menu:
            this.setState({
                branchMenu: !this.state.branchMenu
            });
        };
        /**
         * Callback invoked upon clicking a button to refresh a repository.
         *
         * @param event - event object
         * @returns a promise which resolves upon refreshing a repository
         */
        this._onRefreshClick = async () => {
            this._log({
                severity: 'info',
                message: 'Refreshing...'
            });
            this._suspend(true);
            await Promise.all([sleep(1000), this.props.refresh()]);
            this._suspend(false);
            this._log({
                severity: 'success',
                message: 'Successfully refreshed.'
            });
        };
        /**
         * Callback invoked upon clicking on the feedback modal.
         *
         * @param event - event object
         */
        this._onFeedbackModalClick = () => {
            this._suspend(false);
        };
        /**
         * Callback invoked upon closing a feedback alert.
         *
         * @param event - event object
         */
        this._onFeedbackAlertClose = () => {
            this.setState({
                alert: false
            });
        };
        /**
         * Callback invoked upon clicking a button to view tags.
         *
         * @param event - event object
         */
        this._onTagClick = async () => {
            const result = await showDialog({
                title: 'Checkout tag',
                body: new GitTagDialog(this.props.model)
            });
            if (result.button.accept) {
                this._log({
                    severity: 'info',
                    message: `Switching to ${result.value}...`
                });
                this._suspend(true);
                let response;
                try {
                    response = await this.props.model.checkoutTag(result.value);
                }
                catch (error) {
                    response = {
                        code: -1,
                        message: error.message || error
                    };
                }
                finally {
                    this._suspend(false);
                }
                if (response.code !== 0) {
                    console.error(response.message);
                    this._log({
                        severity: 'error',
                        message: `Fail to checkout tag ${result.value}`
                    });
                }
                else {
                    this._log({
                        severity: 'success',
                        message: `Switched to ${result.value}`
                    });
                }
            }
        };
        const repo = this.props.model.pathRepository;
        this.state = {
            branchMenu: false,
            repoMenu: false,
            repository: repo || '',
            branch: repo ? this.props.model.currentBranch.name : '',
            suspend: false,
            alert: false,
            log: {
                severity: 'info',
                message: ''
            }
        };
    }
    /**
     * Callback invoked immediately after mounting a component (i.e., inserting into a tree).
     */
    componentDidMount() {
        this._addListeners();
    }
    /**
     * Callback invoked when a component will no longer be mounted.
     */
    componentWillUnmount() {
        this._removeListeners();
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement("div", { className: toolbarClass },
            this._renderTopNav(),
            this._renderRepoMenu(),
            this._renderBranchMenu(),
            this._renderFeedback()));
    }
    /**
     * Renders the top navigation.
     *
     * @returns React element
     */
    _renderTopNav() {
        return (React.createElement("div", { className: toolbarNavClass },
            React.createElement("span", { className: spacer }),
            React.createElement(ActionButton, { className: toolbarButtonClass, icon: pullIcon, onClick: this._onPullClick, title: 'Pull latest changes' }),
            React.createElement(ActionButton, { className: toolbarButtonClass, icon: pushIcon, onClick: this._onPushClick, title: 'Push committed changes' }),
            React.createElement(ActionButton, { className: toolbarButtonClass, icon: tagIcon, onClick: this._onTagClick, title: 'Checkout a tag' }),
            React.createElement(ActionButton, { className: toolbarButtonClass, icon: refreshIcon, onClick: this._onRefreshClick, title: 'Refresh the repository to detect local and remote changes' })));
    }
    /**
     * Renders a repository menu.
     *
     * @returns React element
     */
    _renderRepoMenu() {
        return (React.createElement("div", { className: toolbarMenuWrapperClass },
            React.createElement("button", { disabled: true, className: toolbarMenuButtonClass, title: `Current repository: ${this.state.repository}`, onClick: this._onRepositoryClick },
                React.createElement(desktopIcon.react, { className: toolbarMenuButtonIconClass }),
                React.createElement("div", { className: toolbarMenuButtonTitleWrapperClass },
                    React.createElement("p", { className: toolbarMenuButtonTitleClass }, "Current Repository"),
                    React.createElement("p", { className: toolbarMenuButtonSubtitleClass }, PathExt.basename(this.state.repository)))),
            this.state.repoMenu ? null : null));
    }
    /**
     * Renders a branch menu.
     *
     * @returns React element
     */
    _renderBranchMenu() {
        if (!this.props.model.pathRepository) {
            return null;
        }
        return (React.createElement("div", { className: toolbarMenuWrapperClass },
            React.createElement("button", { className: classes(toolbarMenuButtonClass, toolbarMenuButtonEnabledClass), title: `Change the current branch: ${this.state.branch}`, onClick: this._onBranchClick },
                React.createElement(branchIcon.react, { className: toolbarMenuButtonIconClass }),
                React.createElement("div", { className: toolbarMenuButtonTitleWrapperClass },
                    React.createElement("p", { className: toolbarMenuButtonTitleClass }, "Current Branch"),
                    React.createElement("p", { className: toolbarMenuButtonSubtitleClass }, this.state.branch)),
                this.state.branchMenu ? (React.createElement(caretUpIcon.react, { className: toolbarMenuButtonIconClass })) : (React.createElement(caretDownIcon.react, { className: toolbarMenuButtonIconClass }))),
            this.state.branchMenu ? (React.createElement(BranchMenu, { model: this.props.model, branching: this.props.branching, suspend: this.props.suspend })) : null));
    }
    /**
     * Renders a component to provide UI feedback.
     *
     * @returns React element
     */
    _renderFeedback() {
        return (React.createElement(React.Fragment, null,
            React.createElement(SuspendModal, { open: this.props.suspend && this.state.suspend, onClick: this._onFeedbackModalClick }),
            React.createElement(Alert, { open: this.state.alert, message: this.state.log.message, severity: this.state.log.severity, onClose: this._onFeedbackAlertClose })));
    }
    /**
     * Adds model listeners.
     */
    _addListeners() {
        // When the HEAD changes, decent probability that we've switched branches:
        this.props.model.headChanged.connect(this._syncState, this);
        // When the status changes, we may have checked out a new branch (e.g., via the command-line and not via the extension) or changed repositories:
        this.props.model.statusChanged.connect(this._syncState, this);
    }
    /**
     * Removes model listeners.
     */
    _removeListeners() {
        this.props.model.headChanged.disconnect(this._syncState, this);
        this.props.model.statusChanged.disconnect(this._syncState, this);
    }
    /**
     * Syncs the component state with the underlying model.
     */
    _syncState() {
        const repo = this.props.model.pathRepository;
        this.setState({
            repository: repo || '',
            branch: repo ? this.props.model.currentBranch.name : ''
        });
    }
    /**
     * Sets the suspension state.
     *
     * @param bool - boolean indicating whether to suspend UI interaction
     */
    _suspend(bool) {
        if (this.props.suspend) {
            this.setState({
                suspend: bool
            });
        }
    }
    /**
     * Sets the current component log message.
     *
     * @param msg - log message
     */
    _log(msg) {
        this.setState({
            alert: true,
            log: msg
        });
    }
}
//# sourceMappingURL=Toolbar.js.map