import { PathExt } from '@jupyterlab/coreutils';
import { ServerConnection } from '@jupyterlab/services';
import { LinkedList } from '@lumino/collections';
import { Poll } from '@lumino/polling';
import { Signal } from '@lumino/signaling';
import { httpGitRequest } from './git';
import { decodeStage } from './utils';
// Default refresh interval (in milliseconds) for polling the current Git status (NOTE: this value should be the same value as in the plugin settings schema):
const DEFAULT_REFRESH_INTERVAL = 3000; // ms
/**
 * Class for creating a model for retrieving info from, and interacting with, a remote Git repository.
 */
export class GitExtension {
    /**
     * Returns an extension model.
     *
     * @param app - frontend application
     * @param settings - plugin settings
     * @returns extension model
     */
    constructor(serverRoot, app = null, docmanager = null, settings) {
        this._status = [];
        this._pathRepository = null;
        this._diffProviders = {};
        this._isDisposed = false;
        this._markerCache = new Markers(() => this._markChanged.emit());
        this._currentMarker = null;
        this._readyPromise = Promise.resolve();
        this._pendingReadyPromise = 0;
        this._taskList = new LinkedList();
        this._taskID = 0;
        this._headChanged = new Signal(this);
        this._markChanged = new Signal(this);
        this._repositoryChanged = new Signal(this);
        this._statusChanged = new Signal(this);
        this._logger = new Signal(this);
        const self = this;
        this._serverRoot = serverRoot;
        this._app = app;
        this._docmanager = docmanager;
        this._settings = settings || null;
        let interval;
        if (settings) {
            interval = settings.composite.refreshInterval;
            settings.changed.connect(onSettingsChange, this);
        }
        else {
            interval = DEFAULT_REFRESH_INTERVAL;
        }
        const poll = new Poll({
            factory: () => self.refresh(),
            frequency: {
                interval: interval,
                backoff: true,
                max: 300 * 1000
            },
            standby: 'when-hidden'
        });
        this._poll = poll;
        /**
         * Callback invoked upon a change to plugin settings.
         *
         * @private
         * @param settings - settings registry
         */
        function onSettingsChange(settings) {
            const freq = poll.frequency;
            poll.frequency = {
                interval: settings.composite.refreshInterval,
                backoff: freq.backoff,
                max: freq.max
            };
        }
    }
    /**
     * Branch list for the current repository.
     */
    get branches() {
        return this._branches;
    }
    /**
     * List of available Git commands.
     */
    get commands() {
        return this._app ? this._app.commands : null;
    }
    /**
     * The current repository branch.
     */
    get currentBranch() {
        return this._currentBranch;
    }
    /**
     * Boolean indicating whether the model has been disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Boolean indicating whether the model is ready.
     */
    get isReady() {
        return this._pendingReadyPromise === 0;
    }
    /**
     * Promise which fulfills when the model is ready.
     */
    get ready() {
        return this._readyPromise;
    }
    /**
     * Git repository path.
     *
     * ## Notes
     *
     * -   This is the full path of the top-level folder.
     * -   The return value is `null` if a repository path is not defined.
     */
    get pathRepository() {
        return this._pathRepository;
    }
    set pathRepository(v) {
        const change = {
            name: 'pathRepository',
            newValue: null,
            oldValue: this._pathRepository
        };
        if (v === null) {
            this._pendingReadyPromise += 1;
            this._readyPromise.then(() => {
                this._pathRepository = null;
                this._pendingReadyPromise -= 1;
                if (change.newValue !== change.oldValue) {
                    this.refresh().then(() => this._repositoryChanged.emit(change));
                }
            });
        }
        else {
            const currentReady = this._readyPromise;
            this._pendingReadyPromise += 1;
            this._readyPromise = Promise.all([currentReady, this.showTopLevel(v)])
                .then(r => {
                const results = r[1];
                if (results.code === 0) {
                    this._pathRepository = results.top_repo_path;
                    change.newValue = results.top_repo_path;
                }
                else {
                    this._pathRepository = null;
                }
                if (change.newValue !== change.oldValue) {
                    this.refresh().then(() => this._repositoryChanged.emit(change));
                }
            })
                .catch(reason => {
                console.error(`Fail to find Git top level for path ${v}.\n${reason}`);
            });
            void this._readyPromise.then(() => {
                this._pendingReadyPromise -= 1;
            });
        }
    }
    /**
     * The Jupyter front-end application shell.
     */
    get shell() {
        return this._app ? this._app.shell : null;
    }
    /**
     * A list of modified files.
     *
     * ## Notes
     *
     * -   The file list corresponds to the list of files from `git status`.
     */
    get status() {
        return this._status;
    }
    /**
     * A signal emitted when the `HEAD` of the Git repository changes.
     */
    get headChanged() {
        return this._headChanged;
    }
    /**
     * A signal emitted when the current marking of the Git repository changes.
     */
    get markChanged() {
        return this._markChanged;
    }
    /**
     * A signal emitted when the current Git repository changes.
     */
    get repositoryChanged() {
        return this._repositoryChanged;
    }
    /**
     * A signal emitted when the current status of the Git repository changes.
     */
    get statusChanged() {
        return this._statusChanged;
    }
    /**
     * A signal emitted whenever a model event occurs.
     */
    get logger() {
        return this._logger;
    }
    /**
     * Add one or more files to the repository staging area.
     *
     * ## Notes
     *
     * -   If no filename is provided, all files are added.
     *
     * @param filename - files to add
     * @returns promise which resolves upon adding files to the repository staging area
     */
    async add(...filename) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:add:files');
        try {
            response = await httpGitRequest('/git/add', 'POST', {
                add_all: !filename,
                filename: filename || '',
                top_repo_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this.refreshStatus();
        return Promise.resolve(response);
    }
    /**
     * Add all "unstaged" files to the repository staging area.
     *
     * @returns promise which resolves upon adding files to the repository staging area
     */
    async addAllUnstaged() {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:add:files:all_unstaged');
        try {
            response = await httpGitRequest('/git/add_all_unstaged', 'POST', {
                top_repo_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this.refreshStatus();
        return data;
    }
    /**
     * Add all untracked files to the repository staging area.
     *
     * @returns promise which resolves upon adding files to the repository staging area
     */
    async addAllUntracked() {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:add:files:all_untracked');
        try {
            response = await httpGitRequest('/git/add_all_untracked', 'POST', {
                top_repo_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this.refreshStatus();
        return data;
    }
    /**
     * Add a remote Git repository to the current repository.
     *
     * @param url - remote repository URL
     * @param name - remote name
     * @returns promise which resolves upon adding a remote
     */
    async addRemote(url, name) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve();
        }
        const tid = this._addTask('git:add:remote');
        try {
            response = await httpGitRequest('/git/remote/add', 'POST', {
                top_repo_path: path,
                url,
                name
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.text();
            throw new ServerConnection.ResponseError(response, data);
        }
    }
    /**
     * Retrieve the repository commit log.
     *
     * ## Notes
     *
     * -  This API can be used to implicitly check if the current folder is a Git repository.
     *
     * @param count - number of commits to retrieve
     * @returns promise which resolves upon retrieving the repository commit log
     */
    async allHistory(count = 25) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const tid = this._addTask('git:fetch:history');
        try {
            response = await httpGitRequest('/git/all_history', 'POST', {
                current_path: path,
                history_count: count
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.text();
            throw new ServerConnection.ResponseError(response, data);
        }
        return response.json();
    }
    /**
     * Checkout a branch.
     *
     * ## Notes
     *
     * -   If a branch name is provided, checkout the provided branch (with or without creating it)
     * -   If a filename is provided, checkout the file, discarding all changes.
     * -   If nothing is provided, checkout all files, discarding all changes.
     *
     * TODO: Refactor into separate endpoints for each kind of checkout request
     *
     * @param options - checkout options
     * @returns promise which resolves upon performing a checkout
     */
    async checkout(options) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const body = {
            checkout_branch: false,
            new_check: false,
            branchname: '',
            startpoint: '',
            checkout_all: true,
            filename: '',
            top_repo_path: path
        };
        if (options !== undefined) {
            if (options.branchname) {
                body.branchname = options.branchname;
                body.checkout_branch = true;
                body.new_check = options.newBranch === true;
                if (options.newBranch) {
                    body.startpoint = options.startpoint || this._currentBranch.name;
                }
            }
            else if (options.filename) {
                body.filename = options.filename;
                body.checkout_all = false;
            }
        }
        const tid = this._addTask('git:checkout');
        try {
            response = await httpGitRequest('/git/checkout', 'POST', body);
            if (response.ok) {
                if (body.checkout_branch) {
                    const files = (await this.changedFiles(this._currentBranch.name, body.branchname))['files'];
                    if (files) {
                        files.forEach(file => this._revertFile(file));
                    }
                }
                else {
                    this._revertFile(options.filename);
                }
            }
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        if (body.checkout_branch) {
            await this.refreshBranch();
            this._headChanged.emit();
        }
        else {
            this.refreshStatus();
        }
        return data;
    }
    /**
     * Clone a repository.
     *
     * @param path - local path into which the repository will be cloned
     * @param url - Git repository URL
     * @param auth - remote repository authentication information
     * @returns promise which resolves upon cloning a repository
     */
    async clone(path, url, auth) {
        let response;
        const obj = {
            current_path: path,
            clone_url: url,
            auth
        };
        const tid = this._addTask('git:clone');
        try {
            response = await httpGitRequest('/git/clone', 'POST', obj);
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Commit all staged file changes.
     *
     * @param message - commit message
     * @returns promise which resolves upon committing file changes
     */
    async commit(message) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:commit:create');
        try {
            response = await httpGitRequest('/git/commit', 'POST', {
                commit_msg: message,
                top_repo_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this.refreshStatus();
        this._headChanged.emit();
        return response;
    }
    /**
     * Get (or set) Git configuration options.
     *
     * @param options - configuration options to set
     * @returns promise which resolves upon either getting or setting configuration options
     */
    async config(options) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:config:' + (options ? 'set' : 'get'));
        try {
            response = await httpGitRequest('/git/config', 'POST', {
                path,
                options
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return response;
    }
    /**
     * Revert changes made after a specified commit.
     *
     * @param message - commit message
     * @param hash - commit identifier (hash)
     * @returns promise which resolves upon reverting changes
     */
    async revertCommit(message, hash) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:commit:revert');
        const files = (await this.changedFiles(null, null, hash + '^!'))['files'];
        try {
            response = await httpGitRequest('/git/delete_commit', 'POST', {
                commit_id: hash,
                top_repo_path: path
            });
            if (response.ok && files) {
                files.forEach(file => {
                    this._revertFile(file);
                });
            }
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        await this.commit(message);
        return response;
    }
    /**
     * Fetch commit information.
     *
     * @param hash - commit hash
     * @returns promise which resolves upon retrieving commit information
     */
    async detailedLog(hash) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const tid = this._addTask('git:fetch:commit_log');
        try {
            response = await httpGitRequest('/git/detailed_log', 'POST', {
                selected_hash: hash,
                current_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Initialize a new Git repository at a specified path.
     *
     * @param path - path at which initialize a Git repository
     * @returns promise which resolves upon initializing a Git repository
     */
    async init(path) {
        let response;
        const tid = this._addTask('git:init');
        try {
            response = await httpGitRequest('/git/init', 'POST', {
                current_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return response;
    }
    /**
     * Retrieve commit logs.
     *
     * @param count - number of commits
     * @returns promise which resolves upon retrieving commit logs
     */
    async log(count = 25) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const tid = this._addTask('git:fetch:log');
        try {
            response = await httpGitRequest('/git/log', 'POST', {
                current_path: path,
                history_count: count
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Fetch changes from a remote repository.
     *
     * @param auth - remote authentication information
     * @returns promise which resolves upon fetching changes
     */
    async pull(auth) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const obj = {
            current_path: path,
            auth,
            cancel_on_conflict: this._settings
                ? this._settings.composite['cancelPullMergeConflict']
                : false
        };
        const tid = this._addTask('git:pull');
        try {
            response = await httpGitRequest('/git/pull', 'POST', obj);
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this._headChanged.emit();
        return data;
    }
    /**
     * Push local changes to a remote repository.
     *
     * @param auth - remote authentication information
     * @returns promise which resolves upon pushing changes
     */
    async push(auth) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const obj = {
            current_path: path,
            auth
        };
        const tid = this._addTask('git:push');
        try {
            response = await httpGitRequest('/git/push', 'POST', obj);
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this._headChanged.emit();
        return data;
    }
    /**
     * Refresh the repository.
     *
     * @returns promise which resolves upon refreshing the repository
     */
    async refresh() {
        const tid = this._addTask('git:refresh');
        await this.refreshBranch();
        await this.refreshStatus();
        this._removeTask(tid);
    }
    /**
     * Refresh the list of repository branches.
     *
     * @returns promise which resolves upon refreshing repository branches
     */
    async refreshBranch() {
        const tid = this._addTask('git:refresh:branches');
        const response = await this._branch();
        if (response.code === 0) {
            this._branches = response.branches;
            this._currentBranch = response.current_branch;
            if (this._currentBranch) {
                // Set up the marker obj for the current (valid) repo/branch combination
                this._setMarker(this.pathRepository, this._currentBranch.name);
            }
        }
        else {
            this._branches = [];
            this._currentBranch = null;
        }
        this._removeTask(tid);
    }
    /**
     * Refresh the repository status.
     *
     * @returns promise which resolves upon refreshing the repository status
     */
    async refreshStatus() {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            this._setStatus([]);
            return Promise.resolve();
        }
        const tid = this._addTask('git:refresh:status');
        try {
            response = await httpGitRequest('/git/status', 'POST', {
                current_path: path
            });
        }
        catch (err) {
            // TODO we should notify the user
            this._setStatus([]);
            console.error(err);
            return;
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            console.error(data.message);
            // TODO we should notify the user
            this._setStatus([]);
            return;
        }
        this._setStatus(data.files.map(file => {
            return Object.assign(Object.assign({}, file), { status: decodeStage(file.x, file.y) });
        }));
    }
    /**
     * Move files from the "staged" to the "unstaged" area.
     *
     * ## Notes
     *
     * -  If no filename is provided, moves all files from the "staged" to the "unstaged" area.
     *
     * @param filename - file path to be reset
     * @returns promise which resolves upon moving files
     */
    async reset(filename) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const tid = this._addTask('git:reset:changes');
        const reset_all = filename === undefined;
        let files;
        if (reset_all) {
            files = (await this.changedFiles('INDEX', 'HEAD'))['files'];
        }
        try {
            response = await httpGitRequest('/git/reset', 'POST', {
                reset_all: filename === undefined,
                filename: filename === undefined ? null : filename,
                top_repo_path: path
            });
            if (response.ok) {
                if (reset_all) {
                    if (files) {
                        files.forEach(file => {
                            this._revertFile(file);
                        });
                    }
                }
                else {
                    this._revertFile(filename);
                }
            }
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        this.refreshStatus();
        return response;
    }
    /**
     * Reset the repository to a specified commit.
     *
     * ## Notes
     *
     * -   If a commit hash is not provided, resets the repository to `HEAD`.
     *
     * @param hash - commit identifier (hash)
     * @returns promises which resolves upon resetting the repository
     */
    async resetToCommit(hash = '') {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(new Response(JSON.stringify(response)));
        }
        const files = (await this.changedFiles(null, null, hash))['files'];
        const tid = this._addTask('git:reset:hard');
        try {
            response = await httpGitRequest('/git/reset_to_commit', 'POST', {
                commit_id: hash,
                top_repo_path: path
            });
            if (response.ok) {
                if (files) {
                    files.forEach(file => {
                        this._revertFile(file);
                    });
                }
            }
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        if (!response.ok) {
            const data = await response.json();
            throw new ServerConnection.ResponseError(response, data.message);
        }
        await this.refreshBranch();
        this._headChanged.emit();
        return response;
    }
    /**
     * Retrieve the prefix path of a directory `path` with respect to the root repository directory.
     *
     * @param path - directory path
     * @returns promise which resolves upon retrieving the prefix path
     */
    async showPrefix(path) {
        let response;
        const tid = this._addTask('git:fetch:prefix_path');
        try {
            response = await httpGitRequest('/git/show_prefix', 'POST', {
                current_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Retrieve the top level repository path.
     *
     * @param path - current path
     * @returns promise which resolves upon retrieving the top level repository path
     */
    async showTopLevel(path) {
        let response;
        const tid = this._addTask('git:fetch:top_level_path');
        try {
            response = await httpGitRequest('/git/show_top_level', 'POST', {
                current_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Retrieve the list of tags in the repository.
     *
     * @returns promise which resolves upon retrieving the tag list
     */
    async tags() {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(response);
        }
        const tid = this._addTask('git:tag:list');
        try {
            response = await httpGitRequest('/git/tags', 'POST', {
                current_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Checkout the specified tag version
     *
     * @param tag - selected tag version
     * @returns promise which resolves upon checking out the tag version of the repository
     */
    async checkoutTag(tag) {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            response = {
                code: -1,
                message: 'Not in a Git repository.'
            };
            return Promise.resolve(response);
        }
        const tid = this._addTask('git:tag:checkout');
        try {
            response = await httpGitRequest('/git/tag_checkout', 'POST', {
                current_path: path,
                tag_id: tag
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Add a file to the current marker object.
     *
     * @param fname - filename
     * @param mark - mark to set
     */
    addMark(fname, mark) {
        this._currentMarker.add(fname, mark);
    }
    /**
     * Return the current mark associated with a specified filename.
     *
     * @param fname - filename
     * @returns mark
     */
    getMark(fname) {
        return this._currentMarker.get(fname);
    }
    /**
     * Toggle the mark for a file in the current marker object
     *
     * @param fname - filename
     */
    toggleMark(fname) {
        this._currentMarker.toggle(fname);
    }
    /**
     * Register a new diff provider for specified file types
     *
     * @param filetypes File type list
     * @param callback Callback to use for the provided file types
     */
    registerDiffProvider(filetypes, callback) {
        filetypes.forEach(fileType => {
            this._diffProviders[fileType] = callback;
        });
    }
    /**
     * Return the path of a file relative to the Jupyter server root.
     *
     * ## Notes
     *
     * -   If no path is provided, returns the Git repository top folder relative path.
     * -   If no Git repository selected, returns `null`
     *
     * @param path - file path relative to the top folder of the Git repository
     * @returns relative path
     */
    getRelativeFilePath(path) {
        if (this.pathRepository === null || this._serverRoot === void 0) {
            return null;
        }
        return PathExt.join(PathExt.relative(this._serverRoot, this.pathRepository), path || '');
    }
    /**
     * Dispose of model resources.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this._isDisposed = true;
        this._poll.dispose();
        Signal.clearData(this);
    }
    /**
     * Make request to ensure gitignore.
     *
     */
    async ensureGitignore() {
        await this.ready;
        const repositoryPath = this.pathRepository;
        if (repositoryPath === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        const response = await httpGitRequest('/git/ignore', 'POST', {
            top_repo_path: repositoryPath
        });
        this.refreshStatus();
        this._openGitignore();
        return Promise.resolve(response);
    }
    /**
     * Make request to ignore one file.
     *
     * @param filePath File to ignore
     * @param useExtension Whether to ignore the file or its extension
     */
    async ignore(filePath, useExtension) {
        await this.ready;
        const repositoryPath = this.pathRepository;
        if (repositoryPath === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        const response = await httpGitRequest('/git/ignore', 'POST', {
            top_repo_path: repositoryPath,
            file_path: filePath,
            use_extension: useExtension
        });
        this.refreshStatus();
        this._openGitignore();
        return Promise.resolve(response);
    }
    /**
     * Get list of files changed between two commits or two branches
     * @param base id of base commit or base branch for comparison
     * @param remote id of remote commit or remote branch for comparison
     * @param singleCommit id of a single commit
     *
     * @returns the names of the changed files
     */
    async changedFiles(base, remote, singleCommit) {
        try {
            const response = await httpGitRequest('/git/changed_files', 'POST', {
                current_path: this.pathRepository,
                base: base,
                remote: remote,
                single_commit: singleCommit
            });
            if (!response.ok) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /**
     * Make request for a list of all git branches in the repository
     * Retrieve a list of repository branches.
     *
     * @returns promise which resolves upon fetching repository branches
     */
    async _branch() {
        let response;
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a Git repository.'
            });
        }
        const tid = this._addTask('git:fetch:branches');
        try {
            response = await httpGitRequest('/git/branch', 'POST', {
                current_path: path
            });
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
        finally {
            this._removeTask(tid);
        }
        const data = await response.json();
        if (!response.ok) {
            throw new ServerConnection.ResponseError(response, data.message);
        }
        return data;
    }
    /**
     * Set the repository status.
     *
     * @param v - repository status
     */
    _setStatus(v) {
        this._status = v;
        this._statusChanged.emit(this._status);
    }
    /**
     * Set the marker object for a repository path and branch.
     *
     * @returns branch marker
     */
    _setMarker(path, branch) {
        this._currentMarker = this._markerCache.get(path, branch);
        return this._currentMarker;
    }
    /**
     * Adds a task to the list of pending model tasks.
     *
     * @param task - task name
     * @returns task identifier
     */
    _addTask(task) {
        // Generate a unique task identifier:
        const id = this._generateTaskID();
        // Add the task to our list of pending tasks:
        this._taskList.addLast({
            id: id,
            task: task
        });
        // If this task is the only task, broadcast the task...
        if (this._taskList.length === 1) {
            this._logger.emit(task);
        }
        // Return the task identifier to allow consumers to remove the task once completed:
        return id;
    }
    /**
     * Removes a task from the list of pending model tasks.
     *
     * @param id - task identifier
     */
    _removeTask(task) {
        let node = this._taskList.firstNode;
        // Check the first node...
        if (node && node.value.id === task) {
            this._taskList.removeNode(node);
        }
        else {
            // Walk the task list looking for a task with the provided identifier...
            while (node.next) {
                node = node.next;
                if (node.value && node.value.id === task) {
                    this._taskList.removeNode(node);
                    break;
                }
            }
        }
        // Check for pending tasks and broadcast the oldest pending task...
        if (this._taskList.length === 0) {
            this._logger.emit('git:idle');
        }
        else {
            this._logger.emit(this._taskList.first.task);
        }
    }
    /**
     * Generates a unique task identifier.
     *
     * @returns task identifier
     */
    _generateTaskID() {
        this._taskID += 1;
        return this._taskID;
    }
    /**
     * open new editor or show an existing editor of the
     * .gitignore file. If the editor does not have unsaved changes
     * then ensure the editor's content matches the file on disk
     */
    _openGitignore() {
        if (this._docmanager) {
            const widget = this._docmanager.openOrReveal(this.getRelativeFilePath('.gitignore'));
            if (widget && !widget.context.model.dirty) {
                widget.context.revert();
            }
        }
    }
    /**
     * if file is open in JupyterLab find the widget and ensure the JupyterLab
     * version matches the version on disk. Do nothing if the file has unsaved changes
     *
     * @param path path to the file to be reverted
     */
    _revertFile(path) {
        const widget = this._docmanager.findWidget(this.getRelativeFilePath(path));
        if (widget && !widget.context.model.dirty) {
            widget.context.revert();
        }
    }
}
export class BranchMarker {
    constructor(_refresh) {
        this._refresh = _refresh;
        this._marks = {};
    }
    add(fname, mark = true) {
        if (!(fname in this._marks)) {
            this.set(fname, mark);
        }
    }
    get(fname) {
        return this._marks[fname];
    }
    set(fname, mark) {
        this._marks[fname] = mark;
        this._refresh();
    }
    toggle(fname) {
        this.set(fname, !this._marks[fname]);
    }
}
export class Markers {
    constructor(_refresh) {
        this._refresh = _refresh;
        this._branchMarkers = {};
    }
    get(path, branch) {
        const key = Markers.markerKey(path, branch);
        if (key in this._branchMarkers) {
            return this._branchMarkers[key];
        }
        const marker = new BranchMarker(this._refresh);
        this._branchMarkers[key] = marker;
        return marker;
    }
    static markerKey(path, branch) {
        return [path, branch].join(':');
    }
}
//# sourceMappingURL=model.js.map