import { Spinner } from '@jupyterlab/apputils';
import { Widget } from '@lumino/widgets';
/**
 * The UI for the content shown within the Git push/pull modal.
 */
export class GitTagDialog extends Widget {
    /**
     * Instantiates the dialog and makes the relevant service API call.
     */
    constructor(model) {
        super();
        this._model = model;
        this._body = this._createBody();
        this.node.appendChild(this._body);
        this._spinner = new Spinner();
        this.node.appendChild(this._spinner.node);
        this._executeGitApi();
    }
    /**
     * Call the Git REST API
     */
    _executeGitApi() {
        this._model
            .tags()
            .then(response => {
            this._handleResponse(response);
        })
            .catch(error => this._handleError(error.message));
    }
    /**
     * Handles the response from the server by removing the _spinner and showing the appropriate
     * success or error message.
     *
     * @param response the response from the server API call
     */
    async _handleResponse(response) {
        if (response.code === 0) {
            this._handleSuccess(response);
        }
        else {
            this._handleError(response.message);
        }
    }
    /**
     * Handle failed Git tag REST API call
     *
     * @param message Error message
     */
    _handleError(message = 'Unexpected failure. Please check your Jupyter server logs for more details.') {
        this.node.removeChild(this._spinner.node);
        this._spinner.dispose();
        const label = document.createElement('label');
        const text = document.createElement('span');
        text.textContent = 'Tag list fetch failed with error:';
        const errorMessage = document.createElement('span');
        errorMessage.textContent = message;
        errorMessage.setAttribute('style', 'background-color:var(--jp-rendermime-error-background)');
        label.appendChild(text);
        label.appendChild(document.createElement('p'));
        label.appendChild(errorMessage);
        this._body.appendChild(label);
    }
    /**
     * Handle successful Git tag REST API call
     *
     * @param response Git REST API response
     */
    _handleSuccess(response) {
        this.node.removeChild(this._spinner.node);
        this._spinner.dispose();
        const label = document.createElement('label');
        const text = document.createElement('span');
        text.textContent = 'Select the tag to checkout : ';
        // Reverse the tag list to get the more recent on top
        const tags = response.tags.reverse();
        this._list = document.createElement('select');
        tags.forEach(tag => {
            if (tag) {
                const option = document.createElement('option');
                option.value = tag;
                option.textContent = tag;
                this._list.appendChild(option);
            }
        });
        label.appendChild(text);
        this._body.appendChild(label);
        this._body.appendChild(this._list);
    }
    /**
     * Create the dialog body node
     */
    _createBody() {
        const node = document.createElement('div');
        node.className = 'jp-RedirectForm';
        return node;
    }
    /**
     * Returns the input value.
     */
    getValue() {
        return this._list.value;
    }
}
//# sourceMappingURL=TagList.js.map