import { Spinner } from '@jupyterlab/apputils';
import { Widget } from '@lumino/widgets';
import { AUTH_ERROR_MESSAGES } from '../git';
export var Operation;
(function (Operation) {
    Operation["Pull"] = "Pull";
    Operation["Push"] = "Push";
})(Operation || (Operation = {}));
/**
 * The UI for the content shown within the Git push/pull modal.
 */
export class GitPullPushDialog extends Widget {
    /**
     * Instantiates the dialog and makes the relevant service API call.
     */
    constructor(model, operation, auth) {
        super();
        this._model = model;
        this._operation = operation;
        this._body = this.createBody();
        this.node.appendChild(this._body);
        this._spinner = new Spinner();
        this.node.appendChild(this._spinner.node);
        this.executeGitApi(auth);
    }
    /**
     * Executes the relevant service API depending on the _operation and handles response and errors.
     * @param currentFileBrowserPath the path to the current repo
     */
    executeGitApi(auth) {
        switch (this._operation) {
            case Operation.Pull:
                this._model
                    .pull(auth)
                    .then(response => {
                    this.handleResponse(response);
                })
                    .catch(error => this.handleError(error.message));
                break;
            case Operation.Push:
                this._model
                    .push(auth)
                    .then(response => {
                    this.handleResponse(response);
                })
                    .catch(error => this.handleError(error.message));
                break;
            default:
                throw new Error(`Invalid _operation type: ${this._operation}`);
        }
    }
    /**
     * Handles the response from the server by removing the _spinner and showing the appropriate
     * success or error message.
     * @param response the response from the server API call
     */
    async handleResponse(response) {
        if (response.code !== 0) {
            this.handleError(response.message);
        }
        else {
            this.handleSuccess();
        }
    }
    handleError(message = 'Unexpected failure. Please check your Jupyter server logs for more details.') {
        if (AUTH_ERROR_MESSAGES.map(errorMessage => message.indexOf(errorMessage) > -1).indexOf(true) > -1) {
            this.parent.parent.close(); // eslint-disable-line @typescript-eslint/no-non-null-assertion
        }
        this.node.removeChild(this._spinner.node);
        this._spinner.dispose();
        const label = document.createElement('label');
        const text = document.createElement('span');
        text.textContent = `Git ${this._operation} failed with error:`;
        const errorMessage = document.createElement('span');
        errorMessage.textContent = message;
        errorMessage.setAttribute('style', 'background-color:var(--jp-rendermime-error-background)');
        label.appendChild(text);
        label.appendChild(document.createElement('p'));
        label.appendChild(errorMessage);
        this._body.appendChild(label);
    }
    handleSuccess() {
        this.node.removeChild(this._spinner.node);
        this._spinner.dispose();
        const label = document.createElement('label');
        const text = document.createElement('span');
        text.textContent = `Git ${this._operation} completed successfully`;
        label.appendChild(text);
        this._body.appendChild(label);
    }
    createBody() {
        const node = document.createElement('div');
        node.className = 'jp-RedirectForm';
        return node;
    }
}
//# sourceMappingURL=gitPushPull.js.map