{
 "cells": [
  {
   "cell_type": "code",
   "execution_count": 1,
   "metadata": {},
   "outputs": [],
   "source": [
    "import os\n",
    "import binascii\n",
    "import base64\n",
    "from hashlib import sha256, sha512, sha384\n",
    "import hmac\n",
    "\n",
    "class MultisigHMAC:\n",
    "    \n",
    "    SHA256_PRIMITIVE = sha256\n",
    "    SHA256_KEYBYTES = sha256().block_size\n",
    "    SHA256_BYTES = sha256().digest_size\n",
    "\n",
    "    SHA512_PRIMITIVE = sha512\n",
    "    SHA512_KEYBYTES = sha512().block_size\n",
    "    SHA512_BYTES = sha512().digest_size\n",
    "\n",
    "    SHA384_PRIMITIVE = sha384\n",
    "    SHA384_KEYBYTES = sha384().block_size\n",
    "    SHA384_BYTES = sha384().digest_size\n",
    "\n",
    "    PRIMITIVE = SHA256_PRIMITIVE\n",
    "    KEYBYTES = SHA256_KEYBYTES\n",
    "    BYTES = SHA256_BYTES\n",
    "    \n",
    "    def __init__(self, alg = PRIMITIVE):\n",
    "        h = alg()\n",
    "        self.__bytes = h.digest_size\n",
    "        self.__keybytes = h.block_size\n",
    "        self.__alg = alg\n",
    "\n",
    "    def keygen(self, index): # generates a new cryptographically random key\n",
    "        key = os.urandom(self.__keybytes)\n",
    "        return (index,key)\n",
    "    \n",
    "    def seedgen(self): # generates a new cryptographically random master seed\n",
    "        return os.urandom(self.__keybytes)\n",
    "    \n",
    "    def deriveKey(self, masterSeed, index): # derives a new sub key from a master seed\n",
    "        data = bytearray(b'derived')\n",
    "        indexarray = index.to_bytes(length=4, byteorder='little')\n",
    "        _scratch = data + indexarray\n",
    "        h1 = hmac.new(masterSeed, _scratch, self.__alg) ; h1.update(bytearray([0])) ; h1.digest()\n",
    "        h2 = hmac.new(masterSeed, h1.digest(), self.__alg) ; h2.update(bytearray([1])) ; h2.digest()\n",
    "        return (index,h1.digest() + h2.digest())\n",
    "\n",
    "    def popcount(self, x): # counts 1-bits, corresponding to the number of keys\n",
    "        x = x - ((x >> 1) & 0x55555555)\n",
    "        x = (x & 0x33333333) + ((x >> 2) & 0x33333333)\n",
    "        x = (x + (x >> 4)) & 0x0F0F0F0F\n",
    "        x = x + (x >> 8)\n",
    "        x = x + (x >> 16)\n",
    "        return x & 0x0000003F\n",
    "\n",
    "    def keyIndexes(self, x): # x should be of type int. Returns the indexes of the keys\n",
    "        xs = []\n",
    "        i = 0\n",
    "        while(x > 0):\n",
    "            if(x & 0x1):\n",
    "                xs.append(i)\n",
    "            x >>= 1\n",
    "            i += 1\n",
    "        return xs\n",
    "\n",
    "    def nlz(self, x): # counts number of leading zeros\n",
    "        n = 32\n",
    "        c = 16\n",
    "        while(c != 0):\n",
    "            y = x >> c\n",
    "            if(y != 0):\n",
    "                n = n - c\n",
    "                x = y\n",
    "            c = c >> 1\n",
    "        return n - x\n",
    "\n",
    "    def xorBytes(self, a, b):\n",
    "        result = bytearray()\n",
    "        for b1, b2 in zip(a, b):\n",
    "            result.append(b1 ^ b2)\n",
    "        return result\n",
    "\n",
    "    def sign(self, keyObj, data): # signs data with key\n",
    "        assert type(data) == bytes, \"data must be bytes\"\n",
    "        digest = hmac.new(keyObj[1], data, self.__alg).digest()\n",
    "        return (1 << keyObj[0], digest)\n",
    "\n",
    "    def combine(self, signatures): # combines a list of signatures which have all been signed independently\n",
    "        bitfield = 0\n",
    "        sigs = bytearray(self.__bytes)\n",
    "        for i in signatures:\n",
    "            bitfield ^= i[0]\n",
    "            sigs = self.xorBytes(sigs, i[1])\n",
    "        assert self.popcount(bitfield) == len(signatures), \"one or more signatures cancelled out\"\n",
    "        return (bitfield, sigs)\n",
    "\n",
    "    def verify(self, keys, signature, data, threshold): # verifies signature of data against a list of keys\n",
    "        assert threshold > 0, \"threshold must be at least 1\"\n",
    "        assert type(data) == bytes, \"data must be bytes\" \n",
    "        bitfield = signature[0]\n",
    "        nKeys = self.popcount(bitfield)\n",
    "        highestKey = 32 - self.nlz(bitfield)\n",
    "        assert len(keys) >= nKeys and len(keys) >= highestKey, \"Not enough keys given based on signature[0]\"\n",
    "        \n",
    "        if (nKeys < threshold):\n",
    "            return False\n",
    "\n",
    "        usedKeys = self.keyIndexes(bitfield)\n",
    "        sig = signature[1]\n",
    "        for i in usedKeys:\n",
    "            key = keys[i]\n",
    "            keySig = self.sign(key, data)\n",
    "            sig = self.xorBytes(sig, keySig[1])\n",
    "\n",
    "            bitfield ^= keySig[0]\n",
    "\n",
    "        return (bitfield == 0 and sum(sig) == 0)\n",
    "    \n",
    "    def verifyDerived(self, masterSeed, signature, data, threshold): #verifies signature of data against derived keys\n",
    "        assert threshold > 0, \"threshold must be at least 1\"\n",
    "        assert type(data) == bytes, \"data must be bytes\" \n",
    "        bitfield = signature[0]\n",
    "        nKeys = self.popcount(bitfield)\n",
    "        \n",
    "        if (nKeys < threshold):\n",
    "            return False\n",
    "        \n",
    "        usedKeys = self.keyIndexes(bitfield)\n",
    "        sig = signature[1]\n",
    "        \n",
    "        for i in range(len(usedKeys)):\n",
    "            key = self.deriveKey(masterSeed, usedKeys[i])\n",
    "            keySig = self.sign(key, data)\n",
    "            sig = self.xorBytes(sig, keySig[1])\n",
    "            \n",
    "            bitfield ^= keySig[0]\n",
    "        \n",
    "        return (bitfield == 0 and sum(sig) == 0)"
   ]
  }
 ],
 "metadata": {
  "kernelspec": {
   "display_name": "Python 3",
   "language": "python",
   "name": "python3"
  },
  "language_info": {
   "codemirror_mode": {
    "name": "ipython",
    "version": 3
   },
   "file_extension": ".py",
   "mimetype": "text/x-python",
   "name": "python",
   "nbconvert_exporter": "python",
   "pygments_lexer": "ipython3",
   "version": "3.7.4"
  }
 },
 "nbformat": 4,
 "nbformat_minor": 2
}
