import typing

from django.db import models
from django.utils.translation import gettext_lazy as _
from phonenumber_field.modelfields import PhoneNumberField

from ob_dj_otp.core.otp.managers import OneTruePairingManager
from ob_dj_otp.core.otp.providers import SMSProvider


class OneTruePairing(models.Model):
    """ OTP represent attempts for login or registration using One True Pairing with SMS """

    class Statuses(models.TextChoices):
        init = "init", _("init")
        used = "used", _("used")

    class Usages(models.TextChoices):
        auth = "auth"
        register = "register"

    phone_number = PhoneNumberField()
    status = models.CharField(
        max_length=10, choices=Statuses.choices, default=Statuses.init
    )
    usage = models.CharField(max_length=10, choices=Usages.choices)
    user = models.ForeignKey(
        "users.User",
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        help_text=_("User field will be populated for OTP Auth"),
    )
    verification_code = models.CharField(
        max_length=10,
        help_text=_(
            "Verification code generated by backend for authentication or registration"
        ),
    )
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    objects = OneTruePairingManager()

    def __str__(self):
        return f"{self.phone_number.__str__()}"

    def mark_used(self) -> typing.NoReturn:
        self.status = self.Statuses.used
        self.save()

    def send_sms(self) -> typing.NoReturn:
        provider = SMSProvider()
        provider.send_message(
            phone_number=str(self.phone_number),
            verification_code=self.verification_code,
        )
