"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schedule = void 0;
/**
 * Schedule for scheduled event rules
 */
class Schedule {
    constructor() {
    }
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that EventBridge will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error('Cannot supply both \'day\' and \'weekDay\', use at most one');
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
}
exports.Schedule = Schedule;
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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