"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(id, resource) {
        this.id = id;
        this.resource = resource;
    }
    bind() {
        return {
            id: this.id || '',
            arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
            targetResource: this.resource,
        };
    }
}
module.exports = {
    'default rule'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyRuleA44AB831': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'ScheduleExpression': 'rate(10 minutes)',
                        'State': 'ENABLED',
                    },
                },
            },
        });
        test.done();
    },
    'can get rule name'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        new cdk.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                RuleName: rule.ruleName,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            RuleName: { Ref: 'MyRuleA44AB831' },
        }));
        test.done();
    },
    'rule with physical name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            ruleName: 'PhysicalName',
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Name: 'PhysicalName',
        }));
        test.done();
    },
    'eventPattern is rendered properly'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                account: ['account1', 'account2'],
                detail: {
                    foo: [1, 2],
                },
                detailType: ['detailType1'],
                id: ['id1', 'id2'],
                region: ['region1', 'region2', 'region3'],
                resources: ['r1'],
                source: ['src1', 'src2'],
                time: ['t1'],
                version: ['0'],
            },
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyRuleA44AB831': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'EventPattern': {
                            account: ['account1', 'account2'],
                            detail: { foo: [1, 2] },
                            'detail-type': ['detailType1'],
                            id: ['id1', 'id2'],
                            region: ['region1', 'region2', 'region3'],
                            resources: ['r1'],
                            source: ['src1', 'src2'],
                            time: ['t1'],
                            version: ['0'],
                        },
                        'State': 'ENABLED',
                    },
                },
            },
        });
        test.done();
    },
    'fails synthesis if neither eventPattern nor scheudleExpression are specified'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        new rule_1.Rule(stack, 'Rule');
        test.throws(() => app.synth(), /Either 'eventPattern' or 'schedule' must be defined/);
        test.done();
    },
    'addEventPattern can be used to add filters'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule');
        rule.addEventPattern({
            account: ['12345'],
            detail: {
                foo: ['hello'],
            },
        });
        rule.addEventPattern({
            source: ['aws.source'],
            detail: {
                foo: ['bar'],
                goo: {
                    hello: ['world'],
                },
            },
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyRuleA44AB831': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'EventPattern': {
                            'account': [
                                '12345',
                            ],
                            'detail': {
                                'foo': [
                                    'hello',
                                    'bar',
                                ],
                                'goo': {
                                    'hello': [
                                        'world',
                                    ],
                                },
                            },
                            'source': [
                                'aws.source',
                            ],
                        },
                        'State': 'ENABLED',
                    },
                },
            },
        });
        test.done();
    },
    'targets can be added via props or addTarget with input transformer'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
            }),
        };
        const t2 = {
            bind: () => ({
                id: '',
                arn: 'ARN2',
                input: lib_1.RuleTargetInput.fromText(`This is ${lib_1.EventField.fromPath('$.detail.bla')}`),
            }),
        };
        const rule = new rule_1.Rule(stack, 'EventRule', {
            targets: [t1],
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
        });
        rule.addTarget(t2);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'EventRule5A491D2C': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'ScheduleExpression': 'rate(5 minutes)',
                        'State': 'ENABLED',
                        'Targets': [
                            {
                                'Arn': 'ARN1',
                                'Id': 'Target0',
                                'KinesisParameters': {
                                    'PartitionKeyPath': 'partitionKeyPath',
                                },
                            },
                            {
                                'Arn': 'ARN2',
                                'Id': 'Target1',
                                'InputTransformer': {
                                    'InputPathsMap': {
                                        'detail-bla': '$.detail.bla',
                                    },
                                    'InputTemplate': '"This is <detail-bla>"',
                                },
                            },
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'input template can contain tokens'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '', arn: 'ARN2', input: lib_1.RuleTargetInput.fromText('Hello, "world"'),
            }),
        });
        // tokens are used here (FnConcat), but this is a text template so we
        // expect it to be wrapped in double quotes automatically for us.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join('', ['a', 'b']).toString()),
            }),
        });
        // jsonTemplate can be used to format JSON documents with replacements
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN3',
                input: lib_1.RuleTargetInput.fromObject({ foo: lib_1.EventField.fromPath('$.detail.bar') }),
            }),
        });
        // tokens can also used for JSON templates.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN4',
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join(' ', ['hello', '"world"']).toString()),
            }),
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'EventRule5A491D2C': {
                    'Type': 'AWS::Events::Rule',
                    'Properties': {
                        'State': 'ENABLED',
                        'ScheduleExpression': 'rate(1 minute)',
                        'Targets': [
                            {
                                'Arn': 'ARN2',
                                'Id': 'Target0',
                                'Input': '"Hello, \\"world\\""',
                            },
                            {
                                'Arn': 'ARN1',
                                'Id': 'Target1',
                                'Input': '"ab"',
                                'KinesisParameters': {
                                    'PartitionKeyPath': 'partitionKeyPath',
                                },
                            },
                            {
                                'Arn': 'ARN3',
                                'Id': 'Target2',
                                'InputTransformer': {
                                    'InputPathsMap': {
                                        'detail-bar': '$.detail.bar',
                                    },
                                    'InputTemplate': '{"foo":<detail-bar>}',
                                },
                            },
                            {
                                'Arn': 'ARN4',
                                'Id': 'Target3',
                                'Input': '"hello \\"world\\""',
                            },
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'target can declare role which will be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        const role = new iam.Role(stack, 'SomeRole', {
            assumedBy: new iam.ServicePrincipal('nobody'),
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN2',
                role,
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            'Targets': [
                {
                    'Arn': 'ARN2',
                    'Id': 'Target0',
                    'RoleArn': { 'Fn::GetAtt': ['SomeRole6DDC54DD', 'Arn'] },
                },
            ],
        }));
        test.done();
    },
    'asEventRuleTarget can use the ruleArn and a uniqueId of the rule'(test) {
        const stack = new cdk.Stack();
        let receivedRuleArn = 'FAIL';
        let receivedRuleId = 'FAIL';
        const t1 = {
            bind: (eventRule) => {
                receivedRuleArn = eventRule.ruleArn;
                receivedRuleId = eventRule.node.uniqueId;
                return {
                    id: '',
                    arn: 'ARN1',
                    kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
                };
            },
        };
        const rule = new rule_1.Rule(stack, 'EventRule');
        rule.addTarget(t1);
        test.deepEqual(stack.resolve(receivedRuleArn), stack.resolve(rule.ruleArn));
        test.deepEqual(receivedRuleId, rule.node.uniqueId);
        test.done();
    },
    'fromEventRuleArn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const importedRule = rule_1.Rule.fromEventRuleArn(stack, 'ImportedRule', 'arn:aws:events:us-east-2:123456789012:rule/example');
        // THEN
        test.deepEqual(importedRule.ruleArn, 'arn:aws:events:us-east-2:123456789012:rule/example');
        test.deepEqual(importedRule.ruleName, 'example');
        test.done();
    },
    'rule can be disabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            'State': 'DISABLED',
        }));
        test.done();
    },
    'can add multiple targets with the same id'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false,
        });
        rule.addTarget(new SomeTarget());
        rule.addTarget(new SomeTarget());
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    'Arn': 'ARN1',
                    'Id': 'Target0',
                    'KinesisParameters': {
                        'PartitionKeyPath': 'partitionKeyPath',
                    },
                },
                {
                    'Arn': 'ARN1',
                    'Id': 'Target1',
                    'KinesisParameters': {
                        'PartitionKeyPath': 'partitionKeyPath',
                    },
                },
            ],
        }));
        test.done();
    },
    'rule and target must be in the same region'(test) {
        const app = new cdk.App();
        const sourceStack = new cdk.Stack(app, 'SourceStack');
        const rule = new rule_1.Rule(sourceStack, 'Rule');
        const targetStack = new cdk.Stack(app, 'TargetStack', { env: { region: 'us-west-2' } });
        const resource = new cdk.Construct(targetStack, 'Resource');
        test.throws(() => {
            rule.addTarget(new SomeTarget('T', resource));
        }, /Rule and target must be in the same region/);
        test.done();
    },
    'sqsParameters are generated when they are specified in target props'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                sqsParameters: { messageGroupId: 'messageGroupId' },
            }),
        };
        new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
            targets: [t1],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    'Arn': 'ARN1',
                    'Id': 'Target0',
                    'SqsParameters': {
                        'MessageGroupId': 'messageGroupId',
                    },
                },
            ],
        }));
        test.done();
    },
    'associate rule with event bus'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                detail: ['detail'],
            },
            eventBus,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventBusName: {
                Ref: 'EventBus7B8748AA',
            },
        }));
        test.done();
    },
    'throws with eventBus and schedule'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // THEN
        test.throws(() => new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
            eventBus,
        }), /Cannot associate rule with 'eventBus' when using 'schedule'/);
        test.done();
    },
    'for cross-account targets': {
        'requires that the source stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceStack = new cdk.Stack(app, 'SourceStack');
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the source stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetStack = new cdk.Stack(app, 'TargetStack');
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete region'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete region for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the source and target stacks be part of the same App'(test) {
            const sourceApp = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(sourceApp, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetApp = new cdk.App();
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(targetApp, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack and target stack must belong to the same CDK app/);
            test.done();
        },
        'generates the correct rules in the source and target stacks when eventPattern is passed in the constructor'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule', {
                eventPattern: {
                    source: ['some-event'],
                },
            });
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource1 = new cdk.Construct(targetStack, 'Resource1');
            const resource2 = new cdk.Construct(targetStack, 'Resource2');
            rule.addTarget(new SomeTarget('T1', resource1));
            rule.addTarget(new SomeTarget('T2', resource2));
            assert_1.expect(sourceStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T1',
                        'Arn': {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    { 'Ref': 'AWS::Partition' },
                                    `:events:us-west-2:${targetAccount}:event-bus/default`,
                                ],
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T1',
                        'Arn': 'ARN1',
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T2',
                        'Arn': 'ARN1',
                    },
                ],
            }));
            const eventBusPolicyStack = app.node.findChild(`EventBusPolicy-${sourceAccount}-us-west-2-${targetAccount}`);
            assert_1.expect(eventBusPolicyStack).to(assert_1.haveResourceLike('AWS::Events::EventBusPolicy', {
                'Action': 'events:PutEvents',
                'StatementId': `Allow-account-${sourceAccount}`,
                'Principal': sourceAccount,
            }));
            test.done();
        },
        'generates the correct rule in the target stack when addEventPattern in the source rule is used'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource = new cdk.Construct(targetStack, 'Resource1');
            rule.addTarget(new SomeTarget('T', resource));
            rule.addEventPattern({
                source: ['some-event'],
            });
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'some-event',
                    ],
                },
                'State': 'ENABLED',
                'Targets': [
                    {
                        'Id': 'T',
                        'Arn': 'ARN1',
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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