"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2021.8.1"
__checksum__ = "478d3392431974469a74b5f04f82b35843eaef52de192f79a5e57881376f2ec4"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), (16, 9), (25, 61), (86, 26), (112, 12), None, (124, 19), (143, 22), (165, 7), (172, 20), (192, 18), None, (210, 29), (239, 45), (284, 7), (291, 9), (300, 36), (336, 16), (352, 10), (362, 28), None, (390, 54), (444, 8), (452, 18), (470, 19), (489, 13), (502, 14), (516, 14), None, None, (530, 29), (559, 20), (579, 35), (614, 14), (628, 24), (652, 9), None, (661, 25), (686, 27), (713, 8), (721, 13), None, None, (734, 17), (751, 6), (757, 26), (783, 5), (788, 5), (793, 19), (812, 14), (826, 11), (837, 12), (849, 27), None, (876, 11), (887, 11), (898, 7), (905, 29), (934, 18), (952, 27), (979, 46), (1025, 25), (1050, 16), (1066, 8), (1074, 5), (1079, 22), (1101, 18), None, (1119, 36), (1155, 15), (1170, 8), (1178, 11), None, (1189, 5), (1194, 16), (1210, 14), (1224, 18), None, (1242, 14), (1256, 18), (1274, 48), (1322, 19), (1341, 5), (1346, 59), (1405, 14), (1419, 14), (1433, 20), None, (1453, 10), (1463, 13), (1476, 20), (1496, 19), None, (1515, 13), (1528, 19), (1547, 11), (1558, 4), (1562, 22), (1584, 10), (1594, 7), (1601, 14), (1615, 21), (1636, 11), (1647, 21), (1668, 12), (1680, 32), None, (1712, 10), (1722, 14), (1736, 12), (1748, 45), (1793, 15), None, (1808, 11), (1819, 23), (1842, 21), (1863, 26), (1889, 6), (1895, 6), (1901, 7), (1908, 5), (1913, 20), (1933, 23), (1956, 24), (1980, 13), (1993, 15), (2008, 19), (2027, 6), (2033, 61), (2094, 44), (2138, 12), (2150, 23), (2173, 16), (2189, 38), (2227, 6), (2233, 12), (2245, 44), (2289, 6), (2295, 41), (2336, 13), (2349, 23), (2372, 30), (2402, 16), (2418, 8), (2426, 15), (2441, 12), (2453, 19), (2472, 21), (2493, 15), None, (2508, 35), (2543, 21), (2564, 17), (2581, 19), (2600, 26), (2626, 5), (2631, 37), (2668, 30), (2698, 16), (2714, 10), (2724, 17), (2741, 23), (2764, 14), (2778, 17), (2795, 8), (2803, 8), (2811, 7), (2818, 29), (2847, 6), (2853, 18), (2871, 27), (2898, 20), (2918, 17), (2935, 19), (2954, 12), (2966, 40), (3006, 40), (3046, 12), (3058, 48), (3106, 25), (3131, 12), None, (3143, 8), (3151, 25), (3176, 19), (3195, 6), (3201, 23), None, (3224, 30), (3254, 33), (3287, 14), (3301, 12), (3313, 27), None, (3340, 26), (3366, 41), (3407, 50), (3457, 15), (3472, 20), (3492, 15), (3507, 21), (3528, 32), (3560, 24), (3584, 20), (3604, 17), (3621, 60), (3681, 19), (3700, 9), (3709, 12), (3721, 12), (3733, 11), (3744, 10), (3754, 48), (3802, 32), None, (3834, 25), (3859, 23), None, (3882, 8), (3890, 8), (3898, 7), None, (3905, 25), (3930, 17), None, (3947, 21), (3968, 35), (4003, 21), (4024, 10), (4034, 36), (4070, 20), (4090, 22), (4112, 23), (4135, 19), (4154, 12), (4166, 5), (4171, 30), (4201, 24), (4225, 14), (4239, 14), (4253, 47), (4300, 52), None, None, (4352, 51), (4403, 42), None, (4445, 14), None, (4459, 15), (4474, 8), (4482, 21), (4503, 6), (4509, 16), (4525, 17)], [(4542, 8249), (12791, 8940), (21731, 9211), (30942, 7819), (38761, 8362), (47123, 8093), (55216, 8987), (64203, 7885), (72088, 8857), (80945, 8149), (89094, 9423), (98517, 8110), (106627, 8720), (115347, 9981), (125328, 8487), (133815, 8785), (142600, 9196), (151796, 8195), (159991, 8480), (168471, 8098), (176569, 8918), (185487, 8697), (194184, 8939), (203123, 8383), (211506, 8816), (220322, 8466), (228788, 8973), (237761, 9089), (246850, 8022), (254872, 8585), (263457, 8836), (272293, 8293), (280586, 8590), (289176, 8892), (298068, 7781), (305849, 8748), (314597, 8528), (323125, 9193), (332318, 8803), (341121, 8848), (349969, 9486), (359455, 8077), (367532, 8292), (375824, 8333), (384157, 8448), (392605, 8465), (401070, 8512), (409582, 9603), (419185, 8346), (427531, 7989), (435520, 8388), (443908, 8828), (452736, 8920), (461656, 8755), (470411, 8920), (479331, 8595), (487926, 8994), (496920, 8582), (505502, 8942), (514444, 7553), (521997, 8527), (530524, 8636), (539160, 8482), (547642, 8875), (556517, 8763), (565280, 8942), (574222, 8274), (582496, 9223), (591719, 9025), (600744, 8716), (609460, 8677), (618137, 8168), (626305, 7787), (634092, 8928), (643020, 8670), (651690, 9126), (660816, 8118), (668934, 9204), (678138, 8879), (687017, 8136), (695153, 8940), (704093, 7391), (711484, 8366), (719850, 8912), (728762, 8219), (736981, 8640), (745621, 9243), (754864, 8619), (763483, 8904), (772387, 8760), (781147, 9581), (790728, 8067), (798795, 8569), (807364, 8574), (815938, 8523), (824461, 9184), (833645, 8854), (842499, 8385), (850884, 8309), (859193, 8121), (867314, 8187), (875501, 8819), (884320, 8351), (892671, 8392), (901063, 8235), (909298, 9173), (918471, 9118), (927589, 9111), (936700, 9721), (946421, 9030), (955451, 8972), (964423, 8988), (973411, 8633), (982044, 8507), (990551, 8991), (999542, 8902), (1008444, 8563), (1017007, 8633), (1025640, 8372), (1034012, 9385), (1043397, 9179), (1052576, 9112), (1061688, 8665), (1070353, 8875), (1079228, 9752), (1088980, 8373), (1097353, 7825), (1105178, 9205), (1114383, 8595), (1122978, 10154), (1133132, 9106), (1142238, 8210), (1150448, 8852), (1159300, 8596), (1167896, 8441), (1176337, 8707), (1185044, 8234), (1193278, 9157), (1202435, 8237), (1210672, 8422), (1219094, 8910), (1228004, 8937), (1236941, 7943), (1244884, 8351), (1253235, 9085), (1262320, 8382), (1270702, 8462), (1279164, 8574), (1287738, 8299), (1296037, 8987), (1305024, 8755), (1313779, 8830), (1322609, 9053), (1331662, 8301), (1339963, 8703), (1348666, 8822), (1357488, 8318), (1365806, 8704), (1374510, 8215), (1382725, 7764), (1390489, 7839), (1398328, 8826), (1407154, 9327), (1416481, 8347), (1424828, 8302), (1433130, 9348), (1442478, 8528), (1451006, 8149), (1459155, 9077), (1468232, 8660), (1476892, 7674), (1484566, 8475), (1493041, 10007), (1503048, 8080), (1511128, 8188), (1519316, 9099), (1528415, 8574), (1536989, 9070), (1546059, 8354), (1554413, 8191), (1562604, 10799), (1573403, 8945), (1582348, 8690), (1591038, 8845), (1599883, 9547), (1609430, 9608), (1619038, 7929), (1626967, 8797), (1635764, 8150), (1643914, 8630), (1652544, 9379), (1661923, 8179), (1670102, 8941), (1679043, 8816), (1687859, 8420), (1696279, 8638), (1704917, 8378), (1713295, 8440), (1721735, 8752), (1730487, 8400), (1738887, 8719), (1747606, 8146), (1755752, 9106), (1764858, 8613), (1773471, 9312), (1782783, 9076), (1791859, 7717), (1799576, 9051), (1808627, 8415), (1817042, 8829), (1825871, 8862), (1834733, 9064), (1843797, 8676), (1852473, 8947), (1861420, 8796), (1870216, 8644), (1878860, 8618), (1887478, 8348), (1895826, 8705), (1904531, 9063), (1913594, 8767), (1922361, 8035), (1930396, 9713), (1940109, 8594), (1948703, 8575), (1957278, 8470), (1965748, 8618), (1974366, 7974), (1982340, 8944), (1991284, 8742), (2000026, 9437), (2009463, 8723), (2018186, 8138), (2026324, 9150), (2035474, 8597), (2044071, 9539), (2053610, 8287), (2061897, 8268), (2070165, 7652), (2077817, 9158), (2086975, 8932), (2095907, 9162), (2105069, 8400), (2113469, 8785), (2122254, 8445), (2130699, 9111), (2139810, 8369), (2148179, 7940), (2156119, 8511), (2164630, 8037), (2172667, 8948), (2181615, 9145), (2190760, 9039), (2199799, 8532), (2208331, 8555), (2216886, 8606)], [(2225492, 933), (2226425, 773), (2227198, 829), (2228027, 1026), (2229053, 699), (2229752, 861), (2230613, 668), (2231281, 1027), (2232308, 721), (2233029, 809), (2233838, 669), (2234507, 678), (2235185, 873), (2236058, 885), (2236943, 1090), (2238033, 1021), (2239054, 1366), (2240420, 747), (2241167, 1023), (2242190, 873), (2243063, 791), (2243854, 976), (2244830, 1017), (2245847, 860), (2246707, 862), (2247569, 764), (2248333, 1120), (2249453, 1350), (2250803, 842), (2251645, 869), (2252514, 1106), (2253620, 892), (2254512, 697), (2255209, 842), (2256051, 997), (2257048, 986), (2258034, 847), (2258881, 905), (2259786, 846), (2260632, 1225), (2261857, 726), (2262583, 1023), (2263606, 830), (2264436, 784), (2265220, 780), (2266000, 561), (2266561, 1086), (2267647, 1045), (2268692, 919), (2269611, 619), (2270230, 965), (2271195, 753), (2271948, 855), (2272803, 1128), (2273931, 1094), (2275025, 618), (2275643, 782), (2276425, 720), (2277145, 728), (2277873, 913), (2278786, 929), (2279715, 859), (2280574, 1159), (2281733, 1098), (2282831, 857), (2283688, 810), (2284498, 809), (2285307, 531), (2285838, 719), (2286557, 662), (2287219, 826), (2288045, 1001), (2289046, 713), (2289759, 897), (2290656, 700), (2291356, 797), (2292153, 714), (2292867, 797), (2293664, 901), (2294565, 602), (2295167, 919), (2296086, 724), (2296810, 1024), (2297834, 713), (2298547, 804), (2299351, 582), (2299933, 778), (2300711, 859), (2301570, 960), (2302530, 892), (2303422, 1092), (2304514, 1262), (2305776, 919), (2306695, 943), (2307638, 860), (2308498, 578), (2309076, 1009), (2310085, 929), (2311014, 675), (2311689, 763), (2312452, 873), (2313325, 1042), (2314367, 995), (2315362, 629), (2315991, 706), (2316697, 940), (2317637, 533), (2318170, 604), (2318774, 1113), (2319887, 1032), (2320919, 821), (2321740, 866), (2322606, 826), (2323432, 801), (2324233, 990), (2325223, 873), (2326096, 715), (2326811, 692), (2327503, 832), (2328335, 742), (2329077, 1153), (2330230, 790), (2331020, 926), (2331946, 585), (2332531, 821), (2333352, 980), (2334332, 911), (2335243, 1104), (2336347, 767), (2337114, 1096), (2338210, 949), (2339159, 671), (2339830, 939), (2340769, 828), (2341597, 1014), (2342611, 828), (2343439, 796), (2344235, 776), (2345011, 848), (2345859, 706), (2346565, 775), (2347340, 744), (2348084, 824), (2348908, 651), (2349559, 635), (2350194, 637), (2350831, 750), (2351581, 748), (2352329, 824), (2353153, 715), (2353868, 824), (2354692, 631), (2355323, 618), (2355941, 931), (2356872, 806), (2357678, 778), (2358456, 835), (2359291, 1106), (2360397, 880), (2361277, 718), (2361995, 1096), (2363091, 903), (2363994, 720), (2364714, 858), (2365572, 1042), (2366614, 744), (2367358, 745), (2368103, 745), (2368848, 785), (2369633, 813), (2370446, 900), (2371346, 689), (2372035, 1016), (2373051, 838), (2373889, 939), (2374828, 922), (2375750, 821), (2376571, 649), (2377220, 792), (2378012, 802), (2378814, 2027), (2380841, 675), (2381516, 843), (2382359, 777), (2383136, 1106), (2384242, 938), (2385180, 863), (2386043, 666), (2386709, 704), (2387413, 1016), (2388429, 675), (2389104, 651), (2389755, 884), (2390639, 864), (2391503, 1004), (2392507, 854), (2393361, 810), (2394171, 763), (2394934, 930), (2395864, 781), (2396645, 961), (2397606, 785), (2398391, 896), (2399287, 688), (2399975, 846), (2400821, 701), (2401522, 981), (2402503, 992), (2403495, 802), (2404297, 1080), (2405377, 780), (2406157, 933), (2407090, 1006), (2408096, 1146), (2409242, 959), (2410201, 846), (2411047, 1054), (2412101, 791), (2412892, 633), (2413525, 490), (2414015, 908), (2414923, 852), (2415775, 618), (2416393, 1146), (2417539, 623), (2418162, 841), (2419003, 969), (2419972, 997), (2420969, 944), (2421913, 773), (2422686, 1007), (2423693, 819), (2424512, 971), (2425483, 674), (2426157, 700), (2426857, 619), (2427476, 698), (2428174, 486), (2428660, 900), (2429560, 1091), (2430651, 926), (2431577, 751), (2432328, 742), (2433070, 709), (2433779, 1046), (2434825, 636), (2435461, 636), (2436097, 1007), (2437104, 519), (2437623, 1026), (2438649, 2281), (2440930, 746), (2441676, 809), (2442485, 1003), (2443488, 1151), (2444639, 534)], [(2445173, 48), None, (2445221, 35), (2445256, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (2445298, 42), None, (2445340, 25), (2445365, 44), (2445409, 22), (2445431, 18), None, None, None, None, (2445449, 26), None, None, None, None, (2445475, 21), (2445496, 25), None, None, (2445521, 26), None, None, None, None, (2445547, 71), (2445618, 21), (2445639, 23), None, None, None, None, (2445662, 48), None, None, None, None, None, (2445710, 31), None, None, None, None, (2445741, 42), None, (2445783, 22), None, (2445805, 21), None, (2445826, 26), (2445852, 42), None, None, (2445894, 77), (2445971, 27), None, None, None, None, (2445998, 21), (2446019, 21), None, None, (2446040, 34), (2446074, 42), None, None, None, (2446116, 25), None, None, (2446141, 21), None, None, None, None, None, (2446162, 24), (2446186, 21), None, None, (2446207, 26), None, (2446233, 18), None, (2446251, 54), None, None, None, None, None, None, (2446305, 26), None, None, None, (2446331, 20), None, None, (2446351, 64), (2446415, 42), (2446457, 17), (2446474, 17), (2446491, 26), None, (2446517, 26), None, None, None, (2446543, 26), (2446569, 20), (2446589, 26), None, (2446615, 42), (2446657, 63), None, None, None, (2446720, 40), (2446760, 48), None, None, None, (2446808, 47), None, None, None, None, None, None, None, (2446855, 42), None, (2446897, 80), None, (2446977, 9), None, (2446986, 21), (2447007, 42), None, None, (2447049, 65), (2447114, 82), (2447196, 21), None, (2447217, 72), None, None, (2447289, 24), (2447313, 21), None, None, None, None, None, (2447334, 42), (2447376, 21), (2447397, 21), None, (2447418, 42), (2447460, 25), None, (2447485, 38), (2447523, 21), (2447544, 56), None, None, (2447600, 21), (2447621, 19), (2447640, 26), None, (2447666, 16), None, (2447682, 21), None, None, (2447703, 38), None, (2447741, 22), (2447763, 21), (2447784, 21), (2447805, 21), None, (2447826, 63), None, (2447889, 21), (2447910, 42), None, (2447952, 17), None, None, None, None, (2447969, 21), (2447990, 21), None, None, (2448011, 21), None, None, (2448032, 21), None, (2448053, 26), None, (2448079, 50), None, None, None, (2448129, 50), (2448179, 26), (2448205, 21), (2448226, 21), (2448247, 19), None, (2448266, 35), (2448301, 26), (2448327, 23), (2448350, 39), (2448389, 42), None, None, None, None, None, None, (2448431, 21), None, None, None, (2448452, 21), None, None, (2448473, 90), None, (2448563, 239), (2448802, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
