"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyResource = exports.Resource = exports.ResourceBase = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
class ResourceBase extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    /**
     * @deprecated - Throws error in some use cases that have been enabled since this deprecation notice. Use `RestApi.urlForPath()` instead.
     */
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
class Resource extends ResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.api.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.api = props.parent.api;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * The RestApi associated with this Resource
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this.parentResource) {
            throw new Error('parentResource was unexpectedly not defined');
        }
        return this.parentResource.restApi;
    }
}
exports.Resource = Resource;
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 */
class ProxyResource extends Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZXNvdXJjZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBcUc7QUFDckcsaUVBQXVFO0FBQ3ZFLGlDQUEyQztBQUUzQyxpREFBaUQ7QUFDakQscUNBQWlEO0FBNEpqRCxNQUFzQixZQUFhLFNBQVEsZUFBaUI7SUFlMUQsWUFBWSxLQUFnQixFQUFFLEVBQVU7UUFDdEMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUhGLGFBQVEsR0FBcUMsRUFBRyxDQUFDO0lBSWxFLENBQUM7SUFFTSxXQUFXLENBQUMsUUFBZ0IsRUFBRSxPQUF5QjtRQUM1RCxPQUFPLElBQUksUUFBUSxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUUsRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxHQUFHLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDOUUsQ0FBQztJQUVNLFNBQVMsQ0FBQyxVQUFrQixFQUFFLFdBQXlCLEVBQUUsT0FBdUI7UUFDckYsT0FBTyxJQUFJLGVBQU0sQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsV0FBVyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDNUYsQ0FBQztJQUVNLFFBQVEsQ0FBQyxPQUE4QjtRQUM1QyxPQUFPLElBQUksYUFBYSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUUsRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLEdBQUcsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUMzRSxDQUFDO0lBRU0sZ0JBQWdCLENBQUMsT0FBb0I7UUFDMUMsTUFBTSxPQUFPLEdBQStCLEVBQUcsQ0FBQztRQUVoRCxFQUFFO1FBQ0YsK0JBQStCO1FBRS9CLE1BQU0sWUFBWSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksV0FBSSxDQUFDLGVBQWUsQ0FBQztRQUNsRSxPQUFPLENBQUMsOEJBQThCLENBQUMsR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQztRQUV4RSxFQUFFO1FBQ0YsOEJBQThCO1FBRTlCLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUVELElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3pFLE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLE9BQU8sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNySDtRQUVELDZFQUE2RTtRQUM3RSw0REFBNEQ7UUFDNUQsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM5QyxPQUFPLENBQUMsNkJBQTZCLENBQUMsR0FBRyxJQUFJLGFBQWEsR0FBRyxDQUFDO1FBRTlELDhEQUE4RDtRQUM5RCx5R0FBeUc7UUFDekcsSUFBSSxhQUFhLEtBQUssR0FBRyxFQUFFO1lBQ3pCLE9BQU8sQ0FBQyxJQUFJLEdBQUcsWUFBWSxDQUFDO1NBQzdCO1FBRUQsRUFBRTtRQUNGLCtCQUErQjtRQUUvQixJQUFJLFlBQVksR0FBRyxPQUFPLENBQUMsWUFBWSxJQUFJLFdBQUksQ0FBQyxXQUFXLENBQUM7UUFFNUQsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ2hDLElBQUksWUFBWSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQ2xHO1lBRUQsWUFBWSxHQUFHLFdBQUksQ0FBQyxXQUFXLENBQUM7U0FDakM7UUFFRCxPQUFPLENBQUMsOEJBQThCLENBQUMsR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQztRQUV4RSxFQUFFO1FBQ0YsbUNBQW1DO1FBRW5DLElBQUksT0FBTyxDQUFDLGdCQUFnQixFQUFFO1lBQzVCLE9BQU8sQ0FBQyxrQ0FBa0MsQ0FBQyxHQUFHLFVBQVUsQ0FBQztTQUMxRDtRQUVELEVBQUU7UUFDRix5QkFBeUI7UUFFekIsSUFBSSxhQUFhLENBQUM7UUFFbEIsSUFBSSxPQUFPLENBQUMsTUFBTSxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUU7WUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQ25GO1FBRUQsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFO1lBQ2xCLGFBQWEsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDO1NBQzVDO1FBRUQsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO1lBQ3hCLGFBQWEsR0FBRyxDQUFDLENBQUMsQ0FBQztTQUNwQjtRQUVELElBQUksYUFBYSxFQUFFO1lBQ2pCLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxHQUFHLElBQUksYUFBYSxHQUFHLENBQUM7U0FDMUQ7UUFFRCxFQUFFO1FBQ0YsZ0NBQWdDO1FBQ2hDLEVBQUU7UUFFRixJQUFJLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDekIsT0FBTyxDQUFDLCtCQUErQixDQUFDLEdBQUcsSUFBSSxPQUFPLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDO1NBQ25GO1FBRUQsRUFBRTtRQUNGLGFBQWE7UUFFYixNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO1FBRS9FLEVBQUU7UUFDRix5QkFBeUI7UUFFekIsTUFBTSx5QkFBeUIsR0FBNEIsRUFBRyxDQUFDO1FBQy9ELE1BQU0sbUJBQW1CLEdBQTZCLEVBQUcsQ0FBQztRQUUxRCxLQUFLLE1BQU0sQ0FBRSxJQUFJLEVBQUUsS0FBSyxDQUFFLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNyRCxNQUFNLEdBQUcsR0FBRywwQkFBMEIsSUFBSSxFQUFFLENBQUM7WUFDN0MseUJBQXlCLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO1lBQ3ZDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQztTQUNqQztRQUVELE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUUsSUFBSSw4QkFBZSxDQUFDO1lBQ25ELGdCQUFnQixFQUFFLEVBQUUsa0JBQWtCLEVBQUUscUJBQXFCLEVBQUU7WUFDL0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEVBQUUsVUFBVSxFQUFFLEdBQUcsVUFBVSxFQUFFLEVBQUUsa0JBQWtCLEVBQUUseUJBQXlCLEVBQUUsaUJBQWlCLEVBQUUsc0JBQXNCLEVBQUUsRUFBRTthQUM1SDtTQUNGLENBQUMsRUFBRTtZQUNGLGVBQWUsRUFBRTtnQkFDZixFQUFFLFVBQVUsRUFBRSxHQUFHLFVBQVUsRUFBRSxFQUFFLGtCQUFrQixFQUFFLG1CQUFtQixFQUFFO2FBQ3pFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgseUZBQXlGO1FBQ3pGLFNBQVMsc0JBQXNCO1lBQzdCLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTlDLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7Z0JBQ3hCLE9BQU8sU0FBUyxDQUFDO2FBQ2xCO1lBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztZQUVyQyxRQUFRLENBQUMsSUFBSSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7WUFDekQsUUFBUSxDQUFDLElBQUksQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1lBRWpGLE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFM0UsUUFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLFNBQVMsR0FBRyxDQUFDLENBQUM7WUFDbkMsUUFBUSxDQUFDLElBQUksQ0FBQyxnRkFBZ0YsQ0FBQyxDQUFDO1lBQ2hHLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFdEIsT0FBTztnQkFDTCxrQkFBa0IsRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQzthQUN4QyxDQUFDO1FBQ0osQ0FBQztJQUNILENBQUM7SUFFTSxXQUFXLENBQUMsUUFBZ0I7UUFDakMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7T0FFRztJQUNJLFdBQVcsQ0FBQyxRQUFnQixFQUFFLFFBQWtCO1FBQ3JELElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsUUFBUSxDQUFDO0lBQ3JDLENBQUM7SUFFTSxlQUFlLENBQUMsSUFBWTtRQUNqQyxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ1QsT0FBTyxJQUFJLENBQUM7U0FDYjtRQUVELElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUN4QixJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssR0FBRyxFQUFFO2dCQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQzthQUMvRjtZQUVELG9CQUFvQjtZQUNwQixPQUFPLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQzdDO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUM5QixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDM0IsSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLEtBQUssRUFBRSxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMscURBQXFELENBQUMsQ0FBQztTQUN4RTtRQUVELElBQUksUUFBUSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNiLFFBQVEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25DO1FBRUQsT0FBTyxRQUFRLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLEdBQUc7UUFDWixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM1QyxDQUFDO0NBQ0Y7QUFuTkQsb0NBbU5DO0FBRUQsTUFBYSxRQUFTLFNBQVEsWUFBWTtJQVV4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsd0JBQXdCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXpDLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUVuQyxJQUFJLEtBQUssQ0FBQyxNQUFNLFlBQVksWUFBWSxFQUFFO1lBQ3hDLEtBQUssQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDaEQ7UUFFRCxNQUFNLGFBQWEsR0FBcUI7WUFDdEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFNBQVM7WUFDckMsUUFBUSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTtZQUNqQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7U0FDekIsQ0FBQztRQUNGLE1BQU0sUUFBUSxHQUFHLElBQUksa0NBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRWxFLElBQUksQ0FBQyxVQUFVLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUMvQixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDO1FBRTVCLCtDQUErQztRQUMvQyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDO1FBQzlCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUFFLElBQUksQ0FBQyxJQUFJLElBQUksR0FBRyxDQUFDO1NBQUU7UUFDbkQsSUFBSSxDQUFDLElBQUksSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDO1FBRTVCLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDO1FBQ3JELElBQUksVUFBVSxFQUFFO1lBQ2QsVUFBVSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDeEMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsOEVBQThFO1FBQzlFLGtFQUFrRTtRQUNsRSxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQUM7UUFDdEYsSUFBSSxDQUFDLG9CQUFvQixHQUFHO1lBQzFCLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxvQkFBb0I7WUFDcEMsR0FBRyxLQUFLLENBQUMsb0JBQW9CO1NBQzlCLENBQUM7UUFDRixJQUFJLENBQUMsMkJBQTJCLEdBQUcsS0FBSyxDQUFDLDJCQUEyQixJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsMkJBQTJCLENBQUM7UUFFakgsSUFBSSxJQUFJLENBQUMsMkJBQTJCLEVBQUU7WUFDcEMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1NBQ3pEO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsT0FBTztRQUNoQixJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxDQUFDLENBQUM7U0FDaEU7UUFDRCxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDO0lBQ3JDLENBQUM7Q0FDRjtBQWxFRCw0QkFrRUM7QUFvQkQ7OztHQUdHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsUUFBUTtJQU96QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLFFBQVEsRUFBRSxVQUFVO1lBQ3BCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7WUFDNUMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtTQUNqRCxDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQ3pFLElBQUksU0FBUyxFQUFFO1lBQ2IsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3hDO0lBQ0gsQ0FBQztJQUVNLFNBQVMsQ0FBQyxVQUFrQixFQUFFLFdBQXlCLEVBQUUsT0FBdUI7UUFDckYsd0VBQXdFO1FBQ3hFLG9EQUFvRDtRQUNwRCxJQUFJLElBQUksQ0FBQyxjQUFjLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEtBQUssR0FBRyxFQUFFO1lBQzNELDREQUE0RDtZQUM1RCxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLFlBQVksZUFBTSxDQUFDLEVBQUU7Z0JBQzFFLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRSxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7YUFDakU7U0FDRjtRQUNELE9BQU8sS0FBSyxDQUFDLFNBQVMsQ0FBQyxVQUFVLEVBQUUsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzNELENBQUM7Q0FDRjtBQWhDRCxzQ0FnQ0M7QUFFRCxTQUFTLHdCQUF3QixDQUFDLElBQVk7SUFDNUMsOENBQThDO0lBQzlDLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQzlDLElBQUksR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRXZDLGdEQUFnRDtRQUNoRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDdEIsSUFBSSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDeEM7S0FDRjtJQUVELElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQzt1REFDbUMsSUFBSSxFQUFFLENBQUMsQ0FBQztLQUM1RDtBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIElSZXNvdXJjZSBhcyBJUmVzb3VyY2VCYXNlLCBSZXNvdXJjZSBhcyBSZXNvdXJjZUNvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIENmblJlc291cmNlUHJvcHMgfSBmcm9tICcuL2FwaWdhdGV3YXkuZ2VuZXJhdGVkJztcbmltcG9ydCB7IENvcnMsIENvcnNPcHRpb25zIH0gZnJvbSAnLi9jb3JzJztcbmltcG9ydCB7IEludGVncmF0aW9uIH0gZnJvbSAnLi9pbnRlZ3JhdGlvbic7XG5pbXBvcnQgeyBNb2NrSW50ZWdyYXRpb24gfSBmcm9tICcuL2ludGVncmF0aW9ucyc7XG5pbXBvcnQgeyBNZXRob2QsIE1ldGhvZE9wdGlvbnMgfSBmcm9tICcuL21ldGhvZCc7XG5pbXBvcnQgeyBJUmVzdEFwaSwgUmVzdEFwaSB9IGZyb20gJy4vcmVzdGFwaSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSVJlc291cmNlIGV4dGVuZHMgSVJlc291cmNlQmFzZSB7XG4gIC8qKlxuICAgKiBUaGUgcGFyZW50IG9mIHRoaXMgcmVzb3VyY2Ugb3IgdW5kZWZpbmVkIGZvciB0aGUgcm9vdCByZXNvdXJjZS5cbiAgICovXG4gIHJlYWRvbmx5IHBhcmVudFJlc291cmNlPzogSVJlc291cmNlO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVzdCBBUEkgdGhhdCB0aGlzIHJlc291cmNlIGlzIHBhcnQgb2YuXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIC0gVGhyb3dzIGFuIGVycm9yIGlmIHRoaXMgUmVzb3VyY2UgaXMgbm90IGFzc29jaWF0ZWQgd2l0aCBhbiBpbnN0YW5jZSBvZiBgUmVzdEFwaWAuIFVzZSBgYXBpYCBpbnN0ZWFkLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVzdEFwaTogUmVzdEFwaTtcblxuICAvKipcbiAgICogVGhlIHJlc3QgQVBJIHRoYXQgdGhpcyByZXNvdXJjZSBpcyBwYXJ0IG9mLlxuICAgKlxuICAgKiBUaGUgcmVhc29uIHdlIG5lZWQgdGhlIFJlc3RBcGkgb2JqZWN0IGl0c2VsZiBhbmQgbm90IGp1c3QgdGhlIElEIGlzIGJlY2F1c2UgdGhlIG1vZGVsXG4gICAqIGlzIGJlaW5nIHRyYWNrZWQgYnkgdGhlIHRvcC1sZXZlbCBSZXN0QXBpIG9iamVjdCBmb3IgdGhlIHB1cnBvc2Ugb2YgY2FsY3VsYXRpbmcgaXQnc1xuICAgKiBoYXNoIHRvIGRldGVybWluZSB0aGUgSUQgb2YgdGhlIGRlcGxveW1lbnQuIFRoaXMgYWxsb3dzIHVzIHRvIGF1dG9tYXRpY2FsbHkgdXBkYXRlXG4gICAqIHRoZSBkZXBsb3ltZW50IHdoZW4gdGhlIG1vZGVsIG9mIHRoZSBSRVNUIEFQSSBjaGFuZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgYXBpOiBJUmVzdEFwaTtcblxuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSByZXNvdXJjZS5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2VJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZnVsbCBwYXRoIG9mIHRoaXMgcmVzdW9yY2UuXG4gICAqL1xuICByZWFkb25seSBwYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIGludGVncmF0aW9uIHRvIHVzZSBhcyBhIGRlZmF1bHQgZm9yIGFsbCBtZXRob2RzIGNyZWF0ZWQgd2l0aGluIHRoaXNcbiAgICogQVBJIHVubGVzcyBhbiBpbnRlZ3JhdGlvbiBpcyBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcblxuICAvKipcbiAgICogTWV0aG9kIG9wdGlvbnMgdG8gdXNlIGFzIGEgZGVmYXVsdCBmb3IgYWxsIG1ldGhvZHMgY3JlYXRlZCB3aXRoaW4gdGhpc1xuICAgKiBBUEkgdW5sZXNzIGN1c3RvbSBvcHRpb25zIGFyZSBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0TWV0aG9kT3B0aW9ucz86IE1ldGhvZE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIERlZmF1bHQgb3B0aW9ucyBmb3IgQ09SUyBwcmVmbGlnaHQgT1BUSU9OUyBtZXRob2QuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM/OiBDb3JzT3B0aW9ucztcblxuICAvKipcbiAgICogR2V0cyBvciBjcmVhdGUgYWxsIHJlc291cmNlcyBsZWFkaW5nIHVwIHRvIHRoZSBzcGVjaWZpZWQgcGF0aC5cbiAgICpcbiAgICogLSBQYXRoIG1heSBvbmx5IHN0YXJ0IHdpdGggXCIvXCIgaWYgdGhpcyBtZXRob2QgaXMgY2FsbGVkIG9uIHRoZSByb290IHJlc291cmNlLlxuICAgKiAtIEFsbCByZXNvdXJjZXMgYXJlIGNyZWF0ZWQgdXNpbmcgZGVmYXVsdCBvcHRpb25zLlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcmVsYXRpdmUgcGF0aFxuICAgKiBAcmV0dXJucyBhIG5ldyBvciBleGlzdGluZyByZXNvdXJjZS5cbiAgICovXG4gIHJlc291cmNlRm9yUGF0aChwYXRoOiBzdHJpbmcpOiBSZXNvdXJjZTtcblxuICAvKipcbiAgICogRGVmaW5lcyBhIG5ldyBjaGlsZCByZXNvdXJjZSB3aGVyZSB0aGlzIHJlc291cmNlIGlzIHRoZSBwYXJlbnQuXG4gICAqIEBwYXJhbSBwYXRoUGFydCBUaGUgcGF0aCBwYXJ0IGZvciB0aGUgY2hpbGQgcmVzb3VyY2VcbiAgICogQHBhcmFtIG9wdGlvbnMgUmVzb3VyY2Ugb3B0aW9uc1xuICAgKiBAcmV0dXJucyBBIFJlc291cmNlIG9iamVjdFxuICAgKi9cbiAgYWRkUmVzb3VyY2UocGF0aFBhcnQ6IHN0cmluZywgb3B0aW9ucz86IFJlc291cmNlT3B0aW9ucyk6IFJlc291cmNlO1xuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYSBjaGlsZCByZXNvdXJjZSBieSBwYXRoIHBhcnQuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoUGFydCBUaGUgcGF0aCBwYXJ0IG9mIHRoZSBjaGlsZCByZXNvdXJjZVxuICAgKiBAcmV0dXJucyB0aGUgY2hpbGQgcmVzb3VyY2Ugb3IgdW5kZWZpbmVkIGlmIG5vdCBmb3VuZFxuICAgKi9cbiAgZ2V0UmVzb3VyY2UocGF0aFBhcnQ6IHN0cmluZyk6IElSZXNvdXJjZSB8IHVuZGVmaW5lZDtcblxuICAvKipcbiAgICogQWRkcyBhIGdyZWVkeSBwcm94eSByZXNvdXJjZSAoXCJ7cHJveHkrfVwiKSBhbmQgYW4gQU5ZIG1ldGhvZCB0byB0aGlzIHJvdXRlLlxuICAgKiBAcGFyYW0gb3B0aW9ucyBEZWZhdWx0IGludGVncmF0aW9uIGFuZCBtZXRob2Qgb3B0aW9ucy5cbiAgICovXG4gIGFkZFByb3h5KG9wdGlvbnM/OiBQcm94eVJlc291cmNlT3B0aW9ucyk6IFByb3h5UmVzb3VyY2U7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYSBuZXcgbWV0aG9kIGZvciB0aGlzIHJlc291cmNlLlxuICAgKiBAcGFyYW0gaHR0cE1ldGhvZCBUaGUgSFRUUCBtZXRob2RcbiAgICogQHBhcmFtIHRhcmdldCBUaGUgdGFyZ2V0IGJhY2tlbmQgaW50ZWdyYXRpb24gZm9yIHRoaXMgbWV0aG9kXG4gICAqIEBwYXJhbSBvcHRpb25zIE1ldGhvZCBvcHRpb25zLCBzdWNoIGFzIGF1dGhlbnRpY2F0aW9uLlxuICAgKlxuICAgKiBAcmV0dXJucyBUaGUgbmV3bHkgY3JlYXRlZCBgTWV0aG9kYCBvYmplY3QuXG4gICAqL1xuICBhZGRNZXRob2QoaHR0cE1ldGhvZDogc3RyaW5nLCB0YXJnZXQ/OiBJbnRlZ3JhdGlvbiwgb3B0aW9ucz86IE1ldGhvZE9wdGlvbnMpOiBNZXRob2Q7XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gT1BUSU9OUyBtZXRob2QgdG8gdGhpcyByZXNvdXJjZSB3aGljaCByZXNwb25kcyB0byBDcm9zcy1PcmlnaW5cbiAgICogUmVzb3VyY2UgU2hhcmluZyAoQ09SUykgcHJlZmxpZ2h0IHJlcXVlc3RzLlxuICAgKlxuICAgKiBDcm9zcy1PcmlnaW4gUmVzb3VyY2UgU2hhcmluZyAoQ09SUykgaXMgYSBtZWNoYW5pc20gdGhhdCB1c2VzIGFkZGl0aW9uYWxcbiAgICogSFRUUCBoZWFkZXJzIHRvIHRlbGwgYnJvd3NlcnMgdG8gZ2l2ZSBhIHdlYiBhcHBsaWNhdGlvbiBydW5uaW5nIGF0IG9uZVxuICAgKiBvcmlnaW4sIGFjY2VzcyB0byBzZWxlY3RlZCByZXNvdXJjZXMgZnJvbSBhIGRpZmZlcmVudCBvcmlnaW4uIEEgd2ViXG4gICAqIGFwcGxpY2F0aW9uIGV4ZWN1dGVzIGEgY3Jvc3Mtb3JpZ2luIEhUVFAgcmVxdWVzdCB3aGVuIGl0IHJlcXVlc3RzIGFcbiAgICogcmVzb3VyY2UgdGhhdCBoYXMgYSBkaWZmZXJlbnQgb3JpZ2luIChkb21haW4sIHByb3RvY29sLCBvciBwb3J0KSBmcm9tIGl0c1xuICAgKiBvd24uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kZXZlbG9wZXIubW96aWxsYS5vcmcvZW4tVVMvZG9jcy9XZWIvSFRUUC9DT1JTXG4gICAqIEBwYXJhbSBvcHRpb25zIENPUlMgb3B0aW9uc1xuICAgKiBAcmV0dXJucyBhIGBNZXRob2RgIG9iamVjdFxuICAgKi9cbiAgYWRkQ29yc1ByZWZsaWdodChvcHRpb25zOiBDb3JzT3B0aW9ucyk6IE1ldGhvZDtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogQW4gaW50ZWdyYXRpb24gdG8gdXNlIGFzIGEgZGVmYXVsdCBmb3IgYWxsIG1ldGhvZHMgY3JlYXRlZCB3aXRoaW4gdGhpc1xuICAgKiBBUEkgdW5sZXNzIGFuIGludGVncmF0aW9uIGlzIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJbmhlcml0ZWQgZnJvbSBwYXJlbnQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcblxuICAvKipcbiAgICogTWV0aG9kIG9wdGlvbnMgdG8gdXNlIGFzIGEgZGVmYXVsdCBmb3IgYWxsIG1ldGhvZHMgY3JlYXRlZCB3aXRoaW4gdGhpc1xuICAgKiBBUEkgdW5sZXNzIGN1c3RvbSBvcHRpb25zIGFyZSBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSW5oZXJpdGVkIGZyb20gcGFyZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdE1ldGhvZE9wdGlvbnM/OiBNZXRob2RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgQ09SUyBwcmVmbGlnaHQgT1BUSU9OUyBtZXRob2QgdG8gdGhpcyByZXNvdXJjZSBhbmQgYWxsIGNoaWxkXG4gICAqIHJlc291cmNlcy5cbiAgICpcbiAgICogWW91IGNhbiBhZGQgQ09SUyBhdCB0aGUgcmVzb3VyY2UtbGV2ZWwgdXNpbmcgYGFkZENvcnNQcmVmbGlnaHRgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENPUlMgaXMgZGlzYWJsZWRcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucz86IENvcnNPcHRpb25zO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlUHJvcHMgZXh0ZW5kcyBSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBhcmVudCByZXNvdXJjZSBvZiB0aGlzIHJlc291cmNlLiBZb3UgY2FuIGVpdGhlciBwYXNzIGFub3RoZXJcbiAgICogYFJlc291cmNlYCBvYmplY3Qgb3IgYSBgUmVzdEFwaWAgb2JqZWN0IGhlcmUuXG4gICAqL1xuICByZWFkb25seSBwYXJlbnQ6IElSZXNvdXJjZTtcblxuICAvKipcbiAgICogQSBwYXRoIG5hbWUgZm9yIHRoZSByZXNvdXJjZS5cbiAgICovXG4gIHJlYWRvbmx5IHBhdGhQYXJ0OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBSZXNvdXJjZUJhc2UgZXh0ZW5kcyBSZXNvdXJjZUNvbnN0cnVjdCBpbXBsZW1lbnRzIElSZXNvdXJjZSB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBwYXJlbnRSZXNvdXJjZT86IElSZXNvdXJjZTtcbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIC0gIFRocm93cyBhbiBlcnJvciBpZiB0aGlzIFJlc291cmNlIGlzIG5vdCBhc3NvY2lhdGVkIHdpdGggYW4gaW5zdGFuY2Ugb2YgYFJlc3RBcGlgLiBVc2UgYGFwaWAgaW5zdGVhZC5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSByZXN0QXBpOiBSZXN0QXBpO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYXBpOiBJUmVzdEFwaTtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJlc291cmNlSWQ6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHBhdGg6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlZmF1bHRJbnRlZ3JhdGlvbj86IEludGVncmF0aW9uO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZGVmYXVsdE1ldGhvZE9wdGlvbnM/OiBNZXRob2RPcHRpb25zO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zPzogQ29yc09wdGlvbnM7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBjaGlsZHJlbjogeyBbcGF0aFBhcnQ6IHN0cmluZ106IFJlc291cmNlIH0gPSB7IH07XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gIH1cblxuICBwdWJsaWMgYWRkUmVzb3VyY2UocGF0aFBhcnQ6IHN0cmluZywgb3B0aW9ucz86IFJlc291cmNlT3B0aW9ucyk6IFJlc291cmNlIHtcbiAgICByZXR1cm4gbmV3IFJlc291cmNlKHRoaXMsIHBhdGhQYXJ0LCB7IHBhcmVudDogdGhpcywgcGF0aFBhcnQsIC4uLm9wdGlvbnMgfSk7XG4gIH1cblxuICBwdWJsaWMgYWRkTWV0aG9kKGh0dHBNZXRob2Q6IHN0cmluZywgaW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbiwgb3B0aW9ucz86IE1ldGhvZE9wdGlvbnMpOiBNZXRob2Qge1xuICAgIHJldHVybiBuZXcgTWV0aG9kKHRoaXMsIGh0dHBNZXRob2QsIHsgcmVzb3VyY2U6IHRoaXMsIGh0dHBNZXRob2QsIGludGVncmF0aW9uLCBvcHRpb25zIH0pO1xuICB9XG5cbiAgcHVibGljIGFkZFByb3h5KG9wdGlvbnM/OiBQcm94eVJlc291cmNlT3B0aW9ucyk6IFByb3h5UmVzb3VyY2Uge1xuICAgIHJldHVybiBuZXcgUHJveHlSZXNvdXJjZSh0aGlzLCAne3Byb3h5K30nLCB7IHBhcmVudDogdGhpcywgLi4ub3B0aW9ucyB9KTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRDb3JzUHJlZmxpZ2h0KG9wdGlvbnM6IENvcnNPcHRpb25zKSB7XG4gICAgY29uc3QgaGVhZGVyczogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH0gPSB7IH07XG5cbiAgICAvL1xuICAgIC8vIEFjY2Vzcy1Db250cm9sLUFsbG93LUhlYWRlcnNcblxuICAgIGNvbnN0IGFsbG93SGVhZGVycyA9IG9wdGlvbnMuYWxsb3dIZWFkZXJzIHx8IENvcnMuREVGQVVMVF9IRUFERVJTO1xuICAgIGhlYWRlcnNbJ0FjY2Vzcy1Db250cm9sLUFsbG93LUhlYWRlcnMnXSA9IGAnJHthbGxvd0hlYWRlcnMuam9pbignLCcpfSdgO1xuXG4gICAgLy9cbiAgICAvLyBBY2Nlc3MtQ29udHJvbC1BbGxvdy1PcmlnaW5cblxuICAgIGlmIChvcHRpb25zLmFsbG93T3JpZ2lucy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYWxsb3dPcmlnaW5zIG11c3QgY29udGFpbiBhdCBsZWFzdCBvbmUgb3JpZ2luJyk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuYWxsb3dPcmlnaW5zLmluY2x1ZGVzKCcqJykgJiYgb3B0aW9ucy5hbGxvd09yaWdpbnMubGVuZ3RoID4gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIFwiYWxsb3dPcmlnaW5zXCIgLSBjYW5ub3QgbWl4IFwiKlwiIHdpdGggc3BlY2lmaWMgb3JpZ2luczogJHtvcHRpb25zLmFsbG93T3JpZ2lucy5qb2luKCcsJyl9YCk7XG4gICAgfVxuXG4gICAgLy8gd2UgdXNlIHRoZSBmaXJzdCBvcmlnaW4gaGVyZSBhbmQgaWYgdGhlcmUgYXJlIG1vcmUgb3JpZ2lucyBpbiB0aGUgbGlzdCwgd2VcbiAgICAvLyB3aWxsIG1hdGNoIGFnYWluc3QgdGhlbSBpbiB0aGUgcmVzcG9uc2UgdmVsb2NpdHkgdGVtcGxhdGVcbiAgICBjb25zdCBpbml0aWFsT3JpZ2luID0gb3B0aW9ucy5hbGxvd09yaWdpbnNbMF07XG4gICAgaGVhZGVyc1snQWNjZXNzLUNvbnRyb2wtQWxsb3ctT3JpZ2luJ10gPSBgJyR7aW5pdGlhbE9yaWdpbn0nYDtcblxuICAgIC8vIHRoZSBcIlZhcnlcIiBoZWFkZXIgaXMgcmVxdWlyZWQgaWYgd2UgYWxsb3cgYSBzcGVjaWZpYyBvcmlnaW5cbiAgICAvLyBodHRwczovL2RldmVsb3Blci5tb3ppbGxhLm9yZy9lbi1VUy9kb2NzL1dlYi9IVFRQL0hlYWRlcnMvQWNjZXNzLUNvbnRyb2wtQWxsb3ctT3JpZ2luI0NPUlNfYW5kX2NhY2hpbmdcbiAgICBpZiAoaW5pdGlhbE9yaWdpbiAhPT0gJyonKSB7XG4gICAgICBoZWFkZXJzLlZhcnkgPSAnXFwnT3JpZ2luXFwnJztcbiAgICB9XG5cbiAgICAvL1xuICAgIC8vIEFjY2Vzcy1Db250cm9sLUFsbG93LU1ldGhvZHNcblxuICAgIGxldCBhbGxvd01ldGhvZHMgPSBvcHRpb25zLmFsbG93TWV0aG9kcyB8fCBDb3JzLkFMTF9NRVRIT0RTO1xuXG4gICAgaWYgKGFsbG93TWV0aG9kcy5pbmNsdWRlcygnQU5ZJykpIHtcbiAgICAgIGlmIChhbGxvd01ldGhvZHMubGVuZ3RoID4gMSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEFOWSBjYW5ub3QgYmUgdXNlZCB3aXRoIGFueSBvdGhlciBtZXRob2QuIFJlY2VpdmVkOiAke2FsbG93TWV0aG9kcy5qb2luKCcsJyl9YCk7XG4gICAgICB9XG5cbiAgICAgIGFsbG93TWV0aG9kcyA9IENvcnMuQUxMX01FVEhPRFM7XG4gICAgfVxuXG4gICAgaGVhZGVyc1snQWNjZXNzLUNvbnRyb2wtQWxsb3ctTWV0aG9kcyddID0gYCcke2FsbG93TWV0aG9kcy5qb2luKCcsJyl9J2A7XG5cbiAgICAvL1xuICAgIC8vIEFjY2Vzcy1Db250cm9sLUFsbG93LUNyZWRlbnRpYWxzXG5cbiAgICBpZiAob3B0aW9ucy5hbGxvd0NyZWRlbnRpYWxzKSB7XG4gICAgICBoZWFkZXJzWydBY2Nlc3MtQ29udHJvbC1BbGxvdy1DcmVkZW50aWFscyddID0gJ1xcJ3RydWVcXCcnO1xuICAgIH1cblxuICAgIC8vXG4gICAgLy8gQWNjZXNzLUNvbnRyb2wtTWF4LUFnZVxuXG4gICAgbGV0IG1heEFnZVNlY29uZHM7XG5cbiAgICBpZiAob3B0aW9ucy5tYXhBZ2UgJiYgb3B0aW9ucy5kaXNhYmxlQ2FjaGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIG9wdGlvbnMgXCJtYXhBZ2VcIiBhbmQgXCJkaXNhYmxlQ2FjaGVcIiBhcmUgbXV0dWFsbHkgZXhjbHVzaXZlJyk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMubWF4QWdlKSB7XG4gICAgICBtYXhBZ2VTZWNvbmRzID0gb3B0aW9ucy5tYXhBZ2UudG9TZWNvbmRzKCk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuZGlzYWJsZUNhY2hlKSB7XG4gICAgICBtYXhBZ2VTZWNvbmRzID0gLTE7XG4gICAgfVxuXG4gICAgaWYgKG1heEFnZVNlY29uZHMpIHtcbiAgICAgIGhlYWRlcnNbJ0FjY2Vzcy1Db250cm9sLU1heC1BZ2UnXSA9IGAnJHttYXhBZ2VTZWNvbmRzfSdgO1xuICAgIH1cblxuICAgIC8vXG4gICAgLy8gQWNjZXNzLUNvbnRyb2wtRXhwb3NlLUhlYWRlcnNcbiAgICAvL1xuXG4gICAgaWYgKG9wdGlvbnMuZXhwb3NlSGVhZGVycykge1xuICAgICAgaGVhZGVyc1snQWNjZXNzLUNvbnRyb2wtRXhwb3NlLUhlYWRlcnMnXSA9IGAnJHtvcHRpb25zLmV4cG9zZUhlYWRlcnMuam9pbignLCcpfSdgO1xuICAgIH1cblxuICAgIC8vXG4gICAgLy8gc3RhdHVzQ29kZVxuXG4gICAgY29uc3Qgc3RhdHVzQ29kZSA9IG9wdGlvbnMuc3RhdHVzQ29kZSAhPT0gdW5kZWZpbmVkID8gb3B0aW9ucy5zdGF0dXNDb2RlIDogMjA0O1xuXG4gICAgLy9cbiAgICAvLyBwcmVwYXJlIHJlc3BvbnNlUGFyYW1zXG5cbiAgICBjb25zdCBpbnRlZ3JhdGlvblJlc3BvbnNlUGFyYW1zOiB7IFtwOiBzdHJpbmddOiBzdHJpbmcgfSA9IHsgfTtcbiAgICBjb25zdCBtZXRob2RSZXBvbnNlUGFyYW1zOiB7IFtwOiBzdHJpbmddOiBib29sZWFuIH0gPSB7IH07XG5cbiAgICBmb3IgKGNvbnN0IFsgbmFtZSwgdmFsdWUgXSBvZiBPYmplY3QuZW50cmllcyhoZWFkZXJzKSkge1xuICAgICAgY29uc3Qga2V5ID0gYG1ldGhvZC5yZXNwb25zZS5oZWFkZXIuJHtuYW1lfWA7XG4gICAgICBpbnRlZ3JhdGlvblJlc3BvbnNlUGFyYW1zW2tleV0gPSB2YWx1ZTtcbiAgICAgIG1ldGhvZFJlcG9uc2VQYXJhbXNba2V5XSA9IHRydWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMuYWRkTWV0aG9kKCdPUFRJT05TJywgbmV3IE1vY2tJbnRlZ3JhdGlvbih7XG4gICAgICByZXF1ZXN0VGVtcGxhdGVzOiB7ICdhcHBsaWNhdGlvbi9qc29uJzogJ3sgc3RhdHVzQ29kZTogMjAwIH0nIH0sXG4gICAgICBpbnRlZ3JhdGlvblJlc3BvbnNlczogW1xuICAgICAgICB7IHN0YXR1c0NvZGU6IGAke3N0YXR1c0NvZGV9YCwgcmVzcG9uc2VQYXJhbWV0ZXJzOiBpbnRlZ3JhdGlvblJlc3BvbnNlUGFyYW1zLCByZXNwb25zZVRlbXBsYXRlczogcmVuZGVyUmVzcG9uc2VUZW1wbGF0ZSgpIH0sXG4gICAgICBdLFxuICAgIH0pLCB7XG4gICAgICBtZXRob2RSZXNwb25zZXM6IFtcbiAgICAgICAgeyBzdGF0dXNDb2RlOiBgJHtzdGF0dXNDb2RlfWAsIHJlc3BvbnNlUGFyYW1ldGVyczogbWV0aG9kUmVwb25zZVBhcmFtcyB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIC8vIHJlbmRlcnMgdGhlIHJlc3BvbnNlIHRlbXBsYXRlIHRvIG1hdGNoIGFsbCBwb3NzaWJsZSBvcmlnaW5zIChpZiB3ZSBoYXZlIG1vcmUgdGhhbiBvbmUpXG4gICAgZnVuY3Rpb24gcmVuZGVyUmVzcG9uc2VUZW1wbGF0ZSgpIHtcbiAgICAgIGNvbnN0IG9yaWdpbnMgPSBvcHRpb25zLmFsbG93T3JpZ2lucy5zbGljZSgxKTtcblxuICAgICAgaWYgKG9yaWdpbnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHRlbXBsYXRlID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICAgICAgdGVtcGxhdGUucHVzaCgnI3NldCgkb3JpZ2luID0gJGlucHV0LnBhcmFtcyhcIk9yaWdpblwiKSknKTtcbiAgICAgIHRlbXBsYXRlLnB1c2goJyNpZigkb3JpZ2luID09IFwiXCIpICNzZXQoJG9yaWdpbiA9ICRpbnB1dC5wYXJhbXMoXCJvcmlnaW5cIikpICNlbmQnKTtcblxuICAgICAgY29uc3QgY29uZGl0aW9uID0gb3JpZ2lucy5tYXAobyA9PiBgJG9yaWdpbi5tYXRjaGVzKFwiJHtvfVwiKWApLmpvaW4oJyB8fCAnKTtcblxuICAgICAgdGVtcGxhdGUucHVzaChgI2lmKCR7Y29uZGl0aW9ufSlgKTtcbiAgICAgIHRlbXBsYXRlLnB1c2goJyAgI3NldCgkY29udGV4dC5yZXNwb25zZU92ZXJyaWRlLmhlYWRlci5BY2Nlc3MtQ29udHJvbC1BbGxvdy1PcmlnaW4gPSAkb3JpZ2luKScpO1xuICAgICAgdGVtcGxhdGUucHVzaCgnI2VuZCcpO1xuXG4gICAgICByZXR1cm4ge1xuICAgICAgICAnYXBwbGljYXRpb24vanNvbic6IHRlbXBsYXRlLmpvaW4oJ1xcbicpLFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgZ2V0UmVzb3VyY2UocGF0aFBhcnQ6IHN0cmluZyk6IElSZXNvdXJjZSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuY2hpbGRyZW5bcGF0aFBhcnRdO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90cmFja0NoaWxkKHBhdGhQYXJ0OiBzdHJpbmcsIHJlc291cmNlOiBSZXNvdXJjZSkge1xuICAgIHRoaXMuY2hpbGRyZW5bcGF0aFBhcnRdID0gcmVzb3VyY2U7XG4gIH1cblxuICBwdWJsaWMgcmVzb3VyY2VGb3JQYXRoKHBhdGg6IHN0cmluZyk6IFJlc291cmNlIHtcbiAgICBpZiAoIXBhdGgpIHtcbiAgICAgIHJldHVybiB0aGlzO1xuICAgIH1cblxuICAgIGlmIChwYXRoLnN0YXJ0c1dpdGgoJy8nKSkge1xuICAgICAgaWYgKHRoaXMucGF0aCAhPT0gJy8nKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgUGF0aCBtYXkgc3RhcnQgd2l0aCBcIi9cIiBvbmx5IGZvciB0aGUgcmVzb3VyY2UsIGJ1dCB3ZSBhcmUgYXQ6ICR7dGhpcy5wYXRofWApO1xuICAgICAgfVxuXG4gICAgICAvLyB0cmltIHRyYWlsaW5nIFwiL1wiXG4gICAgICByZXR1cm4gdGhpcy5yZXNvdXJjZUZvclBhdGgocGF0aC5zdWJzdHIoMSkpO1xuICAgIH1cblxuICAgIGNvbnN0IHBhcnRzID0gcGF0aC5zcGxpdCgnLycpO1xuICAgIGNvbnN0IG5leHQgPSBwYXJ0cy5zaGlmdCgpO1xuICAgIGlmICghbmV4dCB8fCBuZXh0ID09PSAnJykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdyZXNvdXJjZUZvclBhdGggY2Fubm90IGJlIGNhbGxlZCB3aXRoIGFuIGVtcHR5IHBhdGgnKTtcbiAgICB9XG5cbiAgICBsZXQgcmVzb3VyY2UgPSB0aGlzLmdldFJlc291cmNlKG5leHQpO1xuICAgIGlmICghcmVzb3VyY2UpIHtcbiAgICAgIHJlc291cmNlID0gdGhpcy5hZGRSZXNvdXJjZShuZXh0KTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmVzb3VyY2UucmVzb3VyY2VGb3JQYXRoKHBhcnRzLmpvaW4oJy8nKSk7XG4gIH1cblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgLSBUaHJvd3MgZXJyb3IgaW4gc29tZSB1c2UgY2FzZXMgdGhhdCBoYXZlIGJlZW4gZW5hYmxlZCBzaW5jZSB0aGlzIGRlcHJlY2F0aW9uIG5vdGljZS4gVXNlIGBSZXN0QXBpLnVybEZvclBhdGgoKWAgaW5zdGVhZC5cbiAgICovXG4gIHB1YmxpYyBnZXQgdXJsKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMucmVzdEFwaS51cmxGb3JQYXRoKHRoaXMucGF0aCk7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIFJlc291cmNlIGV4dGVuZHMgUmVzb3VyY2VCYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHBhcmVudFJlc291cmNlPzogSVJlc291cmNlO1xuICBwdWJsaWMgcmVhZG9ubHkgYXBpOiBJUmVzdEFwaTtcbiAgcHVibGljIHJlYWRvbmx5IHJlc291cmNlSWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdEludGVncmF0aW9uPzogSW50ZWdyYXRpb247XG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0TWV0aG9kT3B0aW9ucz86IE1ldGhvZE9wdGlvbnM7XG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM/OiBDb3JzT3B0aW9ucztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVzb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB2YWxpZGF0ZVJlc291cmNlUGF0aFBhcnQocHJvcHMucGF0aFBhcnQpO1xuXG4gICAgdGhpcy5wYXJlbnRSZXNvdXJjZSA9IHByb3BzLnBhcmVudDtcblxuICAgIGlmIChwcm9wcy5wYXJlbnQgaW5zdGFuY2VvZiBSZXNvdXJjZUJhc2UpIHtcbiAgICAgIHByb3BzLnBhcmVudC5fdHJhY2tDaGlsZChwcm9wcy5wYXRoUGFydCwgdGhpcyk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVzb3VyY2VQcm9wczogQ2ZuUmVzb3VyY2VQcm9wcyA9IHtcbiAgICAgIHJlc3RBcGlJZDogcHJvcHMucGFyZW50LmFwaS5yZXN0QXBpSWQsXG4gICAgICBwYXJlbnRJZDogcHJvcHMucGFyZW50LnJlc291cmNlSWQsXG4gICAgICBwYXRoUGFydDogcHJvcHMucGF0aFBhcnQsXG4gICAgfTtcbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5SZXNvdXJjZSh0aGlzLCAnUmVzb3VyY2UnLCByZXNvdXJjZVByb3BzKTtcblxuICAgIHRoaXMucmVzb3VyY2VJZCA9IHJlc291cmNlLnJlZjtcbiAgICB0aGlzLmFwaSA9IHByb3BzLnBhcmVudC5hcGk7XG5cbiAgICAvLyByZW5kZXIgcmVzb3VyY2UgcGF0aCAoc3BlY2lhbCBjYXNlIGZvciByb290KVxuICAgIHRoaXMucGF0aCA9IHByb3BzLnBhcmVudC5wYXRoO1xuICAgIGlmICghdGhpcy5wYXRoLmVuZHNXaXRoKCcvJykpIHsgdGhpcy5wYXRoICs9ICcvJzsgfVxuICAgIHRoaXMucGF0aCArPSBwcm9wcy5wYXRoUGFydDtcblxuICAgIGNvbnN0IGRlcGxveW1lbnQgPSBwcm9wcy5wYXJlbnQuYXBpLmxhdGVzdERlcGxveW1lbnQ7XG4gICAgaWYgKGRlcGxveW1lbnQpIHtcbiAgICAgIGRlcGxveW1lbnQubm9kZS5hZGREZXBlbmRlbmN5KHJlc291cmNlKTtcbiAgICAgIGRlcGxveW1lbnQuYWRkVG9Mb2dpY2FsSWQoeyByZXNvdXJjZTogcmVzb3VyY2VQcm9wcyB9KTtcbiAgICB9XG5cbiAgICAvLyBzZXR1cCBkZWZhdWx0cyBiYXNlZCBvbiBwcm9wZXJ0aWVzIGFuZCBpbmhlcml0IGZyb20gcGFyZW50LiBtZXRob2QgZGVmYXVsdHNcbiAgICAvLyBhcmUgaW5oZXJpdGVkIHBlciBwcm9wZXJ0eSwgc28gY2hpbGRyZW4gY2FuIG92ZXJyaWRlIHBpZWNlbWVhbC5cbiAgICB0aGlzLmRlZmF1bHRJbnRlZ3JhdGlvbiA9IHByb3BzLmRlZmF1bHRJbnRlZ3JhdGlvbiB8fCBwcm9wcy5wYXJlbnQuZGVmYXVsdEludGVncmF0aW9uO1xuICAgIHRoaXMuZGVmYXVsdE1ldGhvZE9wdGlvbnMgPSB7XG4gICAgICAuLi5wcm9wcy5wYXJlbnQuZGVmYXVsdE1ldGhvZE9wdGlvbnMsXG4gICAgICAuLi5wcm9wcy5kZWZhdWx0TWV0aG9kT3B0aW9ucyxcbiAgICB9O1xuICAgIHRoaXMuZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zID0gcHJvcHMuZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zIHx8IHByb3BzLnBhcmVudC5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM7XG5cbiAgICBpZiAodGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMpIHtcbiAgICAgIHRoaXMuYWRkQ29yc1ByZWZsaWdodCh0aGlzLmRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBSZXN0QXBpIGFzc29jaWF0ZWQgd2l0aCB0aGlzIFJlc291cmNlXG4gICAqIEBkZXByZWNhdGVkIC0gVGhyb3dzIGFuIGVycm9yIGlmIHRoaXMgUmVzb3VyY2UgaXMgbm90IGFzc29jaWF0ZWQgd2l0aCBhbiBpbnN0YW5jZSBvZiBgUmVzdEFwaWAuIFVzZSBgYXBpYCBpbnN0ZWFkLlxuICAgKi9cbiAgcHVibGljIGdldCByZXN0QXBpKCk6IFJlc3RBcGkge1xuICAgIGlmICghdGhpcy5wYXJlbnRSZXNvdXJjZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdwYXJlbnRSZXNvdXJjZSB3YXMgdW5leHBlY3RlZGx5IG5vdCBkZWZpbmVkJyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnBhcmVudFJlc291cmNlLnJlc3RBcGk7XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBQcm94eVJlc291cmNlT3B0aW9ucyBleHRlbmRzIFJlc291cmNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBBZGRzIGFuIFwiQU5ZXCIgbWV0aG9kIHRvIHRoaXMgcmVzb3VyY2UuIElmIHNldCB0byBgZmFsc2VgLCB5b3Ugd2lsbCBoYXZlIHRvIGV4cGxpY2l0bHlcbiAgICogYWRkIG1ldGhvZHMgdG8gdGhpcyByZXNvdXJjZSBhZnRlciBpdCdzIGNyZWF0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGFueU1ldGhvZD86IGJvb2xlYW47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUHJveHlSZXNvdXJjZVByb3BzIGV4dGVuZHMgUHJveHlSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBhcmVudCByZXNvdXJjZSBvZiB0aGlzIHJlc291cmNlLiBZb3UgY2FuIGVpdGhlciBwYXNzIGFub3RoZXJcbiAgICogYFJlc291cmNlYCBvYmplY3Qgb3IgYSBgUmVzdEFwaWAgb2JqZWN0IGhlcmUuXG4gICAqL1xuICByZWFkb25seSBwYXJlbnQ6IElSZXNvdXJjZTtcbn1cblxuLyoqXG4gKiBEZWZpbmVzIGEge3Byb3h5K30gZ3JlZWR5IHJlc291cmNlIGFuZCBhbiBBTlkgbWV0aG9kIG9uIGEgcm91dGUuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1zZXQtdXAtc2ltcGxlLXByb3h5Lmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIFByb3h5UmVzb3VyY2UgZXh0ZW5kcyBSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBJZiBgcHJvcHMuYW55TWV0aG9kYCBpcyBgdHJ1ZWAsIHRoaXMgd2lsbCBiZSB0aGUgcmVmZXJlbmNlIHRvIHRoZSAnQU5ZJ1xuICAgKiBtZXRob2QgYXNzb2NpYXRlZCB3aXRoIHRoaXMgcHJveHkgcmVzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYW55TWV0aG9kPzogTWV0aG9kO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBQcm94eVJlc291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBhcmVudDogcHJvcHMucGFyZW50LFxuICAgICAgcGF0aFBhcnQ6ICd7cHJveHkrfScsXG4gICAgICBkZWZhdWx0SW50ZWdyYXRpb246IHByb3BzLmRlZmF1bHRJbnRlZ3JhdGlvbixcbiAgICAgIGRlZmF1bHRNZXRob2RPcHRpb25zOiBwcm9wcy5kZWZhdWx0TWV0aG9kT3B0aW9ucyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGFueU1ldGhvZCA9IHByb3BzLmFueU1ldGhvZCAhPT0gdW5kZWZpbmVkID8gcHJvcHMuYW55TWV0aG9kIDogdHJ1ZTtcbiAgICBpZiAoYW55TWV0aG9kKSB7XG4gICAgICB0aGlzLmFueU1ldGhvZCA9IHRoaXMuYWRkTWV0aG9kKCdBTlknKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYWRkTWV0aG9kKGh0dHBNZXRob2Q6IHN0cmluZywgaW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbiwgb3B0aW9ucz86IE1ldGhvZE9wdGlvbnMpOiBNZXRob2Qge1xuICAgIC8vIEluIGNhc2UgdGhpcyBwcm94eSBpcyBtb3VudGVkIHVuZGVyIHRoZSByb290LCBhbHNvIGFkZCB0aGlzIG1ldGhvZCB0b1xuICAgIC8vIHRoZSByb290IHNvIHRoYXQgZW1wdHkgcGF0aHMgYXJlIHByb3hpZWQgYXMgd2VsbC5cbiAgICBpZiAodGhpcy5wYXJlbnRSZXNvdXJjZSAmJiB0aGlzLnBhcmVudFJlc291cmNlLnBhdGggPT09ICcvJykge1xuICAgICAgLy8gc2tpcCBpZiB0aGUgcm9vdCByZXNvdXJjZSBhbHJlYWR5IGhhcyB0aGlzIG1ldGhvZCBkZWZpbmVkXG4gICAgICBpZiAoISh0aGlzLnBhcmVudFJlc291cmNlLm5vZGUudHJ5RmluZENoaWxkKGh0dHBNZXRob2QpIGluc3RhbmNlb2YgTWV0aG9kKSkge1xuICAgICAgICB0aGlzLnBhcmVudFJlc291cmNlLmFkZE1ldGhvZChodHRwTWV0aG9kLCBpbnRlZ3JhdGlvbiwgb3B0aW9ucyk7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBzdXBlci5hZGRNZXRob2QoaHR0cE1ldGhvZCwgaW50ZWdyYXRpb24sIG9wdGlvbnMpO1xuICB9XG59XG5cbmZ1bmN0aW9uIHZhbGlkYXRlUmVzb3VyY2VQYXRoUGFydChwYXJ0OiBzdHJpbmcpIHtcbiAgLy8gc3RyaXAge30gd2hpY2ggaW5kaWNhdGUgdGhpcyBpcyBhIHBhcmFtZXRlclxuICBpZiAocGFydC5zdGFydHNXaXRoKCd7JykgJiYgcGFydC5lbmRzV2l0aCgnfScpKSB7XG4gICAgcGFydCA9IHBhcnQuc3Vic3RyKDEsIHBhcnQubGVuZ3RoIC0gMik7XG5cbiAgICAvLyBwcm94eSByZXNvdXJjZXMgYXJlIGFsbG93ZWQgdG8gZW5kIHdpdGggYSAnKydcbiAgICBpZiAocGFydC5lbmRzV2l0aCgnKycpKSB7XG4gICAgICBwYXJ0ID0gcGFydC5zdWJzdHIoMCwgcGFydC5sZW5ndGggLSAxKTtcbiAgICB9XG4gIH1cblxuICBpZiAoIS9eW2EtekEtWjAtOVxcLlxcX1xcLV0rJC8udGVzdChwYXJ0KSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgUmVzb3VyY2UncyBwYXRoIHBhcnQgb25seSBhbGxvdyBbYS16QS1aMC05Ll8tXSwgYW4gb3B0aW9uYWwgdHJhaWxpbmcgJysnXG4gICAgICBhbmQgY3VybHkgYnJhY2VzIGF0IHRoZSBiZWdpbm5pbmcgYW5kIHRoZSBlbmQ6ICR7cGFydH1gKTtcbiAgfVxufVxuIl19