"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi
 */
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Gets the "execute-api" ARN
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
}
exports.RestApiBase = RestApiBase;
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, props, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing RestApi that can be configured with additional Methods and Resources.
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzdGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc3RhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esd0NBQXdDO0FBQ3hDLHdDQUFrRztBQUVsRyx1Q0FBMkQ7QUFDM0QsaUVBQWdFO0FBRWhFLDZDQUEwQztBQUMxQywrQ0FBOEQ7QUFDOUQseURBQTZFO0FBRzdFLG1DQUE4QztBQUM5Qyx5REFBK0U7QUFDL0UseUNBQXNFO0FBQ3RFLG1DQUE4QztBQUM5Qyw2Q0FBeUQ7QUFFekQsTUFBTSxjQUFjLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0FBNk16RTs7R0FFRztBQUNILE1BQXNCLFdBQVksU0FBUSxlQUFRO0lBd0RoRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXdCLEVBQUc7UUFDbkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFdBQVcsSUFBSSxFQUFFO1NBQ3RDLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUE1REQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFNO1FBQ2pDLE9BQU8sQ0FBQyxLQUFLLElBQUksSUFBSSxPQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssUUFBUSxJQUFJLGNBQWMsSUFBSSxDQUFDLENBQUM7SUFDckUsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLGdCQUFnQjtRQUN6QixPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyxVQUFVO1FBQ25CLE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQztJQUMxQixDQUFDO0lBdUNEOzs7O09BSUc7SUFDSSxVQUFVLENBQUMsT0FBZSxHQUFHO1FBQ2xDLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0hBQW9ILENBQUMsQ0FBQztTQUN2STtRQUVELE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxhQUFhLENBQUMsRUFBVSxFQUFFLE9BQTBCO1FBQ3pELE1BQU0sVUFBVSxHQUFHLElBQUksd0JBQVUsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQzFDLEdBQUcsT0FBTztZQUNWLE9BQU8sRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDckIsSUFBSSxDQUFDLFdBQVcsR0FBRyxVQUFVLENBQUM7U0FDL0I7UUFDRCxPQUFPLFVBQVUsQ0FBQztJQUNwQixDQUFDO0lBRUQ7O09BRUc7SUFDSSxZQUFZLENBQUMsRUFBVSxFQUFFLFFBQXdCLEVBQUU7UUFDeEQsT0FBTyxJQUFJLHNCQUFTLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxnQkFBZ0IsQ0FBQyxTQUFpQixHQUFHLEVBQUUsT0FBZSxJQUFJLEVBQUUsUUFBZ0IsR0FBRztRQUNwRixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQzVEO1FBRUQsSUFBSSxNQUFNLENBQUMsV0FBVyxFQUFFLEtBQUssS0FBSyxFQUFFO1lBQ2xDLE1BQU0sR0FBRyxHQUFHLENBQUM7U0FDZDtRQUVELE9BQU8sWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDOUIsT0FBTyxFQUFFLGFBQWE7WUFDdEIsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3hCLEdBQUcsRUFBRSxHQUFHO1lBQ1IsWUFBWSxFQUFFLEdBQUcsS0FBSyxJQUFJLE1BQU0sR0FBRyxJQUFJLEVBQUU7U0FDMUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksa0JBQWtCLENBQUMsRUFBVSxFQUFFLE9BQStCO1FBQ25FLE9BQU8sSUFBSSxrQ0FBZSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDbkMsT0FBTyxFQUFFLElBQUk7WUFDYixHQUFHLE9BQU87U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxhQUFhLENBQUMsTUFBYztRQUNqQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDakIsQ0FBQztJQUVTLHVCQUF1QixDQUFDLFdBQXVCO1FBQ3ZELE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDaEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDO1lBQy9ELGVBQWUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsbURBQW1ELENBQUMsQ0FBQztTQUNuSCxDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLGlDQUFVLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMvQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsT0FBTztTQUNoQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRVMsbUJBQW1CLENBQUMsS0FBcUI7UUFDakQsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUNoRSxJQUFJLE1BQU0sRUFBRTtZQUVWLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLHVCQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtnQkFDMUQsV0FBVyxFQUFFLGdEQUFnRDtnQkFDN0QsR0FBRyxFQUFFLElBQUk7Z0JBQ1QsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjthQUMzQyxDQUFDLENBQUM7WUFFSCw2R0FBNkc7WUFDN0csMkRBQTJEO1lBQzNELE1BQU0sU0FBUyxHQUFHLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE1BQU0sQ0FBQztZQUVuRixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksYUFBSyxDQUFDLElBQUksRUFBRSxtQkFBbUIsU0FBUyxFQUFFLEVBQUU7Z0JBQ3JFLFVBQVUsRUFBRSxJQUFJLENBQUMsaUJBQWlCO2dCQUNsQyxHQUFHLEtBQUssQ0FBQyxhQUFhO2FBQ3ZCLENBQUMsQ0FBQztZQUVILElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsQ0FBQztTQUNyRzthQUFNO1lBQ0wsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO2dCQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7YUFDM0U7U0FDRjtJQUNILENBQUM7Q0FDRjtBQXpMRCxrQ0F5TEM7QUFFRDs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ0gsTUFBYSxXQUFZLFNBQVEsV0FBVztJQWUxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3BELE1BQU0sUUFBUSxHQUFHLElBQUksaUNBQVUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2hELElBQUksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN2QixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDcEIsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLElBQUksRUFBRSxZQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMvRSxjQUFjLEVBQUUsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxVQUFVO1lBQ25GLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtTQUM3QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxRQUFRLENBQUM7UUFDbEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQzlCLElBQUksQ0FBQyxxQkFBcUIsR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUM7UUFDekQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBRXRFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQyxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUN4RixJQUFJLGNBQWMsRUFBRTtZQUNsQixJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDeEM7SUFDSCxDQUFDO0NBQ0Y7QUF6Q0Qsa0NBeUNDO0FBaUJEOzs7Ozs7O0dBT0c7QUFDSCxNQUFhLE9BQVEsU0FBUSxXQUFXO0lBNkN0QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXNCLEVBQUc7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFOMUI7O1dBRUc7UUFDYSxZQUFPLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUs1QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGlDQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNoRCxJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDdkIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtZQUNwQixjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLHFCQUFxQixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUM7WUFDckQsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtZQUN4QyxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbEUsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQzdCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxHQUFHLFFBQVEsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFFOUIsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUN4RixJQUFJLGNBQWMsRUFBRTtZQUNsQixJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDeEM7UUFFRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEMsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLElBQUksQ0FBQyxhQUFhLENBQUMsY0FBYyxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUN0RDtRQUVELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUN2RSxJQUFJLENBQUMscUJBQXFCLEdBQUcsUUFBUSxDQUFDLGtCQUFrQixDQUFDO0lBQzNELENBQUM7SUExRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFNBQWlCO1FBQ3pFLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixjQUFTLEdBQUcsU0FBUyxDQUFDO1lBU3hDLENBQUM7WUFQQyxJQUFXLElBQUk7Z0JBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQywwR0FBMEcsQ0FBQyxDQUFDO1lBQzlILENBQUM7WUFFRCxJQUFXLHFCQUFxQjtnQkFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQywySEFBMkgsQ0FBQyxDQUFDO1lBQy9JLENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7O09BR0c7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDeEYsTUFBTSxNQUFPLFNBQVEsV0FBVztZQUFoQzs7Z0JBQ2tCLGNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO2dCQUM1QiwwQkFBcUIsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO2dCQUM3QyxTQUFJLEdBQWMsSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztZQUMzRixDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBNkNEOztPQUVHO0lBQ0gsSUFBVyxHQUFHO1FBQ1osT0FBTyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDM0IsQ0FBQztJQUVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLEVBQVUsRUFBRSxPQUF1QjtRQUNsRCxPQUFPLElBQUksZ0JBQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQzFCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQztZQUNqQixHQUFHLE9BQU87U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRLENBQUMsRUFBVSxFQUFFLEtBQW1CO1FBQzdDLE9BQU8sSUFBSSxhQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUN6QixHQUFHLEtBQUs7WUFDUixPQUFPLEVBQUUsSUFBSTtTQUNkLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLG1CQUFtQixDQUFDLEVBQVUsRUFBRSxLQUE4QjtRQUNuRSxPQUFPLElBQUksbUNBQWdCLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNwQyxHQUFHLEtBQUs7WUFDUixPQUFPLEVBQUUsSUFBSTtTQUNkLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGFBQWEsQ0FBQyxNQUFjO1FBQ2pDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFRDs7T0FFRztJQUNPLFFBQVE7UUFDaEIsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDN0IsT0FBTyxDQUFFLDBDQUEwQyxDQUFFLENBQUM7U0FDdkQ7UUFFRCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFTyxrQkFBa0IsQ0FBQyxLQUFtQjs7UUFDNUMsSUFBSSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsRUFBRTtZQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLG1GQUFtRixDQUFDLENBQUM7U0FDdEc7UUFDRCxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsRUFBRTtZQUMvQixPQUFPO2dCQUNMLEtBQUssRUFBRSxLQUFLLENBQUMscUJBQXFCLENBQUMsS0FBSztnQkFDeEMsY0FBYyxjQUFFLEtBQUssQ0FBQyxxQkFBcUIsMENBQUUsWUFBWSwwQ0FBRSxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsYUFBYSxDQUFDO2FBQ3pHLENBQUM7U0FDSDtRQUNELElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixPQUFPLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxhQUFhLEVBQUUsQ0FBQztTQUN2QztRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Q0FDRjtBQXRKRCwwQkFzSkM7QUF1QkQsSUFBWSxnQkFVWDtBQVZELFdBQVksZ0JBQWdCO0lBQzFCOztPQUVHO0lBQ0gscUNBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCw2Q0FBeUIsQ0FBQTtBQUMzQixDQUFDLEVBVlcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFVM0I7QUFFRCxJQUFZLFlBZVg7QUFmRCxXQUFZLFlBQVk7SUFDdEI7O09BRUc7SUFDSCw2QkFBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCxxQ0FBcUIsQ0FBQTtJQUVyQjs7T0FFRztJQUNILG1DQUFtQixDQUFBO0FBQ3JCLENBQUMsRUFmVyxZQUFZLEdBQVosb0JBQVksS0FBWixvQkFBWSxRQWV2QjtBQUVELE1BQU0sWUFBYSxTQUFRLHVCQUFZO0lBV3JDLFlBQVksR0FBZ0IsRUFBRSxLQUFzQixFQUFFLFVBQWtCO1FBQ3RFLEtBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFdEIsSUFBSSxDQUFDLGNBQWMsR0FBRyxTQUFTLENBQUM7UUFDaEMsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztRQUNuRCxJQUFJLENBQUMsb0JBQW9CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO1FBQ3ZELElBQUksQ0FBQywyQkFBMkIsR0FBRyxLQUFLLENBQUMsMkJBQTJCLENBQUM7UUFDckUsSUFBSSxDQUFDLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDZixJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztRQUM3QixJQUFJLENBQUMsSUFBSSxHQUFHLEdBQUcsQ0FBQztRQUVoQixJQUFJLEdBQUcsWUFBWSxPQUFPLEVBQUU7WUFDMUIsSUFBSSxDQUFDLFFBQVEsR0FBRyxHQUFHLENBQUM7U0FDckI7UUFFRCxJQUFJLElBQUksQ0FBQywyQkFBMkIsRUFBRTtZQUNwQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLENBQUM7U0FDekQ7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyxPQUFPO1FBQ2hCLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2xCLE1BQU0sSUFBSSxLQUFLLENBQUMsaUdBQWlHLENBQUMsQ0FBQztTQUNwSDtRQUNELE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN2QixDQUFDO0NBQ0Y7QUFFRCxTQUFTLE1BQU0sQ0FBQyxFQUFPO0lBQ3JCLE9BQU87QUFDVCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSVZwY0VuZHBvaW50IH0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBDZm5PdXRwdXQsIENvbnN0cnVjdCwgSVJlc291cmNlIGFzIElSZXNvdXJjZUJhc2UsIFJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQXBpRGVmaW5pdGlvbiB9IGZyb20gJy4vYXBpLWRlZmluaXRpb24nO1xuaW1wb3J0IHsgQXBpS2V5LCBBcGlLZXlPcHRpb25zLCBJQXBpS2V5IH0gZnJvbSAnLi9hcGkta2V5JztcbmltcG9ydCB7IENmbkFjY291bnQsIENmblJlc3RBcGkgfSBmcm9tICcuL2FwaWdhdGV3YXkuZ2VuZXJhdGVkJztcbmltcG9ydCB7IENvcnNPcHRpb25zIH0gZnJvbSAnLi9jb3JzJztcbmltcG9ydCB7IERlcGxveW1lbnQgfSBmcm9tICcuL2RlcGxveW1lbnQnO1xuaW1wb3J0IHsgRG9tYWluTmFtZSwgRG9tYWluTmFtZU9wdGlvbnMgfSBmcm9tICcuL2RvbWFpbi1uYW1lJztcbmltcG9ydCB7IEdhdGV3YXlSZXNwb25zZSwgR2F0ZXdheVJlc3BvbnNlT3B0aW9ucyB9IGZyb20gJy4vZ2F0ZXdheS1yZXNwb25zZSc7XG5pbXBvcnQgeyBJbnRlZ3JhdGlvbiB9IGZyb20gJy4vaW50ZWdyYXRpb24nO1xuaW1wb3J0IHsgTWV0aG9kLCBNZXRob2RPcHRpb25zIH0gZnJvbSAnLi9tZXRob2QnO1xuaW1wb3J0IHsgTW9kZWwsIE1vZGVsT3B0aW9ucyB9IGZyb20gJy4vbW9kZWwnO1xuaW1wb3J0IHsgUmVxdWVzdFZhbGlkYXRvciwgUmVxdWVzdFZhbGlkYXRvck9wdGlvbnMgfSBmcm9tICcuL3JlcXVlc3R2YWxpZGF0b3InO1xuaW1wb3J0IHsgSVJlc291cmNlLCBSZXNvdXJjZUJhc2UsIFJlc291cmNlT3B0aW9ucyB9IGZyb20gJy4vcmVzb3VyY2UnO1xuaW1wb3J0IHsgU3RhZ2UsIFN0YWdlT3B0aW9ucyB9IGZyb20gJy4vc3RhZ2UnO1xuaW1wb3J0IHsgVXNhZ2VQbGFuLCBVc2FnZVBsYW5Qcm9wcyB9IGZyb20gJy4vdXNhZ2UtcGxhbic7XG5cbmNvbnN0IFJFU1RBUElfU1lNQk9MID0gU3ltYm9sLmZvcignQGF3cy1jZGsvYXdzLWFwaWdhdGV3YXkuUmVzdEFwaUJhc2UnKTtcblxuZXhwb3J0IGludGVyZmFjZSBJUmVzdEFwaSBleHRlbmRzIElSZXNvdXJjZUJhc2Uge1xuICAvKipcbiAgICogVGhlIElEIG9mIHRoaXMgQVBJIEdhdGV3YXkgUmVzdEFwaS5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVzdEFwaUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZXNvdXJjZSBJRCBvZiB0aGUgcm9vdCByZXNvdXJjZS5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVzdEFwaVJvb3RSZXNvdXJjZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFQSSBHYXRld2F5IGRlcGxveW1lbnQgdGhhdCByZXByZXNlbnRzIHRoZSBsYXRlc3QgY2hhbmdlcyBvZiB0aGUgQVBJLlxuICAgKiBUaGlzIHJlc291cmNlIHdpbGwgYmUgYXV0b21hdGljYWxseSB1cGRhdGVkIGV2ZXJ5IHRpbWUgdGhlIFJFU1QgQVBJIG1vZGVsIGNoYW5nZXMuXG4gICAqIGB1bmRlZmluZWRgIHdoZW4gbm8gZGVwbG95bWVudCBpcyBjb25maWd1cmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbGF0ZXN0RGVwbG95bWVudD86IERlcGxveW1lbnQ7XG5cbiAgLyoqXG4gICAqIFJlcHJlc2VudHMgdGhlIHJvb3QgcmVzb3VyY2UgKFwiL1wiKSBvZiB0aGlzIEFQSS4gVXNlIGl0IHRvIGRlZmluZSB0aGUgQVBJIG1vZGVsOlxuICAgKlxuICAgKiAgICBhcGkucm9vdC5hZGRNZXRob2QoJ0FOWScsIHJlZGlyZWN0VG9Ib21lUGFnZSk7IC8vIFwiQU5ZIC9cIlxuICAgKiAgICBhcGkucm9vdC5hZGRSZXNvdXJjZSgnZnJpZW5kcycpLmFkZE1ldGhvZCgnR0VUJywgZ2V0RnJpZW5kc0hhbmRsZXIpOyAvLyBcIkdFVCAvZnJpZW5kc1wiXG4gICAqXG4gICAqL1xuICByZWFkb25seSByb290OiBJUmVzb3VyY2U7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyB0aGUgcHJvcHMgdGhhdCBhbGwgUmVzdCBBUElzIHNoYXJlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVzdEFwaU9wdGlvbnMgZXh0ZW5kcyBSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIGEgRGVwbG95bWVudCBzaG91bGQgYmUgYXV0b21hdGljYWxseSBjcmVhdGVkIGZvciB0aGlzIEFQSSxcbiAgICogYW5kIHJlY3JlYXRlZCB3aGVuIHRoZSBBUEkgbW9kZWwgKHJlc291cmNlcywgbWV0aG9kcykgY2hhbmdlcy5cbiAgICpcbiAgICogU2luY2UgQVBJIEdhdGV3YXkgZGVwbG95bWVudHMgYXJlIGltbXV0YWJsZSwgV2hlbiB0aGlzIG9wdGlvbiBpcyBlbmFibGVkXG4gICAqIChieSBkZWZhdWx0KSwgYW4gQVdTOjpBcGlHYXRld2F5OjpEZXBsb3ltZW50IHJlc291cmNlIHdpbGwgYXV0b21hdGljYWxseVxuICAgKiBjcmVhdGVkIHdpdGggYSBsb2dpY2FsIElEIHRoYXQgaGFzaGVzIHRoZSBBUEkgbW9kZWwgKG1ldGhvZHMsIHJlc291cmNlc1xuICAgKiBhbmQgb3B0aW9ucykuIFRoaXMgbWVhbnMgdGhhdCB3aGVuIHRoZSBtb2RlbCBjaGFuZ2VzLCB0aGUgbG9naWNhbCBJRCBvZlxuICAgKiB0aGlzIENsb3VkRm9ybWF0aW9uIHJlc291cmNlIHdpbGwgY2hhbmdlLCBhbmQgYSBuZXcgZGVwbG95bWVudCB3aWxsIGJlXG4gICAqIGNyZWF0ZWQuXG4gICAqXG4gICAqIElmIHRoaXMgaXMgc2V0LCBgbGF0ZXN0RGVwbG95bWVudGAgd2lsbCByZWZlciB0byB0aGUgYERlcGxveW1lbnRgIG9iamVjdFxuICAgKiBhbmQgYGRlcGxveW1lbnRTdGFnZWAgd2lsbCByZWZlciB0byBhIGBTdGFnZWAgdGhhdCBwb2ludHMgdG8gdGhpc1xuICAgKiBkZXBsb3ltZW50LiBUbyBjdXN0b21pemUgdGhlIHN0YWdlIG9wdGlvbnMsIHVzZSB0aGUgYGRlcGxveU9wdGlvbnNgXG4gICAqIHByb3BlcnR5LlxuICAgKlxuICAgKiBBIENsb3VkRm9ybWF0aW9uIE91dHB1dCB3aWxsIGFsc28gYmUgZGVmaW5lZCB3aXRoIHRoZSByb290IFVSTCBlbmRwb2ludFxuICAgKiBvZiB0aGlzIFJFU1QgQVBJLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBkZXBsb3k/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciB0aGUgQVBJIEdhdGV3YXkgc3RhZ2UgdGhhdCB3aWxsIGFsd2F5cyBwb2ludCB0byB0aGUgbGF0ZXN0XG4gICAqIGRlcGxveW1lbnQgd2hlbiBgZGVwbG95YCBpcyBlbmFibGVkLiBJZiBgZGVwbG95YCBpcyBkaXNhYmxlZCxcbiAgICogdGhpcyB2YWx1ZSBjYW5ub3QgYmUgc2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEJhc2VkIG9uIGRlZmF1bHRzIG9mIGBTdGFnZU9wdGlvbnNgLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95T3B0aW9ucz86IFN0YWdlT3B0aW9ucztcblxuICAvKipcbiAgICogUmV0YWlucyBvbGQgZGVwbG95bWVudCByZXNvdXJjZXMgd2hlbiB0aGUgQVBJIGNoYW5nZXMuIFRoaXMgYWxsb3dzXG4gICAqIG1hbnVhbGx5IHJldmVydGluZyBzdGFnZXMgdG8gcG9pbnQgdG8gb2xkIGRlcGxveW1lbnRzIHZpYSB0aGUgQVdTXG4gICAqIENvbnNvbGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZXRhaW5EZXBsb3ltZW50cz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgbmFtZSBmb3IgdGhlIEFQSSBHYXRld2F5IFJlc3RBcGkgcmVzb3VyY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSUQgb2YgdGhlIFJlc3RBcGkgY29uc3RydWN0LlxuICAgKi9cbiAgcmVhZG9ubHkgcmVzdEFwaU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEN1c3RvbSBoZWFkZXIgcGFyYW1ldGVycyBmb3IgdGhlIHJlcXVlc3QuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NsaS9sYXRlc3QvcmVmZXJlbmNlL2FwaWdhdGV3YXkvaW1wb3J0LXJlc3QtYXBpLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwYXJhbWV0ZXJzLlxuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIEEgcG9saWN5IGRvY3VtZW50IHRoYXQgY29udGFpbnMgdGhlIHBlcm1pc3Npb25zIGZvciB0aGlzIFJlc3RBcGlcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwb2xpY3kuXG4gICAqL1xuICByZWFkb25seSBwb2xpY3k/OiBpYW0uUG9saWN5RG9jdW1lbnQ7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRvIHJvbGwgYmFjayB0aGUgcmVzb3VyY2UgaWYgYSB3YXJuaW5nIG9jY3VycyB3aGlsZSBBUElcbiAgICogR2F0ZXdheSBpcyBjcmVhdGluZyB0aGUgUmVzdEFwaSByZXNvdXJjZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGZhaWxPbldhcm5pbmdzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ29uZmlndXJlIGEgY3VzdG9tIGRvbWFpbiBuYW1lIGFuZCBtYXAgaXQgdG8gdGhpcyBBUEkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gZG9tYWluIG5hbWUgaXMgZGVmaW5lZCwgdXNlIGBhZGREb21haW5OYW1lYCBvciBkaXJlY3RseSBkZWZpbmUgYSBgRG9tYWluTmFtZWAuXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lPzogRG9tYWluTmFtZU9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgY29uZmlndXJlIGFuIEFXUyBDbG91ZFdhdGNoIHJvbGUgZm9yIEFQSSBHYXRld2F5LlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBjbG91ZFdhdGNoUm9sZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEV4cG9ydCBuYW1lIGZvciB0aGUgQ2ZuT3V0cHV0IGNvbnRhaW5pbmcgdGhlIEFQSSBlbmRwb2ludFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHdoZW4gbm8gZXhwb3J0IG5hbWUgaXMgZ2l2ZW4sIG91dHB1dCB3aWxsIGJlIGNyZWF0ZWQgd2l0aG91dCBleHBvcnRcbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50RXhwb3J0TmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBQcm9wcyB0byBjcmVhdGUgYSBuZXcgaW5zdGFuY2Ugb2YgUmVzdEFwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlc3RBcGlQcm9wcyBleHRlbmRzIFJlc3RBcGlPcHRpb25zIHtcbiAgLyoqXG4gICAqIEEgZGVzY3JpcHRpb24gb2YgdGhlIHB1cnBvc2Ugb2YgdGhpcyBBUEkgR2F0ZXdheSBSZXN0QXBpIHJlc291cmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGRlc2NyaXB0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIGJpbmFyeSBtZWRpYSBtaW1lLXR5cGVzIHRoYXQgYXJlIHN1cHBvcnRlZCBieSB0aGUgUmVzdEFwaVxuICAgKiByZXNvdXJjZSwgc3VjaCBhcyBcImltYWdlL3BuZ1wiIG9yIFwiYXBwbGljYXRpb24vb2N0ZXQtc3RyZWFtXCJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBSZXN0QXBpIHN1cHBvcnRzIG9ubHkgVVRGLTgtZW5jb2RlZCB0ZXh0IHBheWxvYWRzLlxuICAgKi9cbiAgcmVhZG9ubHkgYmluYXJ5TWVkaWFUeXBlcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBBIG51bGxhYmxlIGludGVnZXIgdGhhdCBpcyB1c2VkIHRvIGVuYWJsZSBjb21wcmVzc2lvbiAod2l0aCBub24tbmVnYXRpdmVcbiAgICogYmV0d2VlbiAwIGFuZCAxMDQ4NTc2MCAoMTBNKSBieXRlcywgaW5jbHVzaXZlKSBvciBkaXNhYmxlIGNvbXByZXNzaW9uXG4gICAqICh3aGVuIHVuZGVmaW5lZCkgb24gYW4gQVBJLiBXaGVuIGNvbXByZXNzaW9uIGlzIGVuYWJsZWQsIGNvbXByZXNzaW9uIG9yXG4gICAqIGRlY29tcHJlc3Npb24gaXMgbm90IGFwcGxpZWQgb24gdGhlIHBheWxvYWQgaWYgdGhlIHBheWxvYWQgc2l6ZSBpc1xuICAgKiBzbWFsbGVyIHRoYW4gdGhpcyB2YWx1ZS4gU2V0dGluZyBpdCB0byB6ZXJvIGFsbG93cyBjb21wcmVzc2lvbiBmb3IgYW55XG4gICAqIHBheWxvYWQgc2l6ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDb21wcmVzc2lvbiBpcyBkaXNhYmxlZC5cbiAgICovXG4gIHJlYWRvbmx5IG1pbmltdW1Db21wcmVzc2lvblNpemU/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgQVBJIEdhdGV3YXkgUmVzdEFwaSByZXNvdXJjZSB0aGF0IHlvdSB3YW50IHRvIGNsb25lLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmUuXG4gICAqL1xuICByZWFkb25seSBjbG9uZUZyb20/OiBJUmVzdEFwaTtcblxuICAvKipcbiAgICogVGhlIHNvdXJjZSBvZiB0aGUgQVBJIGtleSBmb3IgbWV0ZXJpbmcgcmVxdWVzdHMgYWNjb3JkaW5nIHRvIGEgdXNhZ2VcbiAgICogcGxhbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBNZXRlcmluZyBpcyBkaXNhYmxlZC5cbiAgICovXG4gIHJlYWRvbmx5IGFwaUtleVNvdXJjZVR5cGU/OiBBcGlLZXlTb3VyY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgRW5kcG9pbnRDb25maWd1cmF0aW9uIHByb3BlcnR5IHR5cGUgc3BlY2lmaWVzIHRoZSBlbmRwb2ludCB0eXBlcyBvZiBhIFJFU1QgQVBJXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtYXBpZ2F0ZXdheS1yZXN0YXBpLWVuZHBvaW50Y29uZmlndXJhdGlvbi5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZW5kcG9pbnQgY29uZmlndXJhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRDb25maWd1cmF0aW9uPzogRW5kcG9pbnRDb25maWd1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgdGhlIGVuZHBvaW50IHR5cGVzIG9mIHRoZSBBUEkuIFVzZSB0aGlzIHByb3BlcnR5IHdoZW4gY3JlYXRpbmdcbiAgICogYW4gQVBJLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGVuZHBvaW50IHR5cGVzLlxuICAgKiBAZGVwcmVjYXRlZCB0aGlzIHByb3BlcnR5IGlzIGRlcHJlY2F0ZWQsIHVzZSBlbmRwb2ludENvbmZpZ3VyYXRpb24gaW5zdGVhZFxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRUeXBlcz86IEVuZHBvaW50VHlwZVtdO1xufVxuXG4vKipcbiAqIFByb3BzIHRvIGluc3RhbnRpYXRlIGEgbmV3IFNwZWNSZXN0QXBpXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3BlY1Jlc3RBcGlQcm9wcyBleHRlbmRzIFJlc3RBcGlPcHRpb25zIHtcbiAgLyoqXG4gICAqIEFuIE9wZW5BUEkgZGVmaW5pdGlvbiBjb21wYXRpYmxlIHdpdGggQVBJIEdhdGV3YXkuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LWltcG9ydC1hcGkuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgYXBpRGVmaW5pdGlvbjogQXBpRGVmaW5pdGlvbjtcbn1cblxuLyoqXG4gKiBCYXNlIGltcGxlbWVudGF0aW9uIHRoYXQgYXJlIGNvbW1vbiB0byB2YXJpb3VzIGltcGxlbWVudGF0aW9ucyBvZiBJUmVzdEFwaVxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgUmVzdEFwaUJhc2UgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElSZXN0QXBpIHtcblxuICAvKipcbiAgICogQ2hlY2tzIGlmIHRoZSBnaXZlbiBvYmplY3QgaXMgYW4gaW5zdGFuY2Ugb2YgUmVzdEFwaUJhc2UuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBfaXNSZXN0QXBpQmFzZSh4OiBhbnkpOiB4IGlzIFJlc3RBcGlCYXNlIHtcbiAgICByZXR1cm4geCAhPT0gbnVsbCAmJiB0eXBlb2YoeCkgPT09ICdvYmplY3QnICYmIFJFU1RBUElfU1lNQk9MIGluIHg7XG4gIH1cblxuICAvKipcbiAgICogQVBJIEdhdGV3YXkgZGVwbG95bWVudCB0aGF0IHJlcHJlc2VudHMgdGhlIGxhdGVzdCBjaGFuZ2VzIG9mIHRoZSBBUEkuXG4gICAqIFRoaXMgcmVzb3VyY2Ugd2lsbCBiZSBhdXRvbWF0aWNhbGx5IHVwZGF0ZWQgZXZlcnkgdGltZSB0aGUgUkVTVCBBUEkgbW9kZWwgY2hhbmdlcy5cbiAgICogVGhpcyB3aWxsIGJlIHVuZGVmaW5lZCBpZiBgZGVwbG95YCBpcyBmYWxzZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgbGF0ZXN0RGVwbG95bWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy5fbGF0ZXN0RGVwbG95bWVudDtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgZmlyc3QgZG9tYWluIG5hbWUgbWFwcGVkIHRvIHRoaXMgQVBJLCBpZiBkZWZpbmVkIHRocm91Z2ggdGhlIGBkb21haW5OYW1lYFxuICAgKiBjb25maWd1cmF0aW9uIHByb3AsIG9yIGFkZGVkIHZpYSBgYWRkRG9tYWluTmFtZWBcbiAgICovXG4gIHB1YmxpYyBnZXQgZG9tYWluTmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZG9tYWluTmFtZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhpcyBBUEkgR2F0ZXdheSBSZXN0QXBpLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJlc3RBcGlJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVzb3VyY2UgSUQgb2YgdGhlIHJvb3QgcmVzb3VyY2UuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSByZXN0QXBpUm9vdFJlc291cmNlSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogUmVwcmVzZW50cyB0aGUgcm9vdCByZXNvdXJjZSBvZiB0aGlzIEFQSSBlbmRwb2ludCAoJy8nKS5cbiAgICogUmVzb3VyY2VzIGFuZCBNZXRob2RzIGFyZSBhZGRlZCB0byB0aGlzIHJlc291cmNlLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJvb3Q6IElSZXNvdXJjZTtcblxuICAvKipcbiAgICogQVBJIEdhdGV3YXkgc3RhZ2UgdGhhdCBwb2ludHMgdG8gdGhlIGxhdGVzdCBkZXBsb3ltZW50IChpZiBkZWZpbmVkKS5cbiAgICpcbiAgICogSWYgYGRlcGxveWAgaXMgZGlzYWJsZWQsIHlvdSB3aWxsIG5lZWQgdG8gZXhwbGljaXRseSBhc3NpZ24gdGhpcyB2YWx1ZSBpbiBvcmRlciB0b1xuICAgKiBzZXQgdXAgaW50ZWdyYXRpb25zLlxuICAgKi9cbiAgcHVibGljIGRlcGxveW1lbnRTdGFnZSE6IFN0YWdlO1xuXG4gIHByaXZhdGUgX2xhdGVzdERlcGxveW1lbnQ/OiBEZXBsb3ltZW50O1xuICBwcml2YXRlIF9kb21haW5OYW1lPzogRG9tYWluTmFtZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVzdEFwaU9wdGlvbnMgPSB7IH0pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMucmVzdEFwaU5hbWUgfHwgaWQsXG4gICAgfSk7XG5cbiAgICBPYmplY3QuZGVmaW5lUHJvcGVydHkodGhpcywgUkVTVEFQSV9TWU1CT0wsIHsgdmFsdWU6IHRydWUgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgVVJMIGZvciBhbiBIVFRQIHBhdGguXG4gICAqXG4gICAqIEZhaWxzIGlmIGBkZXBsb3ltZW50U3RhZ2VgIGlzIG5vdCBzZXQgZWl0aGVyIGJ5IGBkZXBsb3lgIG9yIGV4cGxpY2l0bHkuXG4gICAqL1xuICBwdWJsaWMgdXJsRm9yUGF0aChwYXRoOiBzdHJpbmcgPSAnLycpOiBzdHJpbmcge1xuICAgIGlmICghdGhpcy5kZXBsb3ltZW50U3RhZ2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGRldGVybWluZSBkZXBsb3ltZW50IHN0YWdlIGZvciBBUEkgZnJvbSBcImRlcGxveW1lbnRTdGFnZVwiLiBVc2UgXCJkZXBsb3lcIiBvciBleHBsaWNpdGx5IHNldCBcImRlcGxveW1lbnRTdGFnZVwiJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMuZGVwbG95bWVudFN0YWdlLnVybEZvclBhdGgocGF0aCk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBhbiBBUEkgR2F0ZXdheSBkb21haW4gbmFtZSBhbmQgbWFwcyBpdCB0byB0aGlzIEFQSS5cbiAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QgaWRcbiAgICogQHBhcmFtIG9wdGlvbnMgY3VzdG9tIGRvbWFpbiBvcHRpb25zXG4gICAqL1xuICBwdWJsaWMgYWRkRG9tYWluTmFtZShpZDogc3RyaW5nLCBvcHRpb25zOiBEb21haW5OYW1lT3B0aW9ucyk6IERvbWFpbk5hbWUge1xuICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBuZXcgRG9tYWluTmFtZSh0aGlzLCBpZCwge1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIG1hcHBpbmc6IHRoaXMsXG4gICAgfSk7XG4gICAgaWYgKCF0aGlzLl9kb21haW5OYW1lKSB7XG4gICAgICB0aGlzLl9kb21haW5OYW1lID0gZG9tYWluTmFtZTtcbiAgICB9XG4gICAgcmV0dXJuIGRvbWFpbk5hbWU7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHVzYWdlIHBsYW4uXG4gICAqL1xuICBwdWJsaWMgYWRkVXNhZ2VQbGFuKGlkOiBzdHJpbmcsIHByb3BzOiBVc2FnZVBsYW5Qcm9wcyA9IHt9KTogVXNhZ2VQbGFuIHtcbiAgICByZXR1cm4gbmV3IFVzYWdlUGxhbih0aGlzLCBpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgdGhlIFwiZXhlY3V0ZS1hcGlcIiBBUk5cbiAgICogQHJldHVybnMgVGhlIFwiZXhlY3V0ZS1hcGlcIiBBUk4uXG4gICAqIEBkZWZhdWx0IFwiKlwiIHJldHVybnMgdGhlIGV4ZWN1dGUgQVBJIEFSTiBmb3IgYWxsIG1ldGhvZHMvcmVzb3VyY2VzIGluXG4gICAqIHRoaXMgQVBJLlxuICAgKiBAcGFyYW0gbWV0aG9kIFRoZSBtZXRob2QgKGRlZmF1bHQgYCpgKVxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcmVzb3VyY2UgcGF0aC4gTXVzdCBzdGFydCB3aXRoICcvJyAoZGVmYXVsdCBgKmApXG4gICAqIEBwYXJhbSBzdGFnZSBUaGUgc3RhZ2UgKGRlZmF1bHQgYCpgKVxuICAgKi9cbiAgcHVibGljIGFybkZvckV4ZWN1dGVBcGkobWV0aG9kOiBzdHJpbmcgPSAnKicsIHBhdGg6IHN0cmluZyA9ICcvKicsIHN0YWdlOiBzdHJpbmcgPSAnKicpIHtcbiAgICBpZiAoIXBhdGguc3RhcnRzV2l0aCgnLycpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFwicGF0aFwiIG11c3QgYmVnaW4gd2l0aCBhIFwiL1wiOiAnJHtwYXRofSdgKTtcbiAgICB9XG5cbiAgICBpZiAobWV0aG9kLnRvVXBwZXJDYXNlKCkgPT09ICdBTlknKSB7XG4gICAgICBtZXRob2QgPSAnKic7XG4gICAgfVxuXG4gICAgcmV0dXJuIFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnZXhlY3V0ZS1hcGknLFxuICAgICAgcmVzb3VyY2U6IHRoaXMucmVzdEFwaUlkLFxuICAgICAgc2VwOiAnLycsXG4gICAgICByZXNvdXJjZU5hbWU6IGAke3N0YWdlfS8ke21ldGhvZH0ke3BhdGh9YCxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IGdhdGV3YXkgcmVzcG9uc2UuXG4gICAqL1xuICBwdWJsaWMgYWRkR2F0ZXdheVJlc3BvbnNlKGlkOiBzdHJpbmcsIG9wdGlvbnM6IEdhdGV3YXlSZXNwb25zZU9wdGlvbnMpOiBHYXRld2F5UmVzcG9uc2Uge1xuICAgIHJldHVybiBuZXcgR2F0ZXdheVJlc3BvbnNlKHRoaXMsIGlkLCB7XG4gICAgICByZXN0QXBpOiB0aGlzLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnRlcm5hbCBBUEkgdXNlZCBieSBgTWV0aG9kYCB0byBrZWVwIGFuIGludmVudG9yeSBvZiBtZXRob2RzIGF0IHRoZSBBUElcbiAgICogbGV2ZWwgZm9yIHZhbGlkYXRpb24gcHVycG9zZXMuXG4gICAqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF9hdHRhY2hNZXRob2QobWV0aG9kOiBNZXRob2QpIHtcbiAgICBpZ25vcmUobWV0aG9kKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjb25maWd1cmVDbG91ZFdhdGNoUm9sZShhcGlSZXNvdXJjZTogQ2ZuUmVzdEFwaSkge1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0Nsb3VkV2F0Y2hSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2FwaWdhdGV3YXkuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQW1hem9uQVBJR2F0ZXdheVB1c2hUb0Nsb3VkV2F0Y2hMb2dzJyldLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuQWNjb3VudCh0aGlzLCAnQWNjb3VudCcsIHtcbiAgICAgIGNsb3VkV2F0Y2hSb2xlQXJuOiByb2xlLnJvbGVBcm4sXG4gICAgfSk7XG5cbiAgICByZXNvdXJjZS5ub2RlLmFkZERlcGVuZGVuY3koYXBpUmVzb3VyY2UpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGNvbmZpZ3VyZURlcGxveW1lbnQocHJvcHM6IFJlc3RBcGlPcHRpb25zKSB7XG4gICAgY29uc3QgZGVwbG95ID0gcHJvcHMuZGVwbG95ID09PSB1bmRlZmluZWQgPyB0cnVlIDogcHJvcHMuZGVwbG95O1xuICAgIGlmIChkZXBsb3kpIHtcblxuICAgICAgdGhpcy5fbGF0ZXN0RGVwbG95bWVudCA9IG5ldyBEZXBsb3ltZW50KHRoaXMsICdEZXBsb3ltZW50Jywge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ0F1dG9tYXRpY2FsbHkgY3JlYXRlZCBieSB0aGUgUmVzdEFwaSBjb25zdHJ1Y3QnLFxuICAgICAgICBhcGk6IHRoaXMsXG4gICAgICAgIHJldGFpbkRlcGxveW1lbnRzOiBwcm9wcy5yZXRhaW5EZXBsb3ltZW50cyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBlbmNvZGUgdGhlIHN0YWdlIG5hbWUgaW50byB0aGUgY29uc3RydWN0IGlkLCBzbyBpZiB3ZSBjaGFuZ2UgdGhlIHN0YWdlIG5hbWUsIGl0IHdpbGwgcmVjcmVhdGUgYSBuZXcgc3RhZ2UuXG4gICAgICAvLyBzdGFnZSBuYW1lIGlzIHBhcnQgb2YgdGhlIGVuZHBvaW50LCBzbyB0aGF0IG1ha2VzIHNlbnNlLlxuICAgICAgY29uc3Qgc3RhZ2VOYW1lID0gKHByb3BzLmRlcGxveU9wdGlvbnMgJiYgcHJvcHMuZGVwbG95T3B0aW9ucy5zdGFnZU5hbWUpIHx8ICdwcm9kJztcblxuICAgICAgdGhpcy5kZXBsb3ltZW50U3RhZ2UgPSBuZXcgU3RhZ2UodGhpcywgYERlcGxveW1lbnRTdGFnZS4ke3N0YWdlTmFtZX1gLCB7XG4gICAgICAgIGRlcGxveW1lbnQ6IHRoaXMuX2xhdGVzdERlcGxveW1lbnQsXG4gICAgICAgIC4uLnByb3BzLmRlcGxveU9wdGlvbnMsXG4gICAgICB9KTtcblxuICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCAnRW5kcG9pbnQnLCB7IGV4cG9ydE5hbWU6IHByb3BzLmVuZHBvaW50RXhwb3J0TmFtZSwgdmFsdWU6IHRoaXMudXJsRm9yUGF0aCgpIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAocHJvcHMuZGVwbG95T3B0aW9ucykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBzZXQgXFwnZGVwbG95T3B0aW9uc1xcJyBpZiBcXCdkZXBsb3lcXCcgaXMgZGlzYWJsZWQnKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgUkVTVCBBUEkgaW4gQW1hem9uIEFQSSBHYXRld2F5LCBjcmVhdGVkIHdpdGggYW4gT3BlbkFQSSBzcGVjaWZpY2F0aW9uLlxuICpcbiAqIFNvbWUgcHJvcGVydGllcyBub3JtYWxseSBhY2Nlc3NpYmxlIG9uIEBzZWUge0BsaW5rIFJlc3RBcGl9IC0gc3VjaCBhcyB0aGUgZGVzY3JpcHRpb24gLVxuICogbXVzdCBiZSBkZWNsYXJlZCBpbiB0aGUgc3BlY2lmaWNhdGlvbi4gQWxsIFJlc291cmNlcyBhbmQgTWV0aG9kcyBuZWVkIHRvIGJlIGRlZmluZWQgYXNcbiAqIHBhcnQgb2YgdGhlIE9wZW5BUEkgc3BlY2lmaWNhdGlvbiBmaWxlLCBhbmQgY2Fubm90IGJlIGFkZGVkIHZpYSB0aGUgQ0RLLlxuICpcbiAqIEJ5IGRlZmF1bHQsIHRoZSBBUEkgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGRlcGxveWVkIGFuZCBhY2Nlc3NpYmxlIGZyb20gYVxuICogcHVibGljIGVuZHBvaW50LlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpBcGlHYXRld2F5OjpSZXN0QXBpXG4gKi9cbmV4cG9ydCBjbGFzcyBTcGVjUmVzdEFwaSBleHRlbmRzIFJlc3RBcGlCYXNlIHtcbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGlzIEFQSSBHYXRld2F5IFJlc3RBcGkuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVzdEFwaUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZXNvdXJjZSBJRCBvZiB0aGUgcm9vdCByZXNvdXJjZS5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlSb290UmVzb3VyY2VJZDogc3RyaW5nO1xuXG4gIHB1YmxpYyByZWFkb25seSByb290OiBJUmVzb3VyY2U7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNwZWNSZXN0QXBpUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICBjb25zdCBhcGlEZWZDb25maWcgPSBwcm9wcy5hcGlEZWZpbml0aW9uLmJpbmQodGhpcyk7XG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuUmVzdEFwaSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBuYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIHBvbGljeTogcHJvcHMucG9saWN5LFxuICAgICAgZmFpbE9uV2FybmluZ3M6IHByb3BzLmZhaWxPbldhcm5pbmdzLFxuICAgICAgYm9keTogYXBpRGVmQ29uZmlnLmlubGluZURlZmluaXRpb24gPyBhcGlEZWZDb25maWcuaW5saW5lRGVmaW5pdGlvbiA6IHVuZGVmaW5lZCxcbiAgICAgIGJvZHlTM0xvY2F0aW9uOiBhcGlEZWZDb25maWcuaW5saW5lRGVmaW5pdGlvbiA/IHVuZGVmaW5lZCA6IGFwaURlZkNvbmZpZy5zM0xvY2F0aW9uLFxuICAgICAgcGFyYW1ldGVyczogcHJvcHMucGFyYW1ldGVycyxcbiAgICB9KTtcbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gcmVzb3VyY2U7XG4gICAgdGhpcy5yZXN0QXBpSWQgPSByZXNvdXJjZS5yZWY7XG4gICAgdGhpcy5yZXN0QXBpUm9vdFJlc291cmNlSWQgPSByZXNvdXJjZS5hdHRyUm9vdFJlc291cmNlSWQ7XG4gICAgdGhpcy5yb290ID0gbmV3IFJvb3RSZXNvdXJjZSh0aGlzLCBwcm9wcywgdGhpcy5yZXN0QXBpUm9vdFJlc291cmNlSWQpO1xuXG4gICAgdGhpcy5jb25maWd1cmVEZXBsb3ltZW50KHByb3BzKTtcbiAgICBpZiAocHJvcHMuZG9tYWluTmFtZSkge1xuICAgICAgdGhpcy5hZGREb21haW5OYW1lKCdDdXN0b21Eb21haW4nLCBwcm9wcy5kb21haW5OYW1lKTtcbiAgICB9XG5cbiAgICBjb25zdCBjbG91ZFdhdGNoUm9sZSA9IHByb3BzLmNsb3VkV2F0Y2hSb2xlICE9PSB1bmRlZmluZWQgPyBwcm9wcy5jbG91ZFdhdGNoUm9sZSA6IHRydWU7XG4gICAgaWYgKGNsb3VkV2F0Y2hSb2xlKSB7XG4gICAgICB0aGlzLmNvbmZpZ3VyZUNsb3VkV2F0Y2hSb2xlKHJlc291cmNlKTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBBdHRyaWJ1dGVzIHRoYXQgY2FuIGJlIHNwZWNpZmllZCB3aGVuIGltcG9ydGluZyBhIFJlc3RBcGlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZXN0QXBpQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIEFQSSBHYXRld2F5IFJlc3RBcGkuXG4gICAqL1xuICByZWFkb25seSByZXN0QXBpSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJlc291cmNlIElEIG9mIHRoZSByb290IHJlc291cmNlLlxuICAgKi9cbiAgcmVhZG9ubHkgcm9vdFJlc291cmNlSWQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgUkVTVCBBUEkgaW4gQW1hem9uIEFQSSBHYXRld2F5LlxuICpcbiAqIFVzZSBgYWRkUmVzb3VyY2VgIGFuZCBgYWRkTWV0aG9kYCB0byBjb25maWd1cmUgdGhlIEFQSSBtb2RlbC5cbiAqXG4gKiBCeSBkZWZhdWx0LCB0aGUgQVBJIHdpbGwgYXV0b21hdGljYWxseSBiZSBkZXBsb3llZCBhbmQgYWNjZXNzaWJsZSBmcm9tIGFcbiAqIHB1YmxpYyBlbmRwb2ludC5cbiAqL1xuZXhwb3J0IGNsYXNzIFJlc3RBcGkgZXh0ZW5kcyBSZXN0QXBpQmFzZSB7XG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgUmVzdEFwaS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVJlc3RBcGlJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCByZXN0QXBpSWQ6IHN0cmluZyk6IElSZXN0QXBpIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElSZXN0QXBpIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSByZXN0QXBpSWQgPSByZXN0QXBpSWQ7XG5cbiAgICAgIHB1YmxpYyBnZXQgcm9vdCgpOiBJUmVzb3VyY2Uge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ3Jvb3QgaXMgbm90IGNvbmZpZ3VyZWQgd2hlbiBpbXBvcnRlZCB1c2luZyBgZnJvbVJlc3RBcGlJZCgpYC4gVXNlIGBmcm9tUmVzdEFwaUF0dHJpYnV0ZXMoKWAgQVBJIGluc3RlYWQuJyk7XG4gICAgICB9XG5cbiAgICAgIHB1YmxpYyBnZXQgcmVzdEFwaVJvb3RSZXNvdXJjZUlkKCk6IHN0cmluZyB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcigncmVzdEFwaVJvb3RSZXNvdXJjZUlkIGlzIG5vdCBjb25maWd1cmVkIHdoZW4gaW1wb3J0ZWQgdXNpbmcgYGZyb21SZXN0QXBpSWQoKWAuIFVzZSBgZnJvbVJlc3RBcGlBdHRyaWJ1dGVzKClgIEFQSSBpbnN0ZWFkLicpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIFJlc3RBcGkgdGhhdCBjYW4gYmUgY29uZmlndXJlZCB3aXRoIGFkZGl0aW9uYWwgTWV0aG9kcyBhbmQgUmVzb3VyY2VzLlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21SZXN0QXBpQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogUmVzdEFwaUF0dHJpYnV0ZXMpOiBJUmVzdEFwaSB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzdEFwaUJhc2Uge1xuICAgICAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlJZCA9IGF0dHJzLnJlc3RBcGlJZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSByZXN0QXBpUm9vdFJlc291cmNlSWQgPSBhdHRycy5yb290UmVzb3VyY2VJZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSByb290OiBJUmVzb3VyY2UgPSBuZXcgUm9vdFJlc291cmNlKHRoaXMsIHt9LCB0aGlzLnJlc3RBcGlSb290UmVzb3VyY2VJZCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSByZXN0QXBpSWQ6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgcm9vdDogSVJlc291cmNlO1xuXG4gIHB1YmxpYyByZWFkb25seSByZXN0QXBpUm9vdFJlc291cmNlSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2YgbWV0aG9kcyBib3VuZCB0byB0aGlzIFJlc3RBcGlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtZXRob2RzID0gbmV3IEFycmF5PE1ldGhvZD4oKTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVzdEFwaVByb3BzID0geyB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5SZXN0QXBpKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgcG9saWN5OiBwcm9wcy5wb2xpY3ksXG4gICAgICBmYWlsT25XYXJuaW5nczogcHJvcHMuZmFpbE9uV2FybmluZ3MsXG4gICAgICBtaW5pbXVtQ29tcHJlc3Npb25TaXplOiBwcm9wcy5taW5pbXVtQ29tcHJlc3Npb25TaXplLFxuICAgICAgYmluYXJ5TWVkaWFUeXBlczogcHJvcHMuYmluYXJ5TWVkaWFUeXBlcyxcbiAgICAgIGVuZHBvaW50Q29uZmlndXJhdGlvbjogdGhpcy5jb25maWd1cmVFbmRwb2ludHMocHJvcHMpLFxuICAgICAgYXBpS2V5U291cmNlVHlwZTogcHJvcHMuYXBpS2V5U291cmNlVHlwZSxcbiAgICAgIGNsb25lRnJvbTogcHJvcHMuY2xvbmVGcm9tID8gcHJvcHMuY2xvbmVGcm9tLnJlc3RBcGlJZCA6IHVuZGVmaW5lZCxcbiAgICAgIHBhcmFtZXRlcnM6IHByb3BzLnBhcmFtZXRlcnMsXG4gICAgfSk7XG4gICAgdGhpcy5ub2RlLmRlZmF1bHRDaGlsZCA9IHJlc291cmNlO1xuICAgIHRoaXMucmVzdEFwaUlkID0gcmVzb3VyY2UucmVmO1xuXG4gICAgY29uc3QgY2xvdWRXYXRjaFJvbGUgPSBwcm9wcy5jbG91ZFdhdGNoUm9sZSAhPT0gdW5kZWZpbmVkID8gcHJvcHMuY2xvdWRXYXRjaFJvbGUgOiB0cnVlO1xuICAgIGlmIChjbG91ZFdhdGNoUm9sZSkge1xuICAgICAgdGhpcy5jb25maWd1cmVDbG91ZFdhdGNoUm9sZShyZXNvdXJjZSk7XG4gICAgfVxuXG4gICAgdGhpcy5jb25maWd1cmVEZXBsb3ltZW50KHByb3BzKTtcbiAgICBpZiAocHJvcHMuZG9tYWluTmFtZSkge1xuICAgICAgdGhpcy5hZGREb21haW5OYW1lKCdDdXN0b21Eb21haW4nLCBwcm9wcy5kb21haW5OYW1lKTtcbiAgICB9XG5cbiAgICB0aGlzLnJvb3QgPSBuZXcgUm9vdFJlc291cmNlKHRoaXMsIHByb3BzLCByZXNvdXJjZS5hdHRyUm9vdFJlc291cmNlSWQpO1xuICAgIHRoaXMucmVzdEFwaVJvb3RSZXNvdXJjZUlkID0gcmVzb3VyY2UuYXR0clJvb3RSZXNvdXJjZUlkO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBkZXBsb3llZCByb290IFVSTCBvZiB0aGlzIFJFU1QgQVBJLlxuICAgKi9cbiAgcHVibGljIGdldCB1cmwoKSB7XG4gICAgcmV0dXJuIHRoaXMudXJsRm9yUGF0aCgpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhbiBBcGlLZXlcbiAgICovXG4gIHB1YmxpYyBhZGRBcGlLZXkoaWQ6IHN0cmluZywgb3B0aW9ucz86IEFwaUtleU9wdGlvbnMpOiBJQXBpS2V5IHtcbiAgICByZXR1cm4gbmV3IEFwaUtleSh0aGlzLCBpZCwge1xuICAgICAgcmVzb3VyY2VzOiBbdGhpc10sXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgbW9kZWwuXG4gICAqL1xuICBwdWJsaWMgYWRkTW9kZWwoaWQ6IHN0cmluZywgcHJvcHM6IE1vZGVsT3B0aW9ucyk6IE1vZGVsIHtcbiAgICByZXR1cm4gbmV3IE1vZGVsKHRoaXMsIGlkLCB7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIHJlc3RBcGk6IHRoaXMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIG5ldyByZXF1ZXN0IHZhbGlkYXRvci5cbiAgICovXG4gIHB1YmxpYyBhZGRSZXF1ZXN0VmFsaWRhdG9yKGlkOiBzdHJpbmcsIHByb3BzOiBSZXF1ZXN0VmFsaWRhdG9yT3B0aW9ucyk6IFJlcXVlc3RWYWxpZGF0b3Ige1xuICAgIHJldHVybiBuZXcgUmVxdWVzdFZhbGlkYXRvcih0aGlzLCBpZCwge1xuICAgICAgLi4ucHJvcHMsXG4gICAgICByZXN0QXBpOiB0aGlzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEludGVybmFsIEFQSSB1c2VkIGJ5IGBNZXRob2RgIHRvIGtlZXAgYW4gaW52ZW50b3J5IG9mIG1ldGhvZHMgYXQgdGhlIEFQSVxuICAgKiBsZXZlbCBmb3IgdmFsaWRhdGlvbiBwdXJwb3Nlcy5cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2F0dGFjaE1ldGhvZChtZXRob2Q6IE1ldGhvZCkge1xuICAgIHRoaXMubWV0aG9kcy5wdXNoKG1ldGhvZCk7XG4gIH1cblxuICAvKipcbiAgICogUGVyZm9ybXMgdmFsaWRhdGlvbiBvZiB0aGUgUkVTVCBBUEkuXG4gICAqL1xuICBwcm90ZWN0ZWQgdmFsaWRhdGUoKSB7XG4gICAgaWYgKHRoaXMubWV0aG9kcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiBbIFwiVGhlIFJFU1QgQVBJIGRvZXNuJ3QgY29udGFpbiBhbnkgbWV0aG9kc1wiIF07XG4gICAgfVxuXG4gICAgcmV0dXJuIFtdO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVFbmRwb2ludHMocHJvcHM6IFJlc3RBcGlQcm9wcyk6IENmblJlc3RBcGkuRW5kcG9pbnRDb25maWd1cmF0aW9uUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgIGlmIChwcm9wcy5lbmRwb2ludFR5cGVzICYmIHByb3BzLmVuZHBvaW50Q29uZmlndXJhdGlvbikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmx5IG9uZSBvZiB0aGUgUmVzdEFwaSBwcm9wcywgZW5kcG9pbnRUeXBlcyBvciBlbmRwb2ludENvbmZpZ3VyYXRpb24sIGlzIGFsbG93ZWQnKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLmVuZHBvaW50Q29uZmlndXJhdGlvbikge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgdHlwZXM6IHByb3BzLmVuZHBvaW50Q29uZmlndXJhdGlvbi50eXBlcyxcbiAgICAgICAgdnBjRW5kcG9pbnRJZHM6IHByb3BzLmVuZHBvaW50Q29uZmlndXJhdGlvbj8udnBjRW5kcG9pbnRzPy5tYXAodnBjRW5kcG9pbnQgPT4gdnBjRW5kcG9pbnQudnBjRW5kcG9pbnRJZCksXG4gICAgICB9O1xuICAgIH1cbiAgICBpZiAocHJvcHMuZW5kcG9pbnRUeXBlcykge1xuICAgICAgcmV0dXJuIHsgdHlwZXM6IHByb3BzLmVuZHBvaW50VHlwZXMgfTtcbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxufVxuXG4vKipcbiAqIFRoZSBlbmRwb2ludCBjb25maWd1cmF0aW9uIG9mIGEgUkVTVCBBUEksIGluY2x1ZGluZyBWUENzIGFuZCBlbmRwb2ludCB0eXBlcy5cbiAqXG4gKiBFbmRwb2ludENvbmZpZ3VyYXRpb24gaXMgYSBwcm9wZXJ0eSBvZiB0aGUgQVdTOjpBcGlHYXRld2F5OjpSZXN0QXBpIHJlc291cmNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVuZHBvaW50Q29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgZW5kcG9pbnQgdHlwZXMgb2YgYW4gQVBJIG9yIGl0cyBjdXN0b20gZG9tYWluIG5hbWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gZW5kcG9pbnQgdHlwZXMuXG4gICAqL1xuICByZWFkb25seSB0eXBlczogRW5kcG9pbnRUeXBlW107XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBWUEMgRW5kcG9pbnRzIGFnYWluc3Qgd2hpY2ggdG8gY3JlYXRlIFJvdXRlNTMgQUxJQVNlc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIEFMSUFTZXMgYXJlIGNyZWF0ZWQgZm9yIHRoZSBlbmRwb2ludC5cbiAgICovXG4gIHJlYWRvbmx5IHZwY0VuZHBvaW50cz86IElWcGNFbmRwb2ludFtdO1xufVxuXG5leHBvcnQgZW51bSBBcGlLZXlTb3VyY2VUeXBlIHtcbiAgLyoqXG4gICAqIFRvIHJlYWQgdGhlIEFQSSBrZXkgZnJvbSB0aGUgYFgtQVBJLUtleWAgaGVhZGVyIG9mIGEgcmVxdWVzdC5cbiAgICovXG4gIEhFQURFUiA9ICdIRUFERVInLFxuXG4gIC8qKlxuICAgKiBUbyByZWFkIHRoZSBBUEkga2V5IGZyb20gdGhlIGBVc2FnZUlkZW50aWZpZXJLZXlgIGZyb20gYSBjdXN0b20gYXV0aG9yaXplci5cbiAgICovXG4gIEFVVEhPUklaRVIgPSAnQVVUSE9SSVpFUicsXG59XG5cbmV4cG9ydCBlbnVtIEVuZHBvaW50VHlwZSB7XG4gIC8qKlxuICAgKiBGb3IgYW4gZWRnZS1vcHRpbWl6ZWQgQVBJIGFuZCBpdHMgY3VzdG9tIGRvbWFpbiBuYW1lLlxuICAgKi9cbiAgRURHRSA9ICdFREdFJyxcblxuICAvKipcbiAgICogRm9yIGEgcmVnaW9uYWwgQVBJIGFuZCBpdHMgY3VzdG9tIGRvbWFpbiBuYW1lLlxuICAgKi9cbiAgUkVHSU9OQUwgPSAnUkVHSU9OQUwnLFxuXG4gIC8qKlxuICAgKiBGb3IgYSBwcml2YXRlIEFQSSBhbmQgaXRzIGN1c3RvbSBkb21haW4gbmFtZS5cbiAgICovXG4gIFBSSVZBVEUgPSAnUFJJVkFURSdcbn1cblxuY2xhc3MgUm9vdFJlc291cmNlIGV4dGVuZHMgUmVzb3VyY2VCYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHBhcmVudFJlc291cmNlPzogSVJlc291cmNlO1xuICBwdWJsaWMgcmVhZG9ubHkgYXBpOiBSZXN0QXBpQmFzZTtcbiAgcHVibGljIHJlYWRvbmx5IHJlc291cmNlSWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRJbnRlZ3JhdGlvbj86IEludGVncmF0aW9uIHwgdW5kZWZpbmVkO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdE1ldGhvZE9wdGlvbnM/OiBNZXRob2RPcHRpb25zIHwgdW5kZWZpbmVkO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zPzogQ29yc09wdGlvbnMgfCB1bmRlZmluZWQ7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfcmVzdEFwaT86IFJlc3RBcGk7XG5cbiAgY29uc3RydWN0b3IoYXBpOiBSZXN0QXBpQmFzZSwgcHJvcHM6IFJlc291cmNlT3B0aW9ucywgcmVzb3VyY2VJZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoYXBpLCAnRGVmYXVsdCcpO1xuXG4gICAgdGhpcy5wYXJlbnRSZXNvdXJjZSA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLmRlZmF1bHRJbnRlZ3JhdGlvbiA9IHByb3BzLmRlZmF1bHRJbnRlZ3JhdGlvbjtcbiAgICB0aGlzLmRlZmF1bHRNZXRob2RPcHRpb25zID0gcHJvcHMuZGVmYXVsdE1ldGhvZE9wdGlvbnM7XG4gICAgdGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMgPSBwcm9wcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM7XG4gICAgdGhpcy5hcGkgPSBhcGk7XG4gICAgdGhpcy5yZXNvdXJjZUlkID0gcmVzb3VyY2VJZDtcbiAgICB0aGlzLnBhdGggPSAnLyc7XG5cbiAgICBpZiAoYXBpIGluc3RhbmNlb2YgUmVzdEFwaSkge1xuICAgICAgdGhpcy5fcmVzdEFwaSA9IGFwaTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMpIHtcbiAgICAgIHRoaXMuYWRkQ29yc1ByZWZsaWdodCh0aGlzLmRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEdldCB0aGUgUmVzdEFwaSBhc3NvY2lhdGVkIHdpdGggdGhpcyBSZXNvdXJjZS5cbiAgICogQGRlcHJlY2F0ZWQgLSBUaHJvd3MgYW4gZXJyb3IgaWYgdGhpcyBSZXNvdXJjZSBpcyBub3QgYXNzb2NpYXRlZCB3aXRoIGFuIGluc3RhbmNlIG9mIGBSZXN0QXBpYC4gVXNlIGBhcGlgIGluc3RlYWQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHJlc3RBcGkoKTogUmVzdEFwaSB7XG4gICAgaWYgKCF0aGlzLl9yZXN0QXBpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Jlc3RBcGkgaXMgbm90IGF2YWlsYWJsZSBvbiBSZXNvdXJjZSBub3QgY29ubmVjdGVkIHRvIGFuIGluc3RhbmNlIG9mIFJlc3RBcGkuIFVzZSBgYXBpYCBpbnN0ZWFkJyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLl9yZXN0QXBpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGlnbm9yZShfeDogYW55KSB7XG4gIHJldHVybjtcbn0iXX0=