from __future__ import annotations
import collections
import dataclasses
import logging
import os
import pathlib
import requests
import threading
import typing

CHUNK_SIZE = 1048576
DOWNLOAD_SUFFIX = ".download"
TIMEOUT = 60.0


def size_to_string(size: int) -> str:
    if size < 1000:
        return f'{"{:.0f}".format(size)} B'
    if size < 1000000:
        return f'{"{:.2f}".format(size / 1000)} kB'
    if size < 1000000000:
        return f'{"{:.2f}".format(size / 1000000)} MB'
    if size < 1000000000000:
        return f'{"{:.2f}".format(size / 1000000000)} GB'
    return f'{"{:.2f}".format(size / 1000000000000)} TB'


def run(session: requests.Session, task: tuple[str, pathlib.Path, pathlib.Path]):
    url, path, download_path = task
    logging.info(f'downloading "{url}" to "{path}"')
    response: typing.Optional[requests.Response] = None
    mode = "wb"
    if download_path.is_file():
        response = session.get(
            url,
            timeout=TIMEOUT,
            stream=True,
            headers={"Range": f"bytes={download_path.stat().st_size}-"},
        )
        if response.status_code == 206:
            mode = "ab"
        else:
            logging.info(f'range request failed for "{url}" ({response.status_code})')
            response = None
    if response is None:
        response = session.get(
            url,
            timeout=TIMEOUT,
            stream=True,
        )
        response.raise_for_status()
    with open(download_path, mode) as download_file:
        for chunk in response.iter_content(chunk_size=CHUNK_SIZE):
            download_file.write(chunk)
    response.close()
    response.raise_for_status()
    download_path.rename(path)


def worker_target(queue: collections.deque):
    with requests.Session() as session:
        while True:
            try:
                run(session=session, task=queue.popleft())
            except IndexError:
                break


@dataclasses.dataclass
class Series:
    name: str
    description: str
    scale_to_sizes: dict[int, tuple[int, ...]]
    url_pattern: str

    def size(self, scales: typing.Optional[set[int]] = None):
        if scales is None:
            scales = set(self.scale_to_sizes.keys())
        return sum(
            sum(sizes)
            for scale, sizes in self.scale_to_sizes.items()
            if scale in scales
        )

    def size_as_string(self, scales: typing.Optional[set[int]] = None):
        return size_to_string(self.size(scales=scales))

    def index_files(
        self,
        cache_directory: typing.Union[bytes, str, os.PathLike],
        scales: typing.Optional[set[int]] = None,
    ) -> list[pathlib.Path]:
        if isinstance(cache_directory, bytes):
            cache_directory = pathlib.Path(cache_directory.decode()).resolve()
        elif isinstance(cache_directory, str):
            cache_directory = pathlib.Path(cache_directory).resolve()
        else:
            cache_directory = pathlib.Path(cache_directory)
        assert isinstance(cache_directory, pathlib.Path)
        cache_directory = cache_directory.resolve() / self.name
        cache_directory.mkdir(parents=True, exist_ok=True)
        if scales is None:
            scales = set(self.scale_to_sizes.keys())
        urls: list[str] = []
        paths: list[pathlib.Path] = []
        sizes: list[int] = []
        for scale in scales:
            if len(self.scale_to_sizes[scale]) == 1:
                url = self.url_pattern.format(scale=f"{scale:02d}", index="")
                urls.append(url)
                paths.append(cache_directory / url.rsplit("/", 1)[1])
                sizes.append(self.scale_to_sizes[scale][0])
            else:
                for index, size in enumerate(self.scale_to_sizes[scale]):
                    url = self.url_pattern.format(
                        scale=f"{scale:02d}", index=f"-{index:02d}"
                    )
                    urls.append(url)
                    paths.append(cache_directory / url.rsplit("/", 1)[1])
                    sizes.append(size)
        download_size = 0
        tasks = collections.deque()
        for url, path, size in zip(urls, paths, sizes):
            if not path.is_file():
                download_path = pathlib.Path(f"{path}{DOWNLOAD_SUFFIX}")
                tasks.append((url, path, download_path))
                if download_path.is_file():
                    download_size += max(0, size - download_path.stat().st_size)
                else:
                    download_size += size
        if len(tasks) > 0:
            logging.info(
                f"downloading {len(tasks)} file{'' if len(tasks) == 1 else 's'} ({size_to_string(download_size)})"
            )
        if len(tasks) == 1:
            with requests.Session() as session:
                run(session=session, task=tasks.popleft())
        elif len(tasks) > 1:
            cpu_count = os.cpu_count()
            if cpu_count is None:
                cpu_count = 8
            queue = collections.deque(tasks)
            workers = tuple(
                threading.Thread(target=worker_target, daemon=True, args=(queue,))
                for _ in range(0, min(len(tasks), cpu_count))
            )
            for worker in workers:
                worker.start()
            for worker in workers:
                worker.join()
        return paths


series_4100 = Series(
    name="4100",
    description="built from the Tycho-2 catalog, good for images wider than 1 degree, recommended",
    scale_to_sizes={
        7: (164995200,),
        8: (94550400,),
        9: (49772160,),
        10: (24871680,),
        11: (10206720,),
        12: (5296320,),
        13: (2733120,),
        14: (1382400,),
        15: (740160,),
        16: (408960,),
        17: (247680,),
        18: (187200,),
        19: (144000,),
    },
    url_pattern="http://data.astrometry.net/4100/index-41{scale}.fits",
)

series_4200 = Series(
    name="4200",
    description="built from the near-infared 2MASS survey, runs out of stars at the low end, most users will probably prefer series_4100 or series_5200",
    scale_to_sizes={
        0: (
            246795840,
            364579200,
            420926400,
            416496960,
            214660800,
            338376960,
            84029760,
            213698880,
            157475520,
            76285440,
            260159040,
            164030400,
            411209280,
            418233600,
            410235840,
            402595200,
            99645120,
            240500160,
            100776960,
            220417920,
            369328320,
            272295360,
            412148160,
            425825280,
            249143040,
            152904960,
            137122560,
            72100800,
            436216320,
            370733760,
            433333440,
            408332160,
            254223360,
            95057280,
            217563840,
            95339520,
            420033600,
            410950080,
            408844800,
            386363520,
            432164160,
            405305280,
            435795840,
            354392640,
            341804160,
            416839680,
            190644480,
            349151040,
        ),
        1: (
            187090560,
            209136960,
            218096640,
            217244160,
            180342720,
            204203520,
            137736000,
            180383040,
            167486400,
            131322240,
            189083520,
            169917120,
            216938880,
            217284480,
            216653760,
            215182080,
            147634560,
            185886720,
            148864320,
            181843200,
            209808000,
            192003840,
            216794880,
            218456640,
            187617600,
            164430720,
            160580160,
            127255680,
            218767680,
            210222720,
            218730240,
            216665280,
            188078400,
            145068480,
            181690560,
            144936000,
            217912320,
            216702720,
            216339840,
            212448960,
            218692800,
            215902080,
            218718720,
            206504640,
            204203520,
            217664640,
            177580800,
            205701120,
        ),
        2: (
            105431040,
            108570240,
            109169280,
            109166400,
            103504320,
            108227520,
            95267520,
            103510080,
            100785600,
            94360320,
            106251840,
            101142720,
            109166400,
            109166400,
            109154880,
            109120320,
            96802560,
            105384960,
            96923520,
            104480640,
            108763200,
            106329600,
            109146240,
            109152000,
            105773760,
            100347840,
            99434880,
            93795840,
            109172160,
            108898560,
            109172160,
            109149120,
            105359040,
            96379200,
            104374080,
            96384960,
            109172160,
            109157760,
            109140480,
            108996480,
            109172160,
            109117440,
            109160640,
            108616320,
            108547200,
            109169280,
            102824640,
            108581760,
        ),
        3: (
            54869760,
            54924480,
            54924480,
            54924480,
            54803520,
            54921600,
            53867520,
            54803520,
            54645120,
            53628480,
            54889920,
            54673920,
            54924480,
            54924480,
            54924480,
            54924480,
            54198720,
            54872640,
            54239040,
            54838080,
            54924480,
            54889920,
            54924480,
            54918720,
            54889920,
            54552960,
            54478080,
            53478720,
            54924480,
            54924480,
            54924480,
            54924480,
            54858240,
            54112320,
            54840960,
            54106560,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54815040,
            54924480,
        ),
        4: (
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27345600,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27345600,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
        ),
        5: (
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
            54924480,
        ),
        6: (
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
            27354240,
        ),
        7: (
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
            13786560,
        ),
        8: (81835200,),
        9: (41178240,),
        10: (20517120,),
        11: (8017920,),
        12: (4167360,),
        13: (2157120,),
        14: (1100160,),
        15: (596160,),
        16: (339840,),
        17: (213120,),
        18: (164160,),
        19: (132480,),
    },
    url_pattern="http://data.astrometry.net/4200/index-42{scale}{index}.fits",
)

series_5000 = Series(
    name="5000",
    description="an older version from Gaia-DR2 but without Tycho-2 stars merged in, our belief is that series_5200 will work better than this one",
    scale_to_sizes={
        0: (
            602527680,
            821560320,
            894294720,
            894355200,
            589901760,
            794934720,
            361938240,
            598510080,
            570792960,
            380790720,
            707958720,
            550376640,
            893496960,
            895158720,
            891624960,
            881403840,
            420030720,
            630408960,
            381280320,
            602064000,
            839197440,
            672670080,
            892581120,
            899565120,
            708750720,
            502954560,
            564868800,
            363983040,
            900936000,
            869284800,
            900017280,
            897598080,
            707420160,
            431991360,
            612426240,
            378023040,
            899455680,
            893678400,
            891017280,
            857471040,
            900967680,
            891339840,
            900054720,
            848606400,
            845691840,
            899769600,
            619254720,
            834531840,
        ),
        1: (
            401878080,
            445181760,
            450904320,
            451601280,
            401022720,
            444389760,
            320886720,
            405817920,
            399574080,
            328492800,
            436150080,
            392316480,
            451644480,
            451638720,
            451630080,
            451419840,
            343912320,
            415549440,
            328688640,
            408009600,
            448542720,
            420592320,
            451618560,
            451607040,
            435870720,
            376116480,
            399395520,
            321534720,
            451635840,
            451183680,
            451425600,
            451661760,
            427754880,
            348387840,
            411612480,
            327087360,
            451578240,
            451647360,
            451586880,
            450547200,
            451653120,
            451601280,
            451391040,
            450587520,
            450780480,
            451632960,
            416954880,
            449032320,
        ),
        2: (
            223741440,
            225299520,
            225218880,
            225339840,
            223796160,
            225311040,
            213871680,
            224352000,
            224308800,
            216244800,
            225256320,
            223862400,
            225342720,
            225342720,
            225342720,
            225336960,
            219458880,
            224743680,
            216463680,
            224354880,
            225216000,
            224668800,
            225342720,
            225339840,
            225264960,
            222577920,
            224510400,
            213920640,
            225339840,
            225331200,
            225302400,
            225342720,
            225011520,
            219317760,
            224654400,
            215078400,
            225336960,
            225342720,
            225342720,
            225342720,
            225342720,
            225342720,
            225276480,
            225342720,
            225342720,
            225336960,
            224930880,
            225339840,
        ),
        3: (
            113333760,
            113336640,
            113328000,
            113336640,
            113336640,
            113336640,
            113310720,
            113336640,
            113336640,
            113330880,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113333760,
            113339520,
            113328000,
            113336640,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113316480,
            113339520,
            113339520,
            113336640,
            113339520,
            113339520,
            113336640,
            113339520,
            113322240,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
            113330880,
            113339520,
            113339520,
            113339520,
            113339520,
            113339520,
        ),
        4: (
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
            56401920,
        ),
        5: (
            113336640,
            113336640,
            113336640,
            113336640,
            113336640,
            113336640,
            113336640,
            113336640,
            113336640,
            113336640,
            113339520,
            113339520,
        ),
        6: (
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56399040,
            56401920,
            56401920,
        ),
        7: (
            28393920,
            28393920,
            28393920,
            28393920,
            28393920,
            28393920,
            28393920,
            28393920,
            28393920,
            28393920,
            28396800,
            28396800,
        ),
    },
    url_pattern="http://data.astrometry.net/5000/index-50{scale}{index}.fits",
)

series_5200 = Series(
    name="5200",
    description="LIGHT version built from Tycho-2 + Gaia-DR2, good for images narrower than 1 degree, combine with 4100-series for broader scale coverage, the LIGHT version contains smaller files with no additional Gaia-DR2 information tagged along, recommended",
    scale_to_sizes={
        0: (
            331254720,
            390939840,
            403902720,
            404320320,
            330050880,
            385070400,
            220253760,
            335672640,
            330275520,
            232519680,
            369031680,
            321315840,
            404179200,
            404461440,
            403865280,
            402094080,
            256230720,
            346944960,
            232718400,
            337397760,
            394488000,
            354277440,
            404029440,
            405181440,
            369100800,
            300502080,
            331217280,
            221544000,
            405420480,
            400069440,
            405123840,
            404873280,
            364734720,
            262658880,
            342768960,
            229703040,
            405146880,
            404213760,
            403758720,
            398079360,
            405432000,
            403804800,
            405126720,
            396587520,
            396120960,
            405221760,
            347711040,
            394099200,
        ),
        1: (
            193930560,
            202158720,
            203051520,
            203238720,
            193861440,
            202037760,
            179784000,
            195557760,
            193671360,
            181425600,
            200658240,
            192412800,
            203244480,
            203244480,
            203244480,
            203207040,
            184104000,
            197179200,
            181428480,
            195912000,
            202656960,
            197994240,
            203241600,
            203235840,
            200583360,
            189679680,
            193628160,
            179853120,
            203238720,
            203160960,
            203181120,
            203247360,
            199247040,
            184878720,
            196551360,
            180898560,
            203230080,
            203247360,
            203235840,
            203060160,
            203247360,
            203238720,
            203155200,
            203065920,
            203100480,
            203238720,
            197409600,
            202803840,
        ),
        2: (
            101157120,
            101419200,
            101399040,
            101424960,
            101171520,
            101422080,
            99532800,
            101260800,
            101255040,
            99927360,
            101410560,
            101180160,
            101424960,
            101424960,
            101424960,
            101424960,
            100445760,
            101324160,
            99959040,
            101260800,
            101399040,
            101312640,
            101424960,
            101424960,
            101410560,
            100972800,
            101289600,
            99547200,
            101424960,
            101424960,
            101419200,
            101424960,
            101370240,
            100431360,
            101312640,
            99725760,
            101424960,
            101424960,
            101424960,
            101424960,
            101424960,
            101424960,
            101404800,
            101424960,
            101424960,
            101424960,
            101355840,
            101424960,
        ),
        3: (
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51024960,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51024960,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
            51027840,
        ),
        4: (
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
            25413120,
        ),
        5: (
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
            12810240,
        ),
        6: (
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
            6410880,
        ),
    },
    url_pattern="https://portal.nersc.gov/project/cosmo/temp/dstn/index-5200/LITE/index-52{scale}{index}.fits",
)

series_5200_heavy = Series(
    name="5200_heavy",
    description="HEAVY version same as series_5200, but with additional Gaia-DR2 information (magnitude in G, BP, RP, proper motions and parallaxes), handy if you want that extra Gaia information for matched stars",
    scale_to_sizes={
        0: (
            628182720,
            861978240,
            940265280,
            940288320,
            615424320,
            833783040,
            377830080,
            624107520,
            595036800,
            397177920,
            740514240,
            572990400,
            939381120,
            941172480,
            937359360,
            926343360,
            436101120,
            657648000,
            397221120,
            627638400,
            881055360,
            702855360,
            938413440,
            945904320,
            740482560,
            523745280,
            588116160,
            380021760,
            947384640,
            913409280,
            946405440,
            943801920,
            740171520,
            450014400,
            639025920,
            393868800,
            945789120,
            939591360,
            936691200,
            900529920,
            947419200,
            936993600,
            946448640,
            890959680,
            887976000,
            946131840,
            645143040,
            875747520,
        ),
        1: (
            421462080,
            467988480,
            474137280,
            474883200,
            420840000,
            467208000,
            335151360,
            425880000,
            419477760,
            343249920,
            458409600,
            411477120,
            474923520,
            474920640,
            474912000,
            474684480,
            359182080,
            436233600,
            343284480,
            428149440,
            471614400,
            441578880,
            474897600,
            474883200,
            457931520,
            394289280,
            419195520,
            335903040,
            474917760,
            474436800,
            474690240,
            474943680,
            449426880,
            364556160,
            432216000,
            341576640,
            474854400,
            474929280,
            474863040,
            473739840,
            474932160,
            474877440,
            474658560,
            473785920,
            473999040,
            474914880,
            437659200,
            472101120,
        ),
        2: (
            235238400,
            236908800,
            236822400,
            236952000,
            235327680,
            236923200,
            224879040,
            235912320,
            235877760,
            227393280,
            236862720,
            235402560,
            236952000,
            236952000,
            236952000,
            236946240,
            230690880,
            236318400,
            227583360,
            235903680,
            236816640,
            236237760,
            236952000,
            236949120,
            236868480,
            234066240,
            236085120,
            224953920,
            236949120,
            236940480,
            236908800,
            236952000,
            236606400,
            230616000,
            236232000,
            226105920,
            236946240,
            236952000,
            236952000,
            236952000,
            236952000,
            236952000,
            236882880,
            236952000,
            236952000,
            236949120,
            236511360,
            236949120,
        ),
        3: (
            119171520,
            119174400,
            119165760,
            119174400,
            119174400,
            119174400,
            119148480,
            119174400,
            119174400,
            119168640,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119168640,
            119174400,
            119162880,
            119171520,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119148480,
            119174400,
            119174400,
            119171520,
            119174400,
            119174400,
            119171520,
            119174400,
            119160000,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
            119162880,
            119174400,
            119174400,
            119174400,
            119174400,
            119174400,
        ),
        4: (
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
            59299200,
        ),
        5: (
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
            29851200,
        ),
        6: (
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
            14886720,
        ),
    },
    url_pattern="https://portal.nersc.gov/project/cosmo/temp/dstn/index-5200/index-52{scale}{index}.fits",
)

series_6000 = Series(
    name="6000",
    description="very specialized, uses GALEX Near-UV measurements, and only a narrow range of scales",
    scale_to_sizes={
        4: (
            64344960,
            100103040,
            99924480,
            49023360,
            101787840,
            41091840,
            101851200,
            58057920,
            100033920,
            45106560,
            40947840,
            90276480,
        ),
        5: (
            33347520,
            51082560,
            50924160,
            25499520,
            51897600,
            21340800,
            51848640,
            29931840,
            51010560,
            23376960,
            21133440,
            46264320,
        ),
        6: (
            17205120,
            25885440,
            25781760,
            13155840,
            26236800,
            11076480,
            26190720,
            15402240,
            25810560,
            12072960,
            10857600,
            23552640,
        ),
    },
    url_pattern="http://data.astrometry.net/6000/index-60{scale}{index}.fits",
)

series_6100 = Series(
    name="6100",
    description="very specialized, uses GALEX Far-UV measurements, and only a narrow range of scales",
    scale_to_sizes={
        4: (
            21119040,
            77250240,
            81109440,
            23204160,
            79211520,
            21502080,
            77898240,
            24462720,
            84444480,
            23414400,
            15445440,
            70266240,
        ),
        5: (
            18696960,
            46584000,
            47574720,
            18466560,
            47318400,
            15042240,
            47471040,
            20338560,
            48214080,
            18017280,
            14310720,
            42056640,
        ),
        6: (
            12792960,
            24978240,
            25125120,
            11321280,
            25312320,
            9158400,
            25424640,
            12795840,
            25191360,
            10797120,
            9262080,
            22633920,
        ),
    },
    url_pattern="http://data.astrometry.net/6100/index-61{scale}{index}.fits",
)
