import logging

import aiohttp

from .device import Device
from .data import Data
from .state import State
from .errors import raise_error, RequestError, UnsupportedError

Logger = logging.getLogger(__name__)

SUPPORTED_API_VERSION = "v1"

SUPPORTED_DEVICES = [
    "HWE-P1",
    "SDM230-wifi",
    "SDM630-wifi",
    "HWE-SKT"
]

class HomeWizardEnergy:
    """Communicate with a HomeWizard Energy device."""

    def __init__(self, host):
        Logger.debug("__init__ HomeWizardEnergy")
        self._host = host
        self._clientsession = self._get_clientsession()
        
        # Endpoints
        self.device = None
        self.data = None
        self.state = None

    async def __aenter__(self):
        return self

    async def __aexit__(self, exc_type, exc, tb):
        await self.close()

    def _get_clientsession(self):
        """
        Get a clientsession that is tuned for communication with the Energy device
        """

        connector = aiohttp.TCPConnector(
            enable_cleanup_closed=True, # Home Assistant sets it so lets do it also
            limit_per_host=1, # Device can handle a limited amount of connections, only take what we need
        )

        return aiohttp.ClientSession(connector=connector)

    async def initialize(self):
        self.device = Device(self.request)
        status = await self.device.update()
        if (status):
            
            # Validate 'device'
            if (self.device.api_version != SUPPORTED_API_VERSION):
                Logger.error("This library requires api to have version '%s'", SUPPORTED_API_VERSION)
                return
                
            if (self.device.product_type not in SUPPORTED_DEVICES):
                Logger.error("Device '%s' not supported", self.device.product_type)
                return
                
            # Get /data
            self.data = Data(self.request)
            status = await self.data.update()
            if not status:
                Logger.error("Failed to get 'data'")
            
            # For HWE-SKT: Get /State
            if (self.device.product_type == "HWE-SKT"):
                self.state = State(self.request)
                status = await self.state.update()
                if not status:
                    Logger.error("Failed to get 'state' data")

    async def update(self) -> bool:
        Logger.debug("hwenergy update")
        
        if (self.device is not None):
            status = await self.device.update()
            if not status:
                return False
            
        if (self.data is not None):
            status = await self.data.update()
            if not status:
                return False
            
        if (self.state is not None):
            status = await self.state.update()
            if not status:
                return False
                
        return True
        
    async def close(self):
        await self._clientsession.close()

    async def request(self, method, path, data=None):
        """Make a request to the API."""

        if self._clientsession.closed:
            # Avoid runtime errors when connection is closed.
            # This solves an issue when Updates were scheduled and HA was shutdown
            return None

        url = f'http://{self._host}/{path}'
        Logger.debug(f"URL: {url}")

        try:
            headers = {'Content-Type': 'application/json'}
            Logger.debug('%s, %s, %s' % (method, url, data))
            async with self._clientsession.request(method, url, json=data, headers=headers) as resp:
                Logger.debug('%s, %s' % (resp.status, await resp.text('utf-8')))

                data = None
                if resp.content_type == 'application/json':
                    data = await resp.json()
                                        
                return resp.status, data
        except aiohttp.client_exceptions.ClientError as err:
            raise RequestError(
                'Error requesting data from {}: {}'.format(self._host, err)
            ) from None


def _raise_on_error(data):
    """Check response for error message."""
    raise_error(data['error']['id'], data['error']["description"])
