"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com).
 *
 * @stability stable
 */
class HostedZone extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added.
         *
         * @stability stable
         */
        this.vpcs = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_route53_HostedZoneProps(props);
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.any({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    /**
     * ARN of this hosted zone, such as arn:${Partition}:route53:::hostedzone/${Id}.
     *
     * @stability stable
     */
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * Use when hosted zone ID is known. Hosted zone name becomes unavailable through this query.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param hostedZoneId the ID of the hosted zone to import.
     * @stability stable
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     *
     * Use when both hosted zone ID and hosted zone name are known.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param attrs the HostedZoneAttributes (hosted zone ID and hosted zone name).
     * @stability stable
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_route53_HostedZoneAttributes(attrs);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     *
     * Requires environment, you must specify env for the stack.
     *
     * Use to easily query hosted zones.
     *
     * @see https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     * @stability stable
     */
    static fromLookup(scope, id, query) {
        jsiiDeprecationWarnings._aws_cdk_aws_route53_HostedZoneProviderProps(query);
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     * @stability stable
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
_a = JSII_RTTI_SYMBOL_1;
HostedZone[_a] = { fqn: "@aws-cdk/aws-route53.HostedZone", version: "1.143.0" };
/**
 * Create a Route53 public hosted zone.
 *
 * @stability stable
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_PublicHostedZoneProps(props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
        if (!props.crossAccountZoneDelegationPrincipal && props.crossAccountZoneDelegationRoleName) {
            throw Error('crossAccountZoneDelegationRoleName property is not supported without crossAccountZoneDelegationPrincipal');
        }
        if (props.crossAccountZoneDelegationPrincipal) {
            this.crossAccountZoneDelegationRole = new iam.Role(this, 'CrossAccountZoneDelegationRole', {
                roleName: props.crossAccountZoneDelegationRoleName,
                assumedBy: props.crossAccountZoneDelegationPrincipal,
                inlinePolicies: {
                    delegation: new iam.PolicyDocument({
                        statements: [
                            new iam.PolicyStatement({
                                actions: ['route53:ChangeResourceRecordSets'],
                                resources: [this.hostedZoneArn],
                            }),
                            new iam.PolicyStatement({
                                actions: ['route53:ListHostedZonesByName'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
        }
    }
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param publicHostedZoneId the ID of the public hosted zone to import.
     * @stability stable
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @stability stable
     */
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts options for creating the DNS record, if any.
     * @stability stable
     */
    addDelegation(delegate, opts = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_route53_IPublicHostedZone(delegate);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_ZoneDelegationOptions(opts);
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
_b = JSII_RTTI_SYMBOL_1;
PublicHostedZone[_b] = { fqn: "@aws-cdk/aws-route53.PublicHostedZone", version: "1.143.0" };
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @stability stable
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_PrivateHostedZoneProps(props);
        this.addVpc(props.vpc);
    }
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param privateHostedZoneId the ID of the private hosted zone to import.
     * @stability stable
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
_c = JSII_RTTI_SYMBOL_1;
PrivateHostedZone[_c] = { fqn: "@aws-cdk/aws-route53.PrivateHostedZone", version: "1.143.0" };
//# sourceMappingURL=data:application/json;base64,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