# AutoCalendar :calendar:
**AutoCalendar** is a Python scheduling automation system based on the Google Calendar API ideal for **scheduling research participants**.

Running `autocalendar.py` adds events into Google Calendar by extracting the relevant information from an excel sheet containing participants' details, such as *date*, *time*, and *location*.

Check out [AutoRemind](https://github.com/zen-juen/AutoRemind) too!

## Code Structure

- **Schedule Participants**: `autoallocate()`
  - Parses a doodle poll spreadsheet and automatically allocates participants to an available slot (Useful if participants are told to choose as many available slots as they can)

- **Access Google API**: `setup_oauth()`
  - Sets up OAuth 2.0
  - You would need a `client_secret.json` file: you can do this by going to https://console.developers.google.com/apis/credentials. Click on the 'credentials' tab and then download the client secret file.
- **Extract event details and create event**:
  - `preprocess_file()`: preprocesses the input excel sheet (containing participants' scheduled slots)
  - `extract_info()`: extracts date, time and location based on header column names in the participants file. Argument `filter_column` can also be activated to select only some participants to be added into google calendar.
  - `create_event()`: prepares event details ready for execution. Argument `calendar_id` (defaults to 'primary') can also be modified according to which active google calendar to use by simply specifying the name of the calendar.
  - `add_event()`: executes adding of event into calendar.


## How to Use

### 1) Try it yourself: Automatic Scheduling of Participants

Download your doodle poll into an excel spreadsheet, like so:

![Screenshot1](images/pollxls.PNG)

```
import autocalendar as autocalendar

autocalendar.autoallocate('scheduled.xlsx')
All participants successfully allocated.
```
A 'scheduled.xlsx' file containing participants' final allocated slots is exported with one line of code. 
Feedback is provided on whether participants are successfully allocated or not, and the name of any participant who is not allocated will be printed.

### 2) Try it yourself: Adding Events

You will first need to start with a xlsx file containing information on participants' scheduled slots. For example:

![Screenshot](images/inputxlsx.PNG)

```
import autocalendar as autocalendar

# Set up Oauth to access Google API
service = autocalendar.setup_oath(token_path='../../token.pkl',
                                  client_path='../../client_secret.json')

# Read and tidy excel sheet
participants = autocalendar.preprocess_file('../../../Participants Scheduling/Master_Participant_List.xlsx', header_row=2)

# Extract info
dates, start_points, end_points, locations, to_add = autocalendar.extract_info(participants, date_col='Date_Session1', time_col='Timeslot_Session1', location_col='Location_Session1', filter_column='Calendar_Event', select='No')

# Execute
autocalendar.add_event(service, dates, start_points, end_points, locations, to_add,
                       event_name='MRI study Session 1', description='', timezone='Asia/Singapore',
                       creator_email='mristudy@gmail.com', calendar_id='NTU Calendar',
                       silent=False, name_col='Participant Name', date_col='Date_Session1',
                       location_col='Location_Session1', time_col='Timeslot_Session1')

```

If `silent` in `add_event()` is set to False, you will also get **printed output** on each participant's appointment that is added into Google Calendar!
```
Adding calendar event for Subject1 at 11-8-2020, 11.00am-11.30am, B1-26 
Adding calendar event for Subject2 at 11-8-2020, 10.30am-11.00am, B1-26 
Adding calendar event for Subject3 at 11-8-2020, 4.00pm-4.30pm, B1-26 
Adding calendar event for Subject4 at 13-8-2020, 9.30am-10.00am, B1-26 
Adding calendar event for Subject5 at 12-8-2020, 11.30am-12.00pm, B1-26 
Adding calendar event for Subject6 at 12-8-2020, 12.00pm-12.30pm, B1-26 

```




## Future Direction
- [x] Printed feedback on each event that has been successfully added into calendar
- [ ] Accessibility for longitudinal studies
- [ ] Greater functionality for customizing events by adding arguments from https://developers.google.com/calendar/v3/reference/events
