# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['e2fyi', 'e2fyi.utils', 'e2fyi.utils.aws', 'e2fyi.utils.core']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.9',
 'joblib',
 'pydantic>=0.30',
 'python-dateutil>=2.1,<2.8.1',
 'toml>=0.10.1,<0.11.0',
 'typing-extensions']

extras_require = \
{':sys_platform == "darwin" or sys_platform == "windows"': ['python-magic-bin>=0.4.0,<0.5.0'],
 ':sys_platform == "linux"': ['python-magic>=0.4.0,<0.5.0'],
 'all': ['pandas'],
 'pandas': ['pandas']}

setup_kwargs = {
    'name': 'e2fyi-utils',
    'version': '0.3.0rc1',
    'description': 'General utils for interacting with aws resources.',
    'long_description': '# e2fyi-utils\n\n[![PyPI version](https://badge.fury.io/py/e2fyi-utils.svg)](https://badge.fury.io/py/e2fyi-utils)\n[![Build Status](https://travis-ci.com/e2fyi/py-utils.svg?branch=master)](https://travis-ci.com/github/e2fyi/py-utils)\n[![Coverage Status](https://coveralls.io/repos/github/e2fyi/py-utils/badge.svg?branch=master)](https://coveralls.io/github/e2fyi/py-utils?branch=master)\n[![Documentation Status](https://readthedocs.org/projects/e2fyi-utils/badge/?version=latest)](https://e2fyi-utils.readthedocs.io/en/latest/?badge=latest)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Downloads](https://pepy.tech/badge/e2fyi-utils/month)](https://pepy.tech/project/e2fyi-utils/month)\n\n`e2fyi-utils` is an `e2fyi` namespaced python package with `utils` subpackage\n(i.e. `e2fyi.utils`) which holds a collections of useful helper classes and\nfunctions to interact with various cloud resources.\n\nAPI documentation can be found at [https://e2fyi-utils.readthedocs.io/en/latest/](https://e2fyi-utils.readthedocs.io/en/latest/).\n\nChange logs are available in [CHANGELOG.md](./CHANGELOG.md).\n\n> - Python 3.6 and above\n> - Licensed under [Apache-2.0](./LICENSE).\n\n\n## Quickstart\n\n```bash\n# install the default packages only (e.g. no pandas)\npip install e2fyi-utils[all]\n\n# install all optional packages\npip install e2fyi-utils[all]\n\n# install specific optional packages\npip install e2fyi-utils[pandas]\n```\n\nAvailable optional packages:\n\n- `pandas`\n\n### S3Stream\n\n`S3Stream` represents the data stream of a S3 resource, and provides static\nmethods to convert any python objects into a stream. This is generally used with\n`S3Resource` to upload or download resource from S3 buckets.\n\n> NOTE:\n> - `str`, `float`, `int`, and `bool` will be saved as txt files with mime type "text/plain".\n> - `pydantic` models, `dict` or `list` will be saved as json files with mime type "application/json" (fallback to pickle if unable to serialize into json string).\n> - `pandas` dataframe or series can be saved as either a csv ("application/csv") or json format ("application/json").\n> - path to files will be read with `open` and mime type will be inferred (fallback to "application/octet-stream").\n> - all other python objects will be pickled with `joblib`.\n\n```py\nimport io\n\nimport pandas as pd\n\nfrom e2fyi.utils.aws import S3Stream\nfrom pydantic import BaseModel\n\n# create a s3 stream\nstream = S3Stream(io.StringIO("random text"), "text/plain")\nprint(stream.read())        # prints "random text"\nprint(stream.content_type)  # prints "text/plain"\n\n# string\nstream = S3Stream.from_any("hello world")\nprint(stream.read())        # prints "hello world"\nprint(stream.content_type)  # prints "text/plain"\n\n# dict\nstream = S3Stream.from_any({"foo": "bar"})\nprint(stream.read())        # prints "{"foo": "bar"}"\nprint(stream.content_type)  # prints "application/json"\n\n# pandas dataframe as csv\ndf = pd.DataFrame([{"key": "a", "value": 1}, {"key": "b", "value": 2}])\nstream = S3Stream.from_any(df, index=False)  # do not include index column\nprint(stream.read())        # prints string as csv format for the dataframe\nprint(stream.content_type)  # prints "application/csv"\n\n# pandas dataframe as json\nstream = S3Stream.from_any(df, orient="records")  # orient dataframe as records\nprint(stream.read())        # prints string as json list for the dataframe\nprint(stream.content_type)  # prints "application/json"\n\n\n# pydantic model\nclass Person(BaseModel):\n    name: str\n    age: int\nstream = S3Stream.from_any(Person(name="william", age=21))\nprint(stream.read())        # prints "{"name": "william", "age": 21}"\nprint(stream.content_type)  # prints "application/json"\n\n\n# any other python objects\nclass Pet:\n    name: str\n    age: int\nstream = S3Stream.from_any(Pet(name="kopi", age=1))\nprint(stream.read())        # prints some binary bytes\nprint(stream.content_type)  # prints "application/octet-stream"\n\n```\n\n### S3Resource\n\n`S3Resource` represents a resource in S3 currently or a local resource that will\nbe uploaded to S3. `S3Resource` constructor will automatically attempts to convert\nany inputs into a `S3Stream`, but for more granular control `S3Stream.from_any`\nshould be used instead to create the `S3Stream`.\n\n`S3Resource` is a readable stream - i.e. it has `read`, `seek`, and `close`.\n\n> NOTE:\n>\n> See https://boto3.amazonaws.com/v1/documentation/api/latest/reference/customizations/s3.html\n> for additional keyword arguments that can be passed to S3Resource.\n\n#### Example: Creating S3Resource from local python object or file.\n```py\nimport boto3\n\nfrom e2fyi.utils.aws import S3Resource, S3Stream\n\n# create custom s3 client\ns3client = boto3.client(\n    \'s3\',\n    aws_access_key_id=ACCESS_KEY,\n    aws_secret_access_key=SECRET_KEY\n)\n\n# creates a local copy of s3 resource with S3Stream from a local file\nobj = S3Resource(\n    # full path shld be "prefix/some_file.json"\n    filename="some_file.json",\n    prefix="prefix/",\n    # bucket to download from or upload to\n    bucketname="some_bucket",\n    # or "s3n://" or "s3://"\n    protocol="s3a://",\n    # uses default client if not provided\n    s3client=s3client,\n    # attempts to convert to S3Stream if input is not a S3Stream\n    stream=S3Stream.from_file("./some_path/some_file.json"),\n    # addition kwarg to pass to `s3.upload_fileobj` or `s3.download_fileobj` methods\n    Metadata={"label": "foo"}\n)\nprint(obj.key)  # prints "prefix/some_file.json"\nprint(obj.uri)  # prints "s3a://some_bucket/prefix/some_file.json"\n\n# will attempt to fix prefix and filename if incorrect filename is provided\nobj = S3Resource(\n    filename="subfolder/some_file.json",\n    prefix="prefix/"\n)\nprint(obj.filename)     # prints "some_file.json"\nprint(obj.prefix)       # prints "prefix/subfolder/"\n```\n\n#### Example: Upload S3Resource to S3\n```py\n# creates a local copy of s3 resource with some python object\nobj = S3Resource(\n    filename="some_file.txt",\n    prefix="prefix/",\n    bucketname="some_bucket",\n    stream={"some": "dict"},\n)\n\n# upload obj to s3 bucket "some_bucket" with the key "prefix/some_file.json"\n# with the json string content.\nobj.save()\n\n# upload to s3 bucket "another_bucket" instead with a metadata tag.\nobj.save("another_bucket", MetaData={"label": "foo"})\n```\n\n#### Example: Read S3Resource from S3\n```py\nfrom pydantic import BaseModel\n\n# do not provide a stream input to the S3Resource constructor\nobj = S3Resource(\n    filename="some_file.json",\n    prefix="prefix/",\n    bucketname="some_bucket",\n    content_type="application/json"\n)\n\n# read the resource like a normal file object from S3\ndata = obj.read()\nprint(type(data))       # prints <class \'str\'>\n\n# read and load json string into a dict or list\n# for content_type == "application/json" only\ndata_obj = obj.load()\nprint(type(data_obj))   # prints <class \'dict\'> or <class \'list\'>\n\n\n# read and convert into a pydantic model\nclass Person(BaseModel):\n    name: str\n    age: int\n\n# automatically unpack the dict\ndata_obj = obj.load(lambda name, age: Person(name=name, age=age))\n# alternatively, do not unpack\ndata_obj = obj.load(lambda data: Person(**data), unpack=False)\nprint(type(data_obj))   # prints <class \'Person\'>\n```\n\n### S3Bucket\n\n`S3Bucket` is an abstraction of the actual S3 bucket with methods to interact\nwith the actual S3 bucket (e.g. list objects inside the bucket), and some utility\nmethods.\n\nPrefix rules can also be set during the creation of the `S3Bucket` object - i.e.\nenforce a particular prefix rules for a particular bucket.\n\n#### Quickstart\n```py\nfrom e2fyi.utils.aws import S3Bucket\n\n# prints key for all resources with prefix "some_folder/"\nfor resource in S3Bucket("some_bucket").list("some_folder/"):\n    print(resource.key)\n\n# prints key for the first 2,000 resources with prefix "some_folder/"\nfor resource in S3Bucket("some_bucket").list("some_folder/", max_objects=2000):\n    print(resource.key)\n\n# creates a s3 bucket with prefix rule\nprj_bucket = S3Bucket("some_bucket", get_prefix=lambda prefix: "prj-a/%s" % prefix)\nfor resource in prj_bucket.list("some_folder/"):\n    print(resource.key)  # prints "prj-a/some_folder/<resource_name>"\n    print(resource.stats)  # prints metadata for the resource (e.g. size, etag)\n\n# get obj key in the bucket\nprint(prj_bucket.create_resource_key("foo.json"))  # prints "prj-a/foo.json"\n\n# get obj uri in the bucket\n# prints "s3a://some_bucket/prj-a/foo.json"\nprint(prj_bucket.create_resource_uri("foo.json", "s3a://"))\n\n# create S3Resource in bucket to read in\nfoo = prj_bucket.create_resource("foo.json", "application/json")\n# read "s3a://some_bucket/prj-a/foo.json" and load as a dict (or list)\nfoo_dict = foo.load()\n\n# create S3Resource in bucket and save to "s3a://some_bucket/prj-a/foo.json"\nprj_bucket.create_resource("foo.json", obj={"foo": "bar"}).save()\n```\n\n### e2fyi.utils.core.Maybe\n\n`Maybe` represents an uncertain object (exception might be raised so no value\nwill be returned). This is generally used inside a function where all exceptions\nwill be caught.\n\n> NOTE:\n> - `Maybe.value` is the actual returned value.\n> - `Maybe.exception` is the exception caught (if any).\n> - `Maybe.with_default` method can be used to provide a default value if no value\nis returned.\n> - `Maybe.is_ok` method can be used to check if any value is returned.\n\n```py\nimport logging\n\nfrom e2fyi.utils.core import Maybe\n\n\ndef load_from_file(filepath: str) -> Maybe[string]:\n    """loads the content of a file."""\n    try:\n        with open(filepath, "r") as fp:\n            return Maybe(fp.read())\n    except IOError as err:\n        return Maybe(exception=err)\n\ndata = load_from_file("some_file.json")\n\n# print with a default value fallback\nprint(data.with_default("default value"))\n\n# print data if ok, else log exception\nif data.is_ok:\n    print(data)\nelse:\n    logging.exception(data.exception)\n\n```\n',
    'author': 'eterna2',
    'author_email': 'eterna2@hotmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.1,<4',
}


setup(**setup_kwargs)
