# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['judoscale',
 'judoscale.celery',
 'judoscale.core',
 'judoscale.django',
 'judoscale.flask',
 'judoscale.rq']

package_data = \
{'': ['*']}

install_requires = \
['requests<3.0.0']

extras_require = \
{'celery-redis': ['celery[redis]>=4.4.0,<6.0.0'],
 'django': ['django>=2.1.0,<5.0.0'],
 'flask': ['flask>=1.1.0,<3.0.0'],
 'rq': ['rq>=1.0.0,<2.0.0']}

setup_kwargs = {
    'name': 'judoscale',
    'version': '1.1.0',
    'description': 'Official Python adapter for Judoscale — the advanced autoscaler for Heroku',
    'long_description': '# Judoscale\n\nThis is the official Python adapter for [Judoscale](https://elements.heroku.com/addons/judoscale). You can use Judoscale without it, but this gives you request queue time metrics and job queue time (for supported job processors).\n\nIt is recommended to install the specific web framework and/or background job library support as "extras" to the `judoscale` PyPI package. This ensures that checking if the installed web framework and/or background task processing library is supported happens at dependency resolution time.\n\n## Supported web frameworks\n\n- [x] [Django](#using-judoscale-with-django)\n- [x] [Flask](#using-judoscale-with-flask)\n- [ ] FastAPI\n\n## Supported job processors\n\n- [x] [Celery](#using-judoscale-with-celery-and-redis) (with Redis as the broker)\n- [x] [RQ](#using-judoscale-with-rq)\n\n# Using Judoscale with Django\n\nInstall Judoscale for Django with:\n\n```sh\n$ pip install \'judoscale[django]\'\n```\n\nAdd Judoscale app to `settings.py`:\n\n```python\nINSTALLED_APPS = [\n    "judoscale.django",\n    # ... other apps\n]\n```\n\nThis sets up the Judoscale middleware to capture request queue times.\n\nOptionally, you can customize Judoscale in `settings.py`:\n\n```python\nJUDOSCALE = {\n    # LOG_LEVEL defaults to ENV["LOG_LEVEL"] or "INFO".\n    "LOG_LEVEL": "DEBUG",\n\n    # API_BASE_URL defaults to ENV["JUDOSCALE_URL"], which is set for you when you install Judoscale.\n    # This is only exposed for testing purposes.\n    "API_BASE_URL": "https://example.com",\n\n    # REPORT_INTERVAL_SECONDS defaults to 10 seconds.\n    "REPORT_INTERVAL_SECONDS": 5,\n}\n```\n\nOnce deployed, you will see your request queue time metrics available in the Judoscale UI.\n\n# Using Judoscale with Flask\n\nInstall Judoscale for Flask with:\n\n```sh\n$ pip install \'judoscale[flask]\'\n```\n\nThe Flask support for Judoscale is packaged into a Flask extension. Import the extension class and use like you normally would in a Flask application:\n\n```py\n# app.py\n\nfrom judoscale.flask import Judoscale\n\n# If your app is a top-level global\n\napp = Flask("MyFlaskApp")\napp.config.from_object(\'...\')  # or however you configure your app\njudoscale = Judoscale(app)\n\n\n# If your app uses the application factory pattern\n\njudoscale = Judoscale()\n\ndef create_app():\n    app = Flask("MyFlaskApp")\n    app.config.from_object(\'...\')  # or however you configure your app\n    judoscale.init_app(app)\n    return app\n```\n\nThis sets up the Judoscale extension to capture request queue times.\n\nOptionally, you can override Judoscale\'s own configuration via your application\'s [configuration dictionary](https://flask.palletsprojects.com/en/2.2.x/api/#flask.Flask.config). The Judoscale Flask extension looks for a top-level `"JUDOSCALE"` key in `app.config`, which should be a dictionary, and which the extension uses to configure itself as soon as `judoscale.init_app()` is called.\n\n```python\nJUDOSCALE = {\n    # LOG_LEVEL defaults to ENV["LOG_LEVEL"] or "INFO".\n    "LOG_LEVEL": "DEBUG",\n\n    # API_BASE_URL defaults to ENV["JUDOSCALE_URL"], which is set for you when you install Judoscale.\n    # This is only exposed for testing purposes.\n    "API_BASE_URL": "https://example.com",\n\n    # REPORT_INTERVAL_SECONDS defaults to 10 seconds.\n    "REPORT_INTERVAL_SECONDS": 5,\n}\n```\n\nNote the [official recommendations for configuring Flask](https://flask.palletsprojects.com/en/2.2.x/config/#configuration-best-practices).\n\n# Using Judoscale with Celery and Redis\n\nInstall Judoscale for Celery with:\n\n```sh\n$ pip install \'judoscale[celery-redis]\'\n```\n\n> :warning: **NOTE 1:** The Judoscale Celery integration currently only works with the [Redis broker](https://docs.celeryq.dev/en/stable/getting-started/backends-and-brokers/index.html#redis).\n\n> :warning: **NOTE 2:** Using [task priorities](https://docs.celeryq.dev/en/latest/userguide/calling.html#advanced-options) is currently not supported by `judoscale`. You can still use task priorities, but `judoscale` won\'t see and report metrics on any queues other than the default, unprioritised queue.\n\nJudoscale can automatically scale the number of Celery workers based on the queue latency (the age of the oldest pending task in the queue).\n\n## Setting up the integration\n\nTo use the Celery integration, import `judoscale_celery` and call it with the Celery app instance. `judoscale_celery` should be called after you have set up and configured the Celery instance.\n\n```py\nfrom celery import Celery\nfrom judoscale.celery import judoscale_celery\n\ncelery_app = Celery(broker="redis://localhost:6379/0")\n# Further setup\n# celery_app.conf.update(...)\n# ...\n\njudoscale_celery(celery_app)\n```\n\nThis sets up Judoscale to periodically calculate and report queue latency for each Celery queue.\n\nIf you need to change the Judoscale integration configuration, you can pass a dictionary of Judoscale configuration options to `judoscale_celery` to override the default Judoscale config variables:\n\n```py\njudoscale_celery(celery_app, extra_config={"LOG_LEVEL": "DEBUG"})\n```\n\n> :warning: **NOTE:** Calling `judoscale_celery` turns on sending [`task-sent`](https://docs.celeryq.dev/en/stable/userguide/configuration.html#task-send-sent-event) events. This is required for the Celery integration with Judoscale to work.\n\n### Judoscale with Celery and Flask\n\nDepending on how you\'ve structured your Flask app, you should call `judoscale_celery` after your application has finished configuring the Celery app instance. If you have followed the [Flask guide](https://flask.palletsprojects.com/en/2.2.x/patterns/celery/) in the Flask documentation, the easiest place to initialise the Judoscale integration is in the application factory:\n\n```py\ndef create_app():\n    app = Flask(__name__)\n    app.config.from_object(...) # or however you configure your app\n    celery_app = celery_init_app(app)\n    # Initialise the Judoscale integration\n    judoscale_celery(celery_app, extra_config=app.config["JUDOSCALE"])\n    return app\n```\n\n### Judoscale with Celery and Django\n\nIf you have followed the [Django guide](https://docs.celeryq.dev/en/stable/django/first-steps-with-django.html) in the Celery documentation, you should have a module where you initialise the Celery app instance, auto-discover tasks, etc. You should call `judoscale_celery` after you have configured the Celery app instance:\n\n```py\nfrom celery import Celery\nfrom django.conf import settings\nfrom judoscale.celery import judoscale_celery\n\napp = Celery()\napp.config_from_object("django.conf:settings", namespace="CELERY")\napp.autodiscover_tasks()\n# Initialise the Judoscale integration\njudoscale_celery(app, extra_config=settings.JUDOSCALE)\n```\n\n# Using Judoscale with RQ\n\nInstall Judoscale for RQ with:\n\n```sh\n$ pip install \'judoscale[rq]\'\n```\n\nJudoscale can automatically scale the number of RQ workers based on the queue latency (the age of the oldest pending task in the queue).\n\n## Setting up the integration\n\nTo use the RQ integration, import `judoscale_rq` and call it with an instance of `Redis` pointing to the same Redis database that RQ uses.\n\n```py\nfrom redis import Redis\nfrom judoscale.rq import judoscale_rq\n\nredis = Redis(...)\njudoscale_rq(redis)\n```\n\nThis sets up Judoscale to periodically calculate and report queue latency for each RQ queue.\n\nIf you need to change the Judoscale integration configuration, you can pass a dictionary of Judoscale configuration options to `judoscale_rq` to override the default Judoscale config variables:\n\n```py\njudoscale_rq(redis, extra_config={"LOG_LEVEL": "DEBUG"})\n```\n\n### Judoscale with RQ and Flask\n\nThe recommended way to initialise Judoscale for RQ is in the application factory:\n\n```py\njudoscale = Judoscale()\n\ndef create_app():\n    app = Flask(__name__)\n    app.config.from_object("...") # or however you configure your application\n    app.redis = Redis()\n\n    # Initialise the Judoscale integration for Flask\n    judoscale.init_app(app)\n\n    # Initialise the Judoscale integration for RQ\n    judoscale_rq(app.redis)\n\n    return app\n```\n\nThen, in your worker script, make sure that you create an app, which will initialise the Judoscale integration with RQ. Although not required, it\'s useful to run the worker within the Flask app context. If you have followed the [RQ on Heroku pattern](https://python-rq.org/patterns/) for setting up your RQ workers on Heroku, your worker script should look something like this:\n\n```py\nfrom rq.worker import HerokuWorker as Worker\n\napp = create_app()\n\nworker = Worker(..., connection=app.redis)\nwith app.app_context():\n    worker.work()\n```\n\nSee the [run-worker.py script](./sample-apps/flask_rq_sample/run-worker.py) for reference.\n\n### Judoscale with RQ and Django\n\nThe Judoscale integration for RQ is packaged into a separate Django app.\n\nYou should already have `judoscale.django` in your `INSTALLED_APPS`. Next, add the RQ integration app `judoscale.rq`:\n\n```python\nINSTALLED_APPS = [\n    "judoscale.django",\n    "judoscale.rq",\n    # ... other apps\n]\n```\n\nBy default, `judoscale.rq` will connect to the Redis instance as specified by the `REDIS_URL` environment variable. If that is not suitable, you can supply Redis connection information in the `JUDOSCALE` configuration dictionary under the `"REDIS"` key.\n\nAccepted formats are:\n\n- a dictionary containing a single key `"URL"` pointing to a Redis server URL, or;\n- a dictionary of configuration options corresponding to the keyword arguments of the [`Redis` constructor](https://github.com/redis/redis-py/blob/6c708c2e0511364c2c3f21fa1259de05e590632d/redis/client.py#L905).\n\n```py\nJUDOSCALE = {\n    # Configuring with a Redis server URL\n    "REDIS": {\n        "URL": os.getenv("REDISTOGO_URL")\n    }\n\n    # Configuring as kwargs to Redis(...)\n    "REDIS": {\n        "HOST": "localhost",\n        "PORT": 6379,\n        "DB": 0\n    }\n}\n```\n\nIf you are using [Django-RQ](https://github.com/rq/django-rq/tree/master), you can also pull configuration from `RQ_QUEUES` directly:\n\n```py\nRQ_QUEUES = {\n    "high_priority": {\n        "HOST": "...",\n        "PORT": 6379,\n        "DB": 0\n    },\n}\n\nJUDOSCALE = {\n    # ... other configuration options\n    "REDIS": RQ_QUEUES["high_priority"]\n}\n```\n\n> :warning: **NOTE:** Django-RQ enables configuring RQ such that different queues and workers use _different_ Redis instances. Judoscale currently only supports connecting to and monitoring queue latency on a single Redis instance.\n\n## Development\n\nThis repo includes a `sample-apps` directory containing apps you can run locally. These apps use the `judoscale` adapter, but they override `API_BASE_URL` so they\'re not connected to the real Judoscale API. Instead, they post API requests to https://requestinspector.com so you can observe the API behavior.\n\nSee the `README` in a sample app for details on how to set it up and run locally.\n\n### Contributing\n\n`judoscale` uses [Poetry](https://python-poetry.org/) for managing dependencies and packaging the project. Head over to the [installations instructions](https://python-poetry.org/docs/#installing-with-the-official-installer) and install Poetry, if needed.\n\nClone the repo with\n\n```sh\n$ git clone git@github.com:judoscale/judoscale-python.git\n$ cd judoscale-python\n```\n\nVerify that you are on a recent version of Poetry:\n\n```sh\n$ poetry --version\nPoetry (version 1.3.1)\n```\n\nInstall dependencies with Poetry and activate the virtualenv\n\n```sh\n$ poetry install --all-extras\n$ poetry shell\n```\n\nRun tests with\n\n```sh\n$ python -m unittest discover -s tests\n```\n',
    'author': 'Adam McCrea',
    'author_email': 'adam@adamlogic.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/judoscale/judoscale-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
