"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../../lib");
let stack;
let clusterRole;
let serviceRole;
let autoScalingRole;
beforeEach(() => {
    var _a;
    // GIVEN
    stack = new cdk.Stack();
    clusterRole = new iam.Role(stack, 'ClusterRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
    });
    serviceRole = new iam.Role(stack, 'ServiceRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
    });
    autoScalingRole = new iam.Role(stack, 'AutoScalingRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
    });
    (_a = autoScalingRole.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        principals: [
            new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        ],
        actions: [
            'sts:AssumeRole',
        ],
    }));
});
test('Create Cluster with FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with SYNC integrationPattern', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with clusterConfiguration Name from payload', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: sfn.TaskInput.fromDataAt('$.ClusterName').value,
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            'Name.$': '$.ClusterName',
            'Instances': {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            'VisibleToAllUsers': true,
            'JobFlowRole': {
                Ref: 'ClusterRoleD9CA7471',
            },
            'ServiceRole': {
                Ref: 'ServiceRole4288B192',
            },
            'AutoScalingRole': {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with Tags', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        tags: {
            key: 'value',
        },
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Tags: [{
                    Key: 'key',
                    Value: 'value',
                }],
        },
    });
});
test('Create Cluster with Applications', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        applications: [
            { name: 'Hive', version: '0.0' },
            { name: 'Spark', version: '0.0' },
        ],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Applications: [
                { Name: 'Hive', Version: '0.0' },
                { Name: 'Spark', Version: '0.0' },
            ],
        },
    });
});
test('Create Cluster with Bootstrap Actions', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        bootstrapActions: [{
                name: 'Bootstrap',
                scriptBootstrapAction: {
                    path: 's3://null',
                    args: ['Arg'],
                },
            }],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            BootstrapActions: [{
                    Name: 'Bootstrap',
                    ScriptBootstrapAction: {
                        Path: 's3://null',
                        Args: ['Arg'],
                    },
                }],
        },
    });
});
test('Create Cluster with Configurations', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        configurations: [{
                classification: 'classification',
                properties: {
                    Key: 'Value',
                },
            }],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Configurations: [{
                    Classification: 'classification',
                    Properties: {
                        Key: 'Value',
                    },
                }],
        },
    });
});
test('Create Cluster with KerberosAttributes', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        kerberosAttributes: {
            realm: 'realm',
            adDomainJoinPassword: 'password1',
            adDomainJoinUser: 'user',
            crossRealmTrustPrincipalPassword: 'password2',
            kdcAdminPassword: 'password3',
        },
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            KerberosAttributes: {
                Realm: 'realm',
                ADDomainJoinPassword: 'password1',
                ADDomainJoinUser: 'user',
                CrossRealmTrustPrincipalPassword: 'password2',
                KdcAdminPassword: 'password3',
            },
        },
    });
});
test('Create Cluster without Roles', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        name: 'Cluster',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'TaskInstanceRoleB72072BF',
            },
            ServiceRole: {
                Ref: 'TaskServiceRoleBF55F61E',
            },
            AutoScalingRole: {
                Ref: 'TaskAutoScalingRoleD06F8423',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
    // The stack renders the ec2.amazonaws.com Service principal id with a
    // Join to the URLSuffix
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: {
                        Service: {
                            'Fn::Join': [
                                '',
                                [
                                    'ec2.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                ],
                            ],
                        },
                    },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
                {
                    Principal: { Service: 'application-autoscaling.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
});
test('Create Cluster with Instances configuration', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            additionalMasterSecurityGroups: ['MasterGroup'],
            additionalSlaveSecurityGroups: ['SlaveGroup'],
            ec2KeyName: 'Ec2KeyName',
            ec2SubnetId: 'Ec2SubnetId',
            ec2SubnetIds: ['Ec2SubnetId'],
            emrManagedMasterSecurityGroup: 'MasterGroup',
            emrManagedSlaveSecurityGroup: 'SlaveGroup',
            hadoopVersion: 'HadoopVersion',
            instanceCount: 1,
            masterInstanceType: 'MasterInstanceType',
            placement: {
                availabilityZone: 'AvailabilityZone',
                availabilityZones: ['AvailabilityZone'],
            },
            serviceAccessSecurityGroup: 'ServiceAccessGroup',
            slaveInstanceType: 'SlaveInstanceType',
            terminationProtected: true,
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                AdditionalMasterSecurityGroups: ['MasterGroup'],
                AdditionalSlaveSecurityGroups: ['SlaveGroup'],
                Ec2KeyName: 'Ec2KeyName',
                Ec2SubnetId: 'Ec2SubnetId',
                Ec2SubnetIds: ['Ec2SubnetId'],
                EmrManagedMasterSecurityGroup: 'MasterGroup',
                EmrManagedSlaveSecurityGroup: 'SlaveGroup',
                HadoopVersion: 'HadoopVersion',
                InstanceCount: 1,
                KeepJobFlowAliveWhenNoSteps: true,
                MasterInstanceType: 'MasterInstanceType',
                Placement: {
                    AvailabilityZone: 'AvailabilityZone',
                    AvailabilityZones: ['AvailabilityZone'],
                },
                ServiceAccessSecurityGroup: 'ServiceAccessGroup',
                SlaveInstanceType: 'SlaveInstanceType',
                TerminationProtected: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with InstanceFleet', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            instanceFleets: [{
                    instanceFleetType: lib_1.EmrCreateCluster.InstanceRoleType.MASTER,
                    instanceTypeConfigs: [{
                            bidPrice: '1',
                            bidPriceAsPercentageOfOnDemandPrice: 1,
                            configurations: [{
                                    classification: 'Classification',
                                    properties: {
                                        Key: 'Value',
                                    },
                                }],
                            ebsConfiguration: {
                                ebsBlockDeviceConfigs: [{
                                        volumeSpecification: {
                                            iops: 1,
                                            volumeSize: cdk.Size.gibibytes(1),
                                            volumeType: lib_1.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                        },
                                        volumesPerInstance: 1,
                                    }],
                                ebsOptimized: true,
                            },
                            instanceType: 'm5.xlarge',
                            weightedCapacity: 1,
                        }],
                    launchSpecifications: {
                        spotSpecification: {
                            blockDurationMinutes: 1,
                            timeoutAction: lib_1.EmrCreateCluster.SpotTimeoutAction.TERMINATE_CLUSTER,
                            timeoutDurationMinutes: 1,
                        },
                    },
                    name: 'Master',
                    targetOnDemandCapacity: 1,
                    targetSpotCapacity: 1,
                }],
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceFleets: [{
                        InstanceFleetType: 'MASTER',
                        InstanceTypeConfigs: [{
                                BidPrice: '1',
                                BidPriceAsPercentageOfOnDemandPrice: 1,
                                Configurations: [{
                                        Classification: 'Classification',
                                        Properties: {
                                            Key: 'Value',
                                        },
                                    }],
                                EbsConfiguration: {
                                    EbsBlockDeviceConfigs: [{
                                            VolumeSpecification: {
                                                Iops: 1,
                                                SizeInGB: 1,
                                                VolumeType: 'standard',
                                            },
                                            VolumesPerInstance: 1,
                                        }],
                                    EbsOptimized: true,
                                },
                                InstanceType: 'm5.xlarge',
                                WeightedCapacity: 1,
                            }],
                        LaunchSpecifications: {
                            SpotSpecification: {
                                BlockDurationMinutes: 1,
                                TimeoutAction: 'TERMINATE_CLUSTER',
                                TimeoutDurationMinutes: 1,
                            },
                        },
                        Name: 'Master',
                        TargetOnDemandCapacity: 1,
                        TargetSpotCapacity: 1,
                    }],
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
        },
    });
});
test('Create Cluster with InstanceGroup', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            instanceGroups: [{
                    autoScalingPolicy: {
                        constraints: {
                            maxCapacity: 2,
                            minCapacity: 1,
                        },
                        rules: [{
                                action: {
                                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                    simpleScalingPolicyConfiguration: {
                                        adjustmentType: lib_1.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                        coolDown: 1,
                                        scalingAdjustment: 1,
                                    },
                                },
                                description: 'Description',
                                name: 'Name',
                                trigger: {
                                    cloudWatchAlarmDefinition: {
                                        comparisonOperator: lib_1.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                        dimensions: [{
                                                key: 'Key',
                                                value: 'Value',
                                            }],
                                        evaluationPeriods: 1,
                                        metricName: 'Name',
                                        namespace: 'Namespace',
                                        period: cdk.Duration.seconds(300),
                                        statistic: lib_1.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                        threshold: 1,
                                        unit: lib_1.EmrCreateCluster.CloudWatchAlarmUnit.NONE,
                                    },
                                },
                            }, {
                                action: {
                                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                    simpleScalingPolicyConfiguration: {
                                        adjustmentType: lib_1.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                        coolDown: 1,
                                        scalingAdjustment: 1,
                                    },
                                },
                                description: 'Description',
                                name: 'Name',
                                trigger: {
                                    cloudWatchAlarmDefinition: {
                                        comparisonOperator: lib_1.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                        dimensions: [{
                                                key: 'Key',
                                                value: 'Value',
                                            }],
                                        evaluationPeriods: 1,
                                        metricName: 'Name',
                                        namespace: 'Namespace',
                                        period: cdk.Duration.seconds(sfn.JsonPath.numberAt('$.CloudWatchPeriod')),
                                        statistic: lib_1.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                        threshold: 1,
                                        unit: lib_1.EmrCreateCluster.CloudWatchAlarmUnit.NONE,
                                    },
                                },
                            }],
                    },
                    bidPrice: '1',
                    configurations: [{
                            classification: 'Classification',
                            properties: {
                                Key: 'Value',
                            },
                        }],
                    ebsConfiguration: {
                        ebsBlockDeviceConfigs: [{
                                volumeSpecification: {
                                    iops: 1,
                                    volumeSize: cdk.Size.gibibytes(1),
                                    volumeType: lib_1.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                },
                                volumesPerInstance: 1,
                            }],
                        ebsOptimized: true,
                    },
                    instanceCount: 1,
                    instanceRole: lib_1.EmrCreateCluster.InstanceRoleType.MASTER,
                    instanceType: 'm5.xlarge',
                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                    name: 'Name',
                }],
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceGroups: [{
                        AutoScalingPolicy: {
                            Constraints: {
                                MaxCapacity: 2,
                                MinCapacity: 1,
                            },
                            Rules: [{
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1,
                                        },
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            ComparisonOperator: 'GREATER_THAN',
                                            Dimensions: [{
                                                    Key: 'Key',
                                                    Value: 'Value',
                                                }],
                                            EvaluationPeriods: 1,
                                            MetricName: 'Name',
                                            Namespace: 'Namespace',
                                            Period: 300,
                                            Statistic: 'AVERAGE',
                                            Threshold: 1,
                                            Unit: 'NONE',
                                        },
                                    },
                                }, {
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1,
                                        },
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            'ComparisonOperator': 'GREATER_THAN',
                                            'Dimensions': [{
                                                    Key: 'Key',
                                                    Value: 'Value',
                                                }],
                                            'EvaluationPeriods': 1,
                                            'MetricName': 'Name',
                                            'Namespace': 'Namespace',
                                            'Period.$': '$.CloudWatchPeriod',
                                            'Statistic': 'AVERAGE',
                                            'Threshold': 1,
                                            'Unit': 'NONE',
                                        },
                                    },
                                }],
                        },
                        BidPrice: '1',
                        Configurations: [{
                                Classification: 'Classification',
                                Properties: {
                                    Key: 'Value',
                                },
                            }],
                        EbsConfiguration: {
                            EbsBlockDeviceConfigs: [{
                                    VolumeSpecification: {
                                        Iops: 1,
                                        SizeInGB: 1,
                                        VolumeType: 'standard',
                                    },
                                    VolumesPerInstance: 1,
                                }],
                            EbsOptimized: true,
                        },
                        InstanceCount: 1,
                        InstanceRole: 'MASTER',
                        InstanceType: 'm5.xlarge',
                        Market: 'ON_DEMAND',
                        Name: 'Name',
                    }],
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new lib_1.EmrCreateCluster(stack, 'Task', {
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
//# sourceMappingURL=data:application/json;base64,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