"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Terminate cluster with static ClusterId', () => {
    // WHEN
    const task = new tasks.EmrTerminateCluster(stack, 'Task', {
        clusterId: 'ClusterId',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:terminateCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
        },
    });
});
test('Terminate cluster with ClusterId from payload', () => {
    // WHEN
    const task = new tasks.EmrTerminateCluster(stack, 'Task', {
        clusterId: sfn.TaskInput.fromDataAt('$.ClusterId').value,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:terminateCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId',
        },
    });
});
test('Terminate cluster with static ClusterId and SYNC integrationPattern', () => {
    // WHEN
    const task = new tasks.EmrTerminateCluster(stack, 'Task', {
        clusterId: 'ClusterId',
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:terminateCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new tasks.EmrTerminateCluster(stack, 'Task', {
            clusterId: 'ClusterId',
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
//# sourceMappingURL=data:application/json;base64,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