"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../../lib/private/task-utils");
describe('Task utils', () => {
    describe('integration pattern validation', () => {
        test('supported integration pattern', () => {
            // GIVEN
            const supportedPatterns = [aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE];
            expect(() => {
                task_utils_1.validatePatternSupported(aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE, supportedPatterns);
            }).not.toThrow();
        });
        test('fails on unsupported integration pattern', () => {
            // GIVEN
            const supportedPatterns = [aws_stepfunctions_1.IntegrationPattern.RUN_JOB];
            expect(() => {
                task_utils_1.validatePatternSupported(aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN, supportedPatterns);
            }).toThrowError(/Unsupported service integration pattern. Supported Patterns: RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
        });
    });
    describe('integration resource Arn', () => {
        let service;
        let api;
        let stack;
        beforeEach(() => {
            // GIVEN
            service = 'lambda';
            api = 'invoke';
            stack = new cdk.Stack();
        });
        test('get resourceArn for a request/response integration pattern', () => {
            // WHEN
            const resourceArn = task_utils_1.integrationResourceArn(service, api, aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE);
            // THEN
            expect(stack.resolve(resourceArn)).toEqual({
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':states:::lambda:invoke',
                    ],
                ],
            });
        });
        test('get resourceArn for a run job integration pattern', () => {
            // WHEN
            const resourceArn = task_utils_1.integrationResourceArn(service, api, aws_stepfunctions_1.IntegrationPattern.RUN_JOB);
            // THEN
            expect(stack.resolve(resourceArn)).toEqual({
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':states:::lambda:invoke.sync',
                    ],
                ],
            });
        });
        test('get resourceArn for a wait for task token integration pattern', () => {
            // WHEN
            const resourceArn = task_utils_1.integrationResourceArn(service, api, aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN);
            // THEN
            expect(stack.resolve(resourceArn)).toEqual({
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':states:::lambda:invoke.waitForTaskToken',
                    ],
                ],
            });
        });
        test('fails when service or api is not specified', () => {
            expect(() => {
                task_utils_1.integrationResourceArn(service, '', aws_stepfunctions_1.IntegrationPattern.RUN_JOB);
            }).toThrow(/Both 'service' and 'api' must be provided to build the resource ARN./);
            expect(() => {
                task_utils_1.integrationResourceArn('', api, aws_stepfunctions_1.IntegrationPattern.RUN_JOB);
            }).toThrow(/Both 'service' and 'api' must be provided to build the resource ARN./);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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