"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RateLimitedApiKey = exports.ApiKey = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const usage_plan_1 = require("./usage-plan");
/**
 * Base implementation that is common to the various implementations of IApiKey
 */
class ApiKeyBase extends core_1.Resource {
    /**
     * Permits the IAM principal all read operations through this key
     *
     * @param grantee The principal to grant access to
     */
    grantRead(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: readPermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all write operations through this key
     *
     * @param grantee The principal to grant access to
     */
    grantWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: writePermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all read and write operations through this key
     *
     * @param grantee The principal to grant access to
     */
    grantReadWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [...readPermissions, ...writePermissions],
            resourceArns: [this.keyArn],
        });
    }
}
/**
 * An API Gateway ApiKey.
 *
 * An ApiKey can be distributed to API clients that are executing requests
 * for Method resources that require an Api Key.
 */
class ApiKey extends ApiKeyBase {
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_ApiKeyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const resource = new apigateway_generated_1.CfnApiKey(this, 'Resource', {
            customerId: props.customerId,
            description: props.description,
            enabled: (_c = props.enabled) !== null && _c !== void 0 ? _c : true,
            generateDistinctId: props.generateDistinctId,
            name: this.physicalName,
            stageKeys: this.renderStageKeys(props.resources),
            value: props.value,
        });
        this.keyId = resource.ref;
        this.keyArn = core_1.Stack.of(this).formatArn({
            service: 'apigateway',
            account: '',
            resource: '/apikeys',
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: this.keyId,
        });
    }
    /**
     * Import an ApiKey by its Id
     */
    static fromApiKeyId(scope, id, apiKeyId) {
        class Import extends ApiKeyBase {
            constructor() {
                super(...arguments);
                this.keyId = apiKeyId;
                this.keyArn = core_1.Stack.of(this).formatArn({
                    service: 'apigateway',
                    account: '',
                    resource: '/apikeys',
                    arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                    resourceName: apiKeyId,
                });
            }
        }
        return new Import(scope, id);
    }
    renderStageKeys(resources) {
        if (!resources) {
            return undefined;
        }
        return resources.map((resource) => {
            const restApi = resource;
            const restApiId = restApi.restApiId;
            const stageName = restApi.deploymentStage.stageName.toString();
            return { restApiId, stageName };
        });
    }
}
exports.ApiKey = ApiKey;
_a = JSII_RTTI_SYMBOL_1;
ApiKey[_a] = { fqn: "@aws-cdk/aws-apigateway.ApiKey", version: "1.156.1" };
/**
 * An API Gateway ApiKey, for which a rate limiting configuration can be specified.
 *
 * @resource AWS::ApiGateway::ApiKey
 */
class RateLimitedApiKey extends ApiKeyBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_RateLimitedApiKeyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const resource = new ApiKey(this, 'Resource', props);
        if (props.apiStages || props.quota || props.throttle) {
            const usageplan = new usage_plan_1.UsagePlan(this, 'UsagePlanResource', {
                apiStages: props.apiStages,
                quota: props.quota,
                throttle: props.throttle,
            });
            usageplan.addApiKey(resource);
        }
        this.keyId = resource.keyId;
        this.keyArn = resource.keyArn;
    }
}
exports.RateLimitedApiKey = RateLimitedApiKey;
_b = JSII_RTTI_SYMBOL_1;
RateLimitedApiKey[_b] = { fqn: "@aws-cdk/aws-apigateway.RateLimitedApiKey", version: "1.156.1" };
const readPermissions = [
    'apigateway:GET',
];
const writePermissions = [
    'apigateway:POST',
    'apigateway:PUT',
    'apigateway:PATCH',
    'apigateway:DELETE',
];
//# sourceMappingURL=data:application/json;base64,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