from typing import Optional, List
import abc

from .config import Config
from .utils import Utils


class Entity(abc.ABC):
    """Represents the single entity like the static page or the blog post.

    Attributes:
        template (str): Template that is used for the content generation.
        title (Optional[str]): Title (or name) of the page (meta tag).
        description (Optional[str]): Description of the page (meta tag).
        keywords (Optional[str]): Keywords of the page (meta tag).
        url_alias (Optional[str]): Alias for the main page (like my-page).
            If None, page is considered to be the main page (homepage).

        url_list (url_list: List[Optional[str]]): List of URLs that are
            generated by the entity (None refers to homepage).
    """

    def __init__(
        self,
        template: str,
        title: Optional[str] = None,
        description: Optional[str] = None,
        keywords: Optional[str] = None,
        url_alias: Optional[str] = None
    ):
        """Create the new entity.

        Args:
            template (str): Template that is used for the content generation.
            title (Optional[str]): Title (or name) of the page (meta tag).
            description (Optional[str]): Description of the page (meta tag).
            keywords (Optional[str]): Keywords of the page (meta tag).
            url_alias (Optional[str]): Alias for the main page (like my-page).
                If None, page is considered to be the main page (homepage).
        """
        self.template: str = template
        self.description: Optional[str] = description
        self.keywords: Optional[str] = keywords
        self.url_alias: Optional[str] = url_alias
        self.title: Optional[str] = title

        self.url_list: List[Optional[str]] = []

    @abc.abstractmethod
    def generate_page(self, url: str) -> str:
        """Generate the content for the particular URL.

        Args:
            url (str): URL defining what content should be generated.

        Returns:
            str: Content of the page.
        """
        raise NotImplementedError

    @property
    def page_title(self) -> str:
        """Generate the page title in the logic:
            SITE_NAME + SEPARATOR + TITLE

        Args:
            title (str): Title of the current page.

        Returns:
            str: Generated page title as a meta tag.
        """
        title = self.title
        if title is None and self.url_alias is None:
            # If the page is homepage and title is not set-up
            title = Config.site_title_homepage
        elif title is None or len(title) == 0:
            # If the title is not set-up, return just root title
            return Config.site_title
        # Return string in the logic SITE_NAME + SEPARATOR + TITLE
        return "".join(
            [Config.site_title, Config.site_title_separator, title]
        )

    @property
    def url(self):
        """Generate the URL leading to the file (with file suffix)"""
        return Utils.generate_file_path(self.url_alias)
