"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": [
        "export * from './api';",
        ...(options.sampleCode !== false ? ["export * from './sample-api';"] : []),
    ].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.ts": `import { OpenApiGatewayLambdaApi, OpenApiGatewayLambdaApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import * as path from "path";
import spec from "../${options.specDir}/${options.parsedSpecFileName}";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayLambdaApiProps, "spec" | "specPath" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
export class Api extends OpenApiGatewayLambdaApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      specPath: path.resolve(__dirname, "../${options.specDir}/${options.parsedSpecFileName}"),
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Authorizers } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { Cors } from "aws-cdk-lib/aws-apigateway";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      defaultAuthorizer: Authorizers.iam(),
      corsOptions: {
        allowOrigins: Cors.ALL_ORIGINS,
        allowMethods: Cors.ALL_METHODS,
      },
      integrations: {
        sayHello: {
          function: new NodejsFunction(scope, "say-hello"),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "sample-api.say-hello.ts": `import { sayHelloHandler } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler(async ({ input }) => {
  return {
    statusCode: 200,
    headers: {
      "Access-Control-Allow-Origin": "*",
      "Access-Control-Allow-Headers": "*",
    },
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.apiSrcDir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        function: new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        }),
      }])) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,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