import argparse
from typing import List

from wai.logging import LOGGING_WARNING
from ldc.core import DOMAIN_TRANSLATION, DOMAIN_PRETRAIN
from ldc.api.pretrain import PretrainData
from ldc.api.translation import TranslationData
from ldc.api import Filter


class TranslationToPretrain(Filter):
    """
    Converts records of translation records to pretrain ones.
    """

    def __init__(self, lang: str = None, logger_name: str = None, logging_level: str = LOGGING_WARNING):
        """
        Initializes the filter.

        :param lang: the ID of the language to turn into pretrain records
        :type lang: str
        :param logger_name: the name to use for the logger
        :type logger_name: str
        :param logging_level: the logging level to use
        :type logging_level: str
        """
        super().__init__(logger_name=logger_name, logging_level=logging_level)
        self.lang = lang

    def name(self) -> str:
        """
        Returns the name of the handler, used as sub-command.

        :return: the name
        :rtype: str
        """
        return "translation-to-pretrain"

    def description(self) -> str:
        """
        Returns a description of the handler.

        :return: the description
        :rtype: str
        """
        return "Converts records of translation data to pretrain ones, extracting a specific language."

    def domains(self) -> List[str]:
        """
        Returns the domains of the handler.

        :return: the domains
        :rtype: list
        """
        return [DOMAIN_TRANSLATION, DOMAIN_PRETRAIN]

    def accepts(self) -> List:
        """
        Returns the list of classes that are accepted.

        :return: the list of classes
        :rtype: list
        """
        return [TranslationData]

    def generates(self) -> List:
        """
        Returns the list of classes that get produced.

        :return: the list of classes
        :rtype: list
        """
        return [PretrainData]

    def _create_argparser(self) -> argparse.ArgumentParser:
        """
        Creates an argument parser. Derived classes need to fill in the options.

        :return: the parser
        :rtype: argparse.ArgumentParser
        """
        parser = super()._create_argparser()
        parser.add_argument("--lang", type=str, default=None, help="The ID of the language to convert")
        return parser

    def _apply_args(self, ns: argparse.Namespace):
        """
        Initializes the object with the arguments of the parsed namespace.

        :param ns: the parsed arguments
        :type ns: argparse.Namespace
        """
        super()._apply_args(ns)
        self.lang = ns.lang

    def initialize(self):
        """
        Initializes the processing, e.g., for opening files or databases.
        """
        super().initialize()
        if self.lang is None:
            raise Exception("No language ID specified!")

    def _do_process(self, data: TranslationData):
        """
        Processes the data record.

        :param data: the record to process
        :type data: PairData
        :return: the potentially updated record or None if to drop
        """
        if self.lang in data.translations:
            return PretrainData(data.translations[self.lang])
        else:
            return None
