"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => (Object.assign(Object.assign({}, acc), { [k]: v.conditions })), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
        });
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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