"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = exports.PublicKeyFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
var PublicKeyFormat;
(function (PublicKeyFormat) {
    PublicKeyFormat["OPENSSH"] = "OPENSSH";
    PublicKeyFormat["PEM"] = "PEM";
})(PublicKeyFormat = exports.PublicKeyFormat || (exports.PublicKeyFormat = {}));
/**
 * An EC2 Key Pair
 */
class KeyPair extends constructs_1.Construct {
    /**
     * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager
         */
        this.privateKeyArn = '';
        /**
         * ARN of the public key in AWS Secrets Manager
         */
        this.publicKeyArn = '';
        /**
         * The public key.
         *
         * Only filled, when `exposePublicKey = true`
         */
        this.publicKeyValue = '';
        /**
         * Name of the Key Pair
         */
        this.keyPairName = '';
        /**
         * ID of the Key Pair
         */
        this.keyPairID = '';
        this.prefix = '';
        if (props.removeKeySecretsAfterDays &&
            (props.removeKeySecretsAfterDays < 0 ||
                (props.removeKeySecretsAfterDays > 0 &&
                    props.removeKeySecretsAfterDays < 7) ||
                props.removeKeySecretsAfterDays > 30)) {
            aws_cdk_lib_1.Annotations.of(this).addError(`Parameter removeKeySecretsAfterDays must be 0 or between 7 and 30. Got ${props.removeKeySecretsAfterDays}`);
        }
        if (((_b = props.publicKey) === null || _b === void 0 ? void 0 : _b.length) &&
            props.publicKeyFormat !== undefined &&
            props.publicKeyFormat !== PublicKeyFormat.OPENSSH) {
            aws_cdk_lib_1.Annotations.of(this).addError('When importing a key, the format has to be of type OpenSSH');
        }
        const stack = aws_cdk_lib_1.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        if (this.prefix.length + cleanID.length > 62)
            // Cloudformation limits names to 63 characters.
            aws_cdk_lib_1.Annotations.of(this).addError(`Cloudformation limits names to 63 characters.
         Prefix ${this.prefix} is too long to be used as a prefix for your roleName. Define parameter resourcePrefix?:`);
        this.lambda = this.ensureLambda();
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag(createdByTag, ID);
        const kmsPrivate = props.kmsPrivateKey || props.kms;
        const kmsPublic = props.kmsPublicKey || props.kms;
        const key = new aws_cdk_lib_1.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            serviceToken: this.lambda.functionArn,
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KmsPrivate: (kmsPrivate === null || kmsPrivate === void 0 ? void 0 : kmsPrivate.keyArn) || 'alias/aws/secretsmanager',
                KmsPublic: (kmsPublic === null || kmsPublic === void 0 ? void 0 : kmsPublic.keyArn) || 'alias/aws/secretsmanager',
                PublicKey: props.publicKey || '',
                StorePublicKey: props.storePublicKey || false,
                ExposePublicKey: props.exposePublicKey || false,
                PublicKeyFormat: props.publicKeyFormat || PublicKeyFormat.OPENSSH,
                RemoveKeySecretsAfterDays: props.removeKeySecretsAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-ssh-key/',
                StackName: stack,
                Tags: aws_cdk_lib_1.Lazy.any({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPrivateKey !== 'undefined') {
            props.kmsPrivateKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPrivateKey);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPublicKey !== 'undefined') {
            props.kmsPublicKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPublicKey);
            key.node.addDependency(this.lambda.role);
        }
        this.privateKeyArn = key.getAttString('PrivateKeyARN');
        this.publicKeyArn = key.getAttString('PublicKeyARN');
        this.publicKeyValue = key.getAttString('PublicKeyValue');
        this.keyPairName = key.getAttString('KeyPairName');
        this.keyPairID = key.getAttString('KeyPairID');
    }
    ensureLambda() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const resources = [`arn:${stack.partition}:ec2:*:*:key-pair/*`];
        const policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeKeyPairs'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'ec2:CreateKeyPair',
                        'ec2:CreateTags',
                        'ec2:ImportKeyPair',
                    ],
                    conditions: {
                        StringLike: {
                            'aws:RequestTag/CreatedByCfnCustomResource': ID,
                        },
                    },
                    resources,
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    // allow delete/update, only if createdByTag is set
                    actions: ['ec2:CreateTags', 'ec2:DeleteKeyPair', 'ec2:DeleteTags'],
                    conditions: {
                        StringLike: {
                            'ec2:ResourceTag/CreatedByCfnCustomResource': ID,
                        },
                    },
                    resources,
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    // we need this to check if a secret exists before attempting to delete it
                    actions: ['secretsmanager:ListSecrets'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:TagResource',
                    ],
                    conditions: {
                        StringLike: {
                            'aws:RequestTag/CreatedByCfnCustomResource': ID,
                        },
                    },
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    // allow delete/update, only if createdByTag is set
                    actions: [
                        'secretsmanager:DeleteResourcePolicy',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:DescribeSecret',
                        'secretsmanager:GetResourcePolicy',
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:ListSecretVersionIds',
                        'secretsmanager:PutResourcePolicy',
                        'secretsmanager:PutSecretValue',
                        'secretsmanager:RestoreSecret',
                        'secretsmanager:UntagResource',
                        'secretsmanager:UpdateSecret',
                        'secretsmanager:UpdateSecretVersionStage',
                    ],
                    conditions: {
                        StringLike: {
                            'secretsmanager:ResourceTag/CreatedByCfnCustomResource': ID,
                        },
                    },
                    resources: ['*'],
                }),
            ],
        });
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new aws_cdk_lib_1.aws_lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: aws_cdk_lib_1.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager
     */
    grantReadOnPrivateKey(grantee) {
        return this.grantRead(this.privateKeyArn, grantee);
    }
    /**
     * Grants read access to the public key in AWS Secrets Manager
     */
    grantReadOnPublicKey(grantee) {
        return this.grantRead(this.publicKeyArn, grantee);
    }
    grantRead(arn, grantee) {
        const result = aws_cdk_lib_1.aws_iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
_a = JSII_RTTI_SYMBOL_1;
KeyPair[_a] = { fqn: "cdk-ec2-key-pair.KeyPair", version: "3.3.1" };
//# sourceMappingURL=data:application/json;base64,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