import os

import matplotlib.pyplot as plt
import scienceplots
from osgeo import gdal
from owslib.wcs import WebCoverageService

# use science style in matplotlib
plt.style.use("science")
import concurrent.futures
import time

import ee
import numpy as np
import pandas as pd
from pyproj import Transformer
from tqdm import tqdm


def get_county_bbox(county):
    """
    Returns the bounding box of an `ee.Geometry` object in EPSG:3857 projection.

    Args:
        county (ee.Geometry): The geometry of the county.

    Returns:
        tuple: The bounding box coordinates in EPSG:3857 projection as a tuple of (minx, miny, maxx, maxy).

    Example:
        >>> bbox = get_county_bbox(county)
        >>> bbox
    """
    # convert the geometry to EPSG:3857 projection
    transformer = Transformer.from_crs(4326, 3857)

    # get the coordinates of the county geometry
    coords = county.bounds().getInfo()["coordinates"][0]

    # extract the longitude and latitude values into separate lists
    lats, lngs = zip(*coords)

    fsrt_point = transformer.transform(min(lngs), min(lats))
    secd_point = transformer.transform(max(lngs), max(lats))

    return (fsrt_point[0], fsrt_point[1], secd_point[0], secd_point[1])


def remove_files() -> None:
    """
    Removes the temporary GeoTIFF files generated by the `get_county_mean_soil_data` function.

    Example:
        >>> remove_files()
    """
    file_path = os.path.dirname(os.path.abspath(__file__))
    for prop in ["bdod", "cec", "cfvo", "clay", "nitrogen", "phh2o", "sand", "silt", "soc", "ocd"]:
        out_tiff = os.path.join(file_path, "tmp", "soil-data", f"{prop}_0-5cm_mean")
        if os.path.exists(out_tiff):
            os.remove(out_tiff)


def get_county_mean_soil_data(county: ee.Geometry) -> list:
    """
    Returns the mean values of the soil properties for a given county.

    Args:
        county (ee.Geometry): The geometry of the county.

    Returns:
        list: The mean values of the soil properties for the given county.

    Example:
        >>> df = get_county_mean_soil_data(county)
        >>> df
    """
    # get the bounding box for the county
    bbox = get_county_bbox(county)

    # initialize a list to store the mean values of each property
    mean_values = []

    # iterate through each property of interest
    for prop in ["bdod", "cec", "cfvo", "clay", "nitrogen", "phh2o", "sand", "silt", "soc", "ocd"]:
        # initialize a WebCoverageService instance
        wcs = WebCoverageService(
            f"https://maps.isric.org/mapserv?map=/map/{prop}.map", version="1.0.0"
        )

        # get the coverage data for the given bounding box
        response = wcs.getCoverage(
            identifier=f"{prop}_0-5cm_mean",
            crs="urn:ogc:def:crs:EPSG::152160",
            bbox=bbox,
            resx=250,
            resy=250,
            format="GEOTIFF_INT16",
        )

        # save the coverage data to a file
        file_path = os.path.dirname(os.path.abspath(__file__))
        out_tiff = os.path.join(file_path, "tmp", "soil-data", f"{prop}_0-5cm_mean.tif")
        if not os.path.exists(os.path.dirname(out_tiff)):
            os.makedirs(os.path.dirname(out_tiff))

        with open(out_tiff, "wb") as f:
            f.write(response.read())

        # open the saved file and read the data as an array
        ds = gdal.Open(out_tiff)
        arr = ds.ReadAsArray()

        # calculate the mean value of the property and append it to the list of mean values
        mean_values.append(np.nanmean(arr))

    # Create a dataframe, county.toGeoJSON(), mean_values
    return pd.DataFrame([[county.toGeoJSON(), mean_values]], columns=["county", "soil_mean_values"])


def get_soil_all_locations(list_locations: list) -> pd.DataFrame:
    """
    Retrieves Sentinel-2 satellite data for a list of locations and time period of interest, and calculates
    vegetation indices for each location and time step in parallel.

    Args:
    - HIST_RANGE (tuple): A tuple specifying the start and end years of the time period of interest.
    - list_locations (list): A list of polygons defining the locations of interest.

    Returns:
    - A pandas dataframe with vegetation indices for each location and time step.
    """

    range_length = len(list_locations)
    print(f"Starting serial soil data processing for {range_length} locations...")

    # Execute get_sat_parallel function for each location in series
    start_time = time.time()
    results = []
    for location in tqdm(list_locations):
        result = get_county_mean_soil_data(location)
        results.append(result)
    end_time = time.time()

    elapsed_time = end_time - start_time
    print(f"Serial soil data processing completed in {elapsed_time:.2f} seconds.")

    # Concatenate results into a single dataframe
    return pd.concat(results)


# main : to test the function
# bbox = (-1784000, 1356000, -1140000, 1863000)
ee.Initialize()

county_1 = ee.Geometry.Polygon(
    [
        [-85.388717, 33.913044],
        [-85.380885, 33.873508],
        [-85.379455, 33.866291],
        [-85.377426, 33.856047],
        [-85.376403, 33.850656],
        [-85.364595, 33.788446],
        [-85.361844, 33.773951],
        [-85.360491, 33.767958],
        [-85.357402, 33.750104],
        [-85.355252, 33.739245],
        [-85.344054, 33.682684],
        [-85.342722, 33.675953],
        [-85.323792, 33.580339],
        [-85.315340, 33.537646],
        [-85.314994, 33.535898],
        [-85.314843, 33.534951],
        [-85.314091, 33.530218],
        [-85.313999, 33.529807],
        [-85.304439, 33.482884],
        [-85.308211, 33.481579],
        [-85.309250, 33.483137],
        [-85.314852, 33.487603],
        [-85.316028, 33.488267],
        [-85.320893, 33.488359],
        [-85.324856, 33.489161],
        [-85.331061, 33.491014],
        [-85.338280, 33.494700],
        [-85.342544, 33.495961],
        [-85.344923, 33.497608],
        [-85.346705, 33.501148],
        [-85.349958, 33.501216],
        [-85.351594, 33.499600],
        [-85.354491, 33.498866],
        [-85.352576, 33.494538],
        [-85.352573, 33.492438],
        [-85.355315, 33.492480],
        [-85.497455, 33.494624],
        [-85.501645, 33.494456],
        [-85.517310, 33.494524],
        [-85.525130, 33.494781],
        [-85.527515, 33.494608],
        [-85.539922, 33.494743],
        [-85.563763, 33.495081],
        [-85.565653, 33.494992],
        [-85.623387, 33.495371],
        [-85.623645, 33.495373],
        [-85.627835, 33.495624],
        [-85.643482, 33.495885],
        [-85.667220, 33.496293],
        [-85.680346, 33.496623],
        [-85.740983, 33.498376],
        [-85.765427, 33.498593],
        [-85.765308, 33.496862],
        [-85.765631, 33.483477],
        [-85.781244, 33.483625],
        [-85.782689, 33.483638],
        [-85.782735, 33.469349],
        [-85.887782, 33.469427],
        [-85.887675, 33.476768],
        [-85.870053, 33.476757],
        [-85.869308, 33.491574],
        [-85.852421, 33.491644],
        [-85.851890, 33.498742],
        [-85.849839, 33.499690],
        [-85.796852, 33.541849],
        [-85.796054, 33.556220],
        [-85.744118, 33.556075],
        [-85.742348, 33.586553],
        [-85.742202, 33.600002],
        [-85.737379, 33.599823],
        [-85.724517, 33.599430],
        [-85.724953, 33.613539],
        [-85.723670, 33.613492],
        [-85.723650, 33.613491],
        [-85.723074, 33.613505],
        [-85.689954, 33.612909],
        [-85.690140, 33.615815],
        [-85.690716, 33.625105],
        [-85.690684, 33.627789],
        [-85.674932, 33.627254],
        [-85.655810, 33.627166],
        [-85.655549, 33.638127],
        [-85.651295, 33.637972],
        [-85.651283, 33.641570],
        [-85.647190, 33.641529],
        [-85.647040, 33.648772],
        [-85.638579, 33.648413],
        [-85.638586, 33.649158],
        [-85.638049, 33.773339],
        [-85.621245, 33.773508],
        [-85.621019, 33.782331],
        [-85.620890, 33.787944],
        [-85.603469, 33.787755],
        [-85.603465, 33.788474],
        [-85.603391, 33.802344],
        [-85.585985, 33.802095],
        [-85.585201, 33.845920],
        [-85.636882, 33.846495],
        [-85.637126, 33.846497],
        [-85.637012, 33.875273],
        [-85.637014, 33.875944],
        [-85.601899, 33.874865],
        [-85.601913, 33.875101],
        [-85.601858, 33.889750],
        [-85.597063, 33.889703],
        [-85.532482, 33.889152],
        [-85.531812, 33.903049],
        [-85.530094, 33.941423],
        [-85.495820, 33.942394],
        [-85.495289, 33.956910],
        [-85.425627, 33.957069],
        [-85.425444, 33.960645],
        [-85.407831, 33.960548],
        [-85.407637, 33.964204],
        [-85.398837, 33.964129],
        [-85.391495, 33.927068],
        [-85.388717, 33.913044],
    ]
)

county_2 = ee.Geometry.Polygon(
    [
        [-86.030441, 31.618939],
        [-86.004085, 31.619067],
        [-85.999948, 31.619311],
        [-85.980098, 31.619189],
        [-85.926702, 31.618966],
        [-85.904460, 31.618775],
        [-85.899202, 31.618729],
        [-85.878888, 31.618547],
        [-85.832214, 31.618240],
        [-85.789142, 31.617964],
        [-85.789141, 31.587866],
        [-85.789140, 31.540617],
        [-85.789308, 31.490925],
        [-85.789370, 31.487561],
        [-85.789372, 31.487430],
        [-85.790294, 31.431467],
        [-85.789837, 31.385469],
        [-85.789815, 31.383145],
        [-85.789594, 31.372698],
        [-85.789754, 31.358887],
        [-85.789785, 31.356562],
        [-85.789866, 31.350574],
        [-85.790001, 31.342857],
        [-85.789975, 31.341524],
        [-85.790064, 31.337277],
        [-85.790065, 31.336908],
        [-85.790100, 31.336276],
        [-85.790116, 31.330082],
        [-85.790327, 31.323452],
        [-85.790434, 31.320267],
        [-85.790525, 31.317495],
        [-85.791089, 31.297603],
        [-85.791167, 31.295340],
        [-85.791243, 31.294184],
        [-85.791245, 31.294004],
        [-85.791214, 31.293551],
        [-85.791218, 31.293252],
        [-85.791361, 31.291777],
        [-85.791312, 31.269325],
        [-85.791291, 31.209854],
        [-85.791402, 31.196349],
        [-85.847766, 31.196369],
        [-85.857352, 31.196367],
        [-85.857999, 31.196374],
        [-85.865347, 31.196374],
        [-85.882227, 31.196339],
        [-85.938287, 31.196344],
        [-85.949386, 31.196123],
        [-85.950142, 31.196242],
        [-85.960322, 31.195962],
        [-85.966110, 31.195935],
        [-85.999042, 31.195777],
        [-86.013060, 31.195318],
        [-86.058535, 31.194082],
        [-86.116734, 31.193015],
        [-86.116735, 31.183885],
        [-86.116736, 31.182099],
        [-86.125405, 31.182057],
        [-86.125482, 31.192754],
        [-86.141674, 31.192693],
        [-86.193476, 31.192213],
        [-86.194003, 31.211362],
        [-86.193542, 31.213028],
        [-86.193846, 31.219589],
        [-86.193714, 31.243119],
        [-86.193336, 31.250169],
        [-86.193546, 31.262915],
        [-86.193888, 31.283526],
        [-86.194631, 31.327933],
        [-86.194779, 31.336812],
        [-86.195044, 31.352681],
        [-86.195056, 31.353359],
        [-86.196091, 31.410348],
        [-86.196365, 31.425431],
        [-86.193951, 31.440072],
        [-86.193973, 31.446586],
        [-86.194724, 31.524460],
        [-86.194784, 31.529949],
        [-86.182098, 31.530047],
        [-86.156561, 31.530203],
        [-86.156768, 31.537527],
        [-86.150314, 31.537601],
        [-86.143950, 31.537675],
        [-86.144761, 31.570994],
        [-86.145895, 31.617741],
        [-86.124952, 31.618044],
        [-86.123834, 31.618167],
        [-86.094925, 31.618405],
        [-86.058021, 31.618793],
        [-86.030441, 31.618939],
    ]
)

county_3 = ee.Geometry.Polygon(
    [
        [-86.009280, 33.101638],
        [-86.009170, 33.090260],
        [-86.008612, 33.031827],
        [-86.008480, 33.018451],
        [-86.008342, 32.997798],
        [-86.008317, 32.975710],
        [-86.008260, 32.967391],
        [-86.008263, 32.966054],
        [-86.008308, 32.939847],
        [-86.008204, 32.922635],
        [-86.007904, 32.906464],
        [-86.007892, 32.903686],
        [-86.007847, 32.887614],
        [-86.007658, 32.864033],
        [-86.007646, 32.860147],
        [-86.007573, 32.831505],
        [-86.007565, 32.827773],
        [-86.012407, 32.827959],
        [-86.013845, 32.826543],
        [-86.013578, 32.825828],
        [-86.009502, 32.824990],
        [-86.007472, 32.825100],
        [-86.007529, 32.823573],
        [-86.009100, 32.822933],
        [-86.011020, 32.823634],
        [-86.012772, 32.821350],
        [-86.013490, 32.820121],
        [-86.009021, 32.813125],
        [-86.007308, 32.812201],
        [-86.007086, 32.802242],
        [-86.004887, 32.802159],
        [-86.003878, 32.801108],
        [-86.000050, 32.801645],
        [-86.005163, 32.798524],
        [-86.004727, 32.797860],
        [-86.006896, 32.796431],
        [-86.004943, 32.795572],
        [-86.006219, 32.794967],
        [-86.005733, 32.791877],
        [-86.007214, 32.791536],
        [-86.007187, 32.754984],
        [-86.059186, 32.754879],
        [-86.136260, 32.754302],
        [-86.149272, 32.754092],
        [-86.154734, 32.754272],
        [-86.197967, 32.754025],
        [-86.206590, 32.753532],
        [-86.229360, 32.753453],
        [-86.231580, 32.753483],
        [-86.280875, 32.753434],
        [-86.296069, 32.753537],
        [-86.319480, 32.753698],
        [-86.319451, 32.756207],
        [-86.319238, 32.758360],
        [-86.319163, 32.767837],
        [-86.336694, 32.768130],
        [-86.336776, 32.767469],
        [-86.336768, 32.753783],
        [-86.374974, 32.753580],
        [-86.379953, 32.755229],
        [-86.388017, 32.757543],
        [-86.392616, 32.759907],
        [-86.396737, 32.762946],
        [-86.399421, 32.768120],
        [-86.402648, 32.770826],
        [-86.405906, 32.772983],
        [-86.408702, 32.776712],
        [-86.411564, 32.783102],
        [-86.413402, 32.788182],
        [-86.418830, 32.790985],
        [-86.428674, 32.792316],
        [-86.434748, 32.798428],
        [-86.443834, 32.804482],
        [-86.444620, 32.805663],
        [-86.443925, 32.808646],
        [-86.441309, 32.810494],
        [-86.442493, 32.812497],
        [-86.457015, 32.813899],
        [-86.458735, 32.818818],
        [-86.460232, 32.827527],
        [-86.460581, 32.832051],
        [-86.459965, 32.837044],
        [-86.457384, 32.842844],
        [-86.454380, 32.846707],
        [-86.447837, 32.850976],
        [-86.446876, 32.854007],
        [-86.448453, 32.858610],
        [-86.451524, 32.863692],
        [-86.454994, 32.866656],
        [-86.458239, 32.868329],
        [-86.465672, 32.869141],
        [-86.471379, 32.869790],
        [-86.474932, 32.870672],
        [-86.478711, 32.872659],
        [-86.480222, 32.874350],
        [-86.481214, 32.877159],
        [-86.481419, 32.880305],
        [-86.482573, 32.883923],
        [-86.484497, 32.887242],
        [-86.486785, 32.892561],
        [-86.488679, 32.895825],
        [-86.490511, 32.905619],
        [-86.493532, 32.908929],
        [-86.495145, 32.909959],
        [-86.498443, 32.911653],
        [-86.500522, 32.912219],
        [-86.503991, 32.912295],
        [-86.505751, 32.913276],
        [-86.506892, 32.914552],
        [-86.507194, 32.917733],
        [-86.508613, 32.919894],
        [-86.511706, 32.922302],
        [-86.515959, 32.929361],
        [-86.514134, 32.935566],
        [-86.514262, 32.937641],
        [-86.516186, 32.943176],
        [-86.516064, 32.950376],
        [-86.515169, 32.953308],
        [-86.515398, 32.955067],
        [-86.517181, 32.962462],
        [-86.517287, 32.966065],
        [-86.518053, 32.968837],
        [-86.518117, 32.971181],
        [-86.516631, 32.973430],
        [-86.513537, 32.975549],
        [-86.511627, 32.977922],
        [-86.510580, 32.981510],
        [-86.510622, 32.985380],
        [-86.522283, 33.004395],
        [-86.521396, 33.006659],
        [-86.519176, 33.014915],
        [-86.518681, 33.018315],
        [-86.517344, 33.020566],
        [-86.516582, 33.023184],
        [-86.516508, 33.028705],
        [-86.516900, 33.030518],
        [-86.516770, 33.039479],
        [-86.516446, 33.042220],
        [-86.518816, 33.046199],
        [-86.519734, 33.049687],
        [-86.519630, 33.051927],
        [-86.520700, 33.054602],
        [-86.520061, 33.056987],
        [-86.514056, 33.060571],
        [-86.511859, 33.061498],
        [-86.507973, 33.064403],
        [-86.506474, 33.066063],
        [-86.506026, 33.067748],
        [-86.507011, 33.071801],
        [-86.510651, 33.077835],
        [-86.511391, 33.079910],
        [-86.512106, 33.085876],
        [-86.511142, 33.088431],
        [-86.509050, 33.090899],
        [-86.493489, 33.099159],
        [-86.491029, 33.102944],
        [-86.404734, 33.103727],
        [-86.398693, 33.103825],
        [-86.381599, 33.102989],
        [-86.374575, 33.103017],
        [-86.315339, 33.103772],
        [-86.251106, 33.104312],
        [-86.247875, 33.104035],
        [-86.229880, 33.104101],
        [-86.226271, 33.104115],
        [-86.192521, 33.104303],
        [-86.175936, 33.104387],
        [-86.174370, 33.104394],
        [-86.162541, 33.104452],
        [-86.124970, 33.104595],
        [-86.124123, 33.104692],
        [-86.123294, 33.104691],
        [-86.086238, 33.104702],
        [-86.079858, 33.104695],
        [-86.069706, 33.104687],
        [-86.044079, 33.104664],
        [-86.009313, 33.104642],
        [-86.009280, 33.101638],
    ]
)


counties = [county_1, county_2, county_3]

# print(get_county_mean_soil_data(county_1))
print(get_soil_all_locations(counties))
