/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import { Graph } from "../../../../../../cdk-graph";
import * as Dot from "ts-graphviz";
import { Node } from "./nodes";
import { Container } from "./subgraphs";
/**
 * Union of targets supported by edges
 * @internal
 */
type EdgeTarget = Node | Container;
/**
 * BaseEdge class is the base class for defining a {@link Dot.Edge}.
 * @internal
 */
export declare abstract class BaseEdge extends Dot.Edge {
    /** Tail edge target */
    readonly from: EdgeTarget;
    /** Head edge target */
    readonly to: EdgeTarget;
    /** Indicates if edge is [compound](https://graphviz.org/docs/attrs/compound/), meaning it is between a cluster */
    readonly isCompound: boolean;
    /**
     * Indicates if [compound](https://graphviz.org/docs/attrs/compound/) edge targets are synthetically added.
     *
     * For *compound* edges, if the cluster (container) does not have any children a *synthetic* child is
     * added to support edge targeting.
     */
    readonly isSynthetic: boolean;
    /** @internal */
    protected _extraneous: boolean;
    /** Indicates if edge is considered **extraneous** */
    get isExtraneous(): boolean;
    /** Indicates if edge is considered **verbose** */
    get isVerbose(): boolean;
    /** Indicates if edge is a *closed loop*, meaning its *leaf* and *head* are the same entity */
    get isClosedLoop(): boolean;
    /** @internal */
    constructor(from: EdgeTarget, to: EdgeTarget);
}
/**
 * Edge class is the base class for {@link Graph.Edge} based {@link Dot.Edge} entities
 * @internal
 */
export declare class Edge extends BaseEdge {
    /** Reference to the {@link Graph.Edge} that this diagram edge is based on */
    readonly graphEdge: Graph.Edge;
    /** @internal */
    constructor(edge: Graph.Edge, from: EdgeTarget, to: EdgeTarget);
}
/**
 * Link class defines a {@link Graph.Edge} defined by a {@link Graph.Node}
 * @internal
 */
export declare class Link extends Edge {
    /** @internal */
    constructor(edge: Graph.Edge, from: EdgeTarget, to: EdgeTarget);
}
/**
 * ChildLink class defines a {@link Dot.Edge} for a {@link Graph.Edge} that describes a parent-child {@link Graph.Node} relationship
 * @internal
 */
export declare class ChildLink extends BaseEdge {
    /** @internal */
    constructor(from: EdgeTarget, to: EdgeTarget);
}
/**
 * ReferenceLink class defines a {@link Dot.Edge} for a {@link Graph.Reference} edge
 * @internal
 */
export declare class ReferenceLink extends Link {
    /** @internal */
    constructor(edge: Graph.Edge, from: EdgeTarget, to: EdgeTarget);
}
/**
 * DependencyLink class defines a {@link Dot.Edge} for a {@link Graph.Dependency} edge
 * @internal
 */
export declare class DependencyLink extends Link {
    /** @internal */
    constructor(edge: Graph.Edge, from: EdgeTarget, to: EdgeTarget);
}
export {};
