import { Construct, IConstruct } from "constructs";
import { SGEntity, SGUnresolvedReference, Attributes, Tags } from "./serialized-graph";
import { FlagEnum, UUID } from "./types";
import { ConstructInfo } from "../cdk-internals";
/**
 * Generate deterministic UUID based on given value and prefix.
 * @param value The value to hash as UUID
 * @param {string} [prefix=""] Optional prefix used to prevent value conflicts
 */
export declare function generateConsistentUUID(value: any, prefix?: string): string;
/** Get UUID for a given construct */
export declare function getConstructUUID(construct: IConstruct): string;
/** Try to get *logicalId* for given construct */
export declare function tryGetLogicalId(construct: IConstruct): string | undefined;
/** Inferred node props */
export interface InferredNodeProps extends SGEntity {
    readonly logicalId?: string;
    readonly cfnType?: string;
    readonly constructInfo?: ConstructInfo;
    readonly dependencies: UUID[];
    readonly unresolvedReferences: SGUnresolvedReference[];
}
/** Infer node props from construct */
export declare function inferNodeProps(construct: Construct): InferredNodeProps;
/** Extract inspectable attributes from construct */
export declare function extractInspectableAttributes(construct: IConstruct): Attributes | undefined;
/** Pattern of ignored references. Those which are resolved during deploy-time. */
export declare const IGNORE_REF_PATTERN: RegExp;
/** Extract unresolved references from attributes for a given source */
export declare function extractUnresolvedReferences(source: UUID, from: Attributes): SGUnresolvedReference[];
/** Infer construct flags  */
export declare function inferFlags(construct: IConstruct, constructInfo?: ConstructInfo, tags?: Tags): FlagEnum[];
/**
 * Indicates if given construct is an import (eg: `s3.Bucket.fromBucketArn()`)
 */
export declare function isImportConstruct(construct: Construct): boolean;
/**
 * Resolve an imported resources arn to tokenized hash value of arn.
 * @see {@link tokenizeImportArn}
 * @param construct {Construct} Imported resource to resolve arn for.
 * @returns If construct is an imported resource and able to infer the arn for it then the tokenized arn value is returned, otherwise undefined
 */
export declare function resolveImportedConstructArnToken(construct: Construct): string | undefined;
/**
 * Generate token for imported resource arn used to resolve references.
 *
 * Imported resources are CDK `s3.Bucket.fromBucketArn()` like resources
 * that are external from the application.
 * @param value The value to tokenize, which is usually an object with nested `Fn:Join: ...["arn:", ...]` format.
 * @returns Consistent string hash prefixed with `ImportArn-` prefix.
 */
export declare function tokenizeImportArn(value: any): string;
/**
 * Infers CloudFormation Type for a given import resource.
 * @param construct {Construct} Import construct such as `s3.Bucket.fromBucketArn()`.
 * @param constructInfo {ConstructInfo} Construct info like fqn
 * @returns Returns Cloudformation resource type if it can be inferred, otherwise undefined.
 */
export declare function inferImportCfnType(construct: Construct, constructInfo?: ConstructInfo): string | undefined;
