import { Component, Project, Task } from "projen";
import { JavaProject } from "projen/lib/java";
import { PythonProject } from "projen/lib/python";
import { NxWorkspace } from "./nx-workspace";
import { Nx } from "../nx-types";
/**
 * Options for overriding nx build tasks
 * @internal
 */
interface OverrideNxBuildTaskOptions {
    /**
     * Force unlocking task (eg: build task is locked)
     */
    readonly force?: boolean;
    /**
     * Disable further resets of the task by other components in further lifecycle stages
     * (eg eslint resets during preSynthesize)
     */
    readonly disableReset?: boolean;
}
/**
 * Interface that all NXProject implementations should implement.
 */
export interface INxProjectCore {
    /**
     * Return the NxWorkspace instance. This should be implemented using a getter.
     */
    readonly nx: NxWorkspace;
    /**
     * Helper to format `npx nx run-many ...` style command execution in package manager.
     * @param options
     */
    execNxRunManyCommand(options: Nx.RunManyOptions): string;
    /**
     * Helper to format `npx nx run-many ...` style command
     * @param options
     */
    composeNxRunManyCommand(options: Nx.RunManyOptions): string[];
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     *
     * @param name task name
     * @param options options
     */
    addNxRunManyTask(name: string, options: Nx.RunManyOptions): Task;
    /**
     * Create an implicit dependency between two Projects. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     * @throws error if this is called on a dependent which does not have a NXProject component attached.
     */
    addImplicitDependency(dependent: Project, dependee: Project | string): void;
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependent project you want to have the dependency
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependent: JavaProject, dependee: JavaProject): void;
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependent project you want to have the dependency (must be a Poetry Python Project)
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependent: PythonProject, dependee: PythonProject): void;
}
/**
 * License options.
 *
 */
export interface LicenseOptions {
    /**
     * License type (SPDX).
     *
     * @see https://github.com/projen/projen/tree/main/license-text for list of supported licenses
     */
    readonly spdx?: string;
    /**
     * Copyright owner.
     *
     * If the license text for the given spdx has $copyright_owner, this option must be specified.
     */
    readonly copyrightOwner?: string;
    /**
     * Arbitrary license text.
     */
    readonly licenseText?: string;
    /**
     * Whether to disable the generation of default licenses.
     *
     * @default false
     */
    readonly disableDefaultLicenses?: boolean;
}
/**
 * NXConfigurator options.
 */
export interface NxConfiguratorOptions {
    /**
     * Branch that NX affected should run against.
     */
    readonly defaultReleaseBranch?: string;
    /**
     * Default package license config.
     *
     * If nothing is specified, all packages will default to Apache-2.0 (unless they have their own License component).
     */
    readonly licenseOptions?: LicenseOptions;
}
/**
 * Configues common NX related tasks and methods.
 */
export declare class NxConfigurator extends Component implements INxProjectCore {
    readonly nx: NxWorkspace;
    private readonly licenseOptions?;
    private nxPlugins;
    constructor(project: Project, options?: NxConfiguratorOptions);
    patchPoetryEnv(project: PythonProject): void;
    patchPythonProjects(projects: Project[]): void;
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param overrideOptions - Options for overriding the task
     * @returns - The task that was overridden
     * @internal
     */
    _overrideNxBuildTask(task: Task | string | undefined, options: Nx.RunManyOptions, overrideOptions?: OverrideNxBuildTaskOptions): Task | undefined;
    /**
     * Adds a command to upgrade all python subprojects to the given task
     * @param monorepo the monorepo project
     * @param task the task to add the command to
     * @internal
     */
    _configurePythonSubprojectUpgradeDeps(monorepo: Project, task: Task): void;
    /**
     * Returns the install task or creates one with nx installation command added.
     *
     * Note: this should only be called from non-node projects
     *
     * @param nxPlugins additional plugins to install
     * @returns install task
     */
    ensureNxInstallTask(nxPlugins: {
        [key: string]: string;
    }): Task;
    /**
     * Helper to format `npx nx run-many ...` style command execution in package manager.
     * @param options
     */
    execNxRunManyCommand(options: Nx.RunManyOptions): string;
    /**
     * Helper to format `npx nx run-many ...` style command
     * @param options
     */
    composeNxRunManyCommand(options: Nx.RunManyOptions): string[];
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name: string, options: Nx.RunManyOptions): Task;
    /**
     * Create an implicit dependency between two Projects. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     * @throws error if this is called on a dependent which does not have a NXProject component attached.
     */
    addImplicitDependency(dependent: Project, dependee: Project | string): void;
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependent project you want to have the dependency
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependent: JavaProject, dependee: JavaProject): void;
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependent project you want to have the dependency (must be a Poetry Python Project)
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependent: PythonProject, dependee: PythonProject): void;
    /**
     * Ensures that all non-root projects have NxProject applied.
     * @internal
     */
    protected _ensureNxProjectGraph(): void;
    /**
     * Emits package.json for non-node NX monorepos.
     * @internal
     */
    private _emitPackageJson;
    private _invokeInstallCITasks;
    /**
     * Add licenses to any subprojects which don't already have a license.
     */
    private _addLicenses;
    preSynthesize(): void;
    /**
     * @inheritDoc
     */
    synth(): void;
    /**
     * Ensures subprojects don't have a default task
     */
    private resetDefaultTask;
}
export {};
