/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import { ApiGatewayIntegrationResponse, IntegrationRenderProps } from "./integration";
/**
 * A set of integration responses. You can extend this to implement your own integration response set
 */
export declare abstract class IntegrationResponseSet {
    /**
     * Render the integration responses into the Integration
     * @param props the integration render props (same used in the Integration)
     */
    abstract render(props: IntegrationRenderProps): {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
/**
 * An integration response set that catches all errors and returns a 500
 */
export declare class CatchAllErrorIntegrationResponseSet extends IntegrationResponseSet {
    render(props: IntegrationRenderProps): {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
/**
 * Options for the DefaultPassthroughIntegrationResponseSet
 */
export interface DefaultPassthroughIntegrationResponseSetProps {
    /**
     * Override the status code returned by the default integration response
     * @default 200
     */
    readonly statusCode?: number;
}
/**
 * An integration response set which adds a passthrough for the default response
 */
export declare class DefaultPassthroughIntegrationResponseSet extends IntegrationResponseSet {
    private readonly statusCode;
    constructor(props?: DefaultPassthroughIntegrationResponseSetProps);
    render(props: IntegrationRenderProps): {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
/**
 * An integration response set for S3 errors, that returns a JSON payload with the S3 error message
 */
export declare class S3JsonErrorMessageIntegrationResponseSet extends IntegrationResponseSet {
    private response;
    render(props: IntegrationRenderProps): {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
/**
 * Properties for a custom integration response set
 */
export interface CustomIntegrationResponseSetProps {
    /**
     * The responses to add to the integration response
     */
    readonly responses?: {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
/**
 * A custom integration response set where you can specify the responses as you wish
 */
export declare class CustomIntegrationResponseSet extends IntegrationResponseSet {
    private readonly responses?;
    constructor(props: CustomIntegrationResponseSetProps);
    render(props: IntegrationRenderProps): {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
/**
 * An integration response set which combines the provided integration response sets.
 * Response sets are combined in order, so later responses will override previous ones for
 * overlapping response status patterns.
 */
export declare class CompositeIntegrationResponseSet extends IntegrationResponseSet {
    private readonly responseSets;
    constructor(...responseSets: IntegrationResponseSet[]);
    render(props: IntegrationRenderProps): {
        [responseStatusPattern: string]: ApiGatewayIntegrationResponse;
    };
}
