/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import { MavenCompileOptions } from "projen/lib/java";
import { JavaVersion, NodeVersion, PythonVersion } from "../languages";
/**
 * Utilities for java runtime versions
 */
declare class JavaRuntimeVersionUtils {
    /**
     * Get the maven compile options for the given java runtime
     */
    static getMavenCompileOptions: (runtimeVersion?: JavaVersion) => MavenCompileOptions;
    /**
     * Return the CDK lambda runtime constant for the given java version
     * @see https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_lambda.Runtime.html
     */
    static getLambdaRuntime: (runtimeVersion: JavaVersion) => string;
}
/**
 * Utilities for node runtime versions
 */
declare class NodeRuntimeVersionUtils {
    /**
     * Return the CDK lambda runtime constant for the given node version
     * @see https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_lambda.Runtime.html
     */
    static getLambdaRuntime: (runtimeVersion: NodeVersion) => string;
    /**
     * Return the target node version for esbuild
     * @see https://esbuild.github.io/api/#target
     */
    static getEsbuildNodeTarget: (runtimeVersion: NodeVersion) => string;
}
/**
 * Utilities for python runtime versions
 */
declare class PythonRuntimeVersionUtils {
    /**
     * Return the CDK lambda runtime constant for the given python version
     * @see https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_lambda.Runtime.html
     */
    static getLambdaRuntime: (runtimeVersion: PythonVersion) => string;
    /**
     * Return the version string used for a dependency on python
     */
    static getPythonDependencyVersion: (runtimeVersion: PythonVersion) => string;
    /**
     * Return the version string used for packaging python lambdas with pip
     */
    static getPipPackagingPythonVersion: (runtimeVersion: PythonVersion) => string;
    /**
     * Return the version string for python
     */
    private static getPythonVersionString;
}
/**
 * A collection of utilities for runtime versions.
 */
export declare class RuntimeVersionUtils {
    /**
     * Java utilities
     */
    static JAVA: typeof JavaRuntimeVersionUtils;
    /**
     * Node utilities
     */
    static NODE: typeof NodeRuntimeVersionUtils;
    /**
     * Python utilities
     */
    static PYTHON: typeof PythonRuntimeVersionUtils;
}
export {};
