"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findCfnTypeAssets = exports.parseCfnType = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const assets_1 = require("../../generated/assets");
const utils_1 = require("../../utils");
const helpers_1 = require("../resources/helpers");
const helpers_2 = require("../services/helpers");
/** Parse CloudFormation resource type string into parts */
function parseCfnType(cfnType) {
    return cfnType.match(/(?<provider>\w+)::(?<serviceName>\w+)::(?<resourceName>\w+)/).groups;
}
exports.parseCfnType = parseCfnType;
// VPC is nested in EC2 within CloudFormation but separate in other systems
const VpcAssetComparables = Object.fromEntries(Object.keys(assets_1.AwsAsset.Resources).reduce((entries, key) => {
    if (key.startsWith("vpc_")) {
        entries.push([(0, utils_1.normalizeComparisonString)(key.replace("vpc_", "")), key]);
    }
    return entries;
}, []));
/** Find asset type details for given CloudFormation type */
function findCfnTypeAssets(cfnType) {
    const cfn = parseCfnType(cfnType);
    let serviceName;
    let resourceName;
    // handle edge cases (eg: EC2::VPCxxx is vpc:xxx in assets)
    if (cfn.serviceName === "EC2") {
        if (cfn.resourceName.startsWith("VPC")) {
            serviceName = "vpc";
            cfn.resourceName = cfn.resourceName.replace(/^VPC/, "");
        }
        else if ((0, utils_1.normalizeComparisonString)(cfn.resourceName) in VpcAssetComparables) {
            serviceName = "vpc";
            cfn.resourceName = (0, utils_1.normalizeComparisonString)(cfn.resourceName);
        }
    }
    if (serviceName == null) {
        try {
            serviceName = (0, helpers_2.resolveServiceName)(cfn.serviceName);
        }
        catch (e) {
            console.warn(e.message, cfnType);
        }
    }
    if (resourceName == null) {
        // There are lots of low-level cfn resource definitions without mappings to other systems,
        // for this reason we just ignore unresolved resources without spamming the console or
        // bubbling the error as this is expected in large percent of cases.
        try {
            resourceName = (0, helpers_1.resolveResourceName)(cfn.resourceName, serviceName);
        }
        catch { }
    }
    const generalIcon = resolveGeneralIcon(cfn.resourceName);
    return {
        serviceName,
        resourceName,
        generalIcon,
    };
}
exports.findCfnTypeAssets = findCfnTypeAssets;
/** Resolve general icon for a given value.  */
function resolveGeneralIcon(value) {
    value = (0, utils_1.normalizeComparisonString)(value);
    if (value.match(/(configuration|config|setting)$/i)) {
        return "config";
    }
    if (value.match(/^(client)?(certificate)$/i)) {
        return "ssl_padlock";
    }
    if (value.match(/(user)$/i)) {
        return "user";
    }
    if (value.match(/(policy|permissions?)$/i)) {
        return "policy";
    }
    if (value.match(/(key)$/i)) {
        return "key";
    }
    if (value.match(/(role)$/i)) {
        return "role";
    }
    if (value.match(/(server)$/i)) {
        return "traditional_server";
    }
    if (value.match(/(database)$/i)) {
        return "generic_database";
    }
    if (value.match(/(log)s?$/i)) {
        return "logs";
    }
    if (value.match(/(alarm|alert)s?$/i)) {
        return "alarm";
    }
    if (value.match(/(event)s?$/i)) {
        return "event";
    }
    if (value.match(/(rule)s?$/i)) {
        return "rule";
    }
    if (value.match(/(branch)$/i)) {
        return "git_repository";
    }
    if (value.match(/^(app|application)$/i)) {
        return "generic_application";
    }
    if (value.match(/^(documentation)/i)) {
        return "document";
    }
    if (value.match(/^(model)$/i)) {
        return "document";
    }
    if (value.match(/(template)/i)) {
        return "template";
    }
    if (value.match(/(attachment)$/i)) {
        return "attachment";
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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