"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractResolvedServiceName = exports.resolveServiceName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const variants_1 = require("./variants");
const assets_1 = require("../../generated/assets");
const utils_1 = require("../../utils");
/** @internal */
const SERVICE_NAME_LOOKUP = Object.fromEntries([
    ...Object.keys(assets_1.AwsAsset.Services).flatMap((key) => {
        return [
            // constant self lookup
            [key, key],
            [(0, utils_1.normalizeComparisonString)(key), key],
        ];
    }),
    ...Object.entries(variants_1.NORMALIZED_SERVICE_VARIANTS).flatMap(([key, variants]) => {
        if (variants == null)
            return [];
        return variants.map((variant) => [(0, utils_1.normalizeComparisonString)(variant), key]);
    }),
]);
/** @internal */
function resolveServiceName(value) {
    const comparableValue = (0, utils_1.normalizeComparisonString)(value);
    // constant lookup first for perf
    if (comparableValue in SERVICE_NAME_LOOKUP) {
        return SERVICE_NAME_LOOKUP[comparableValue];
    }
    // attempt know mappings
    if (comparableValue.startsWith("media") &&
        SERVICE_NAME_LOOKUP[`elemental${comparableValue}`]) {
        const serviceName = SERVICE_NAME_LOOKUP[`elemental${comparableValue}`];
        SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
        return serviceName;
    }
    // prefixed lookup (WAFRegion => WAF, CodeGuruProfile => CodeGuru)
    for (const serviceName of Object.keys(assets_1.AwsAsset.Services)) {
        if (comparableValue.startsWith((0, utils_1.normalizeComparisonString)(serviceName))) {
            // add to lookup for constant lookkup in subsequent calls;
            SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
            return serviceName;
        }
    }
    throw new Error(`Failed to resolve serviceName ${value} (${comparableValue})`);
}
exports.resolveServiceName = resolveServiceName;
/**
 * Extract service name parts from value that is or contains a service name or variant of service name.
 * @internal
 */
function extractResolvedServiceName(value) {
    const parts = (0, utils_1.normalizeIdentifier)(value).split("_");
    for (let i = parts.length; i > 0; i--) {
        try {
            const match = parts.slice(0, i).join("_");
            const serviceName = resolveServiceName(match);
            let rest = value.replace(match, "");
            if (rest.length === 0) {
                rest = undefined;
            }
            const resolvedValue = `${serviceName}${rest}`;
            return { serviceName, match, rest, resolvedValue };
        }
        catch {
            continue;
        }
    }
    return;
}
exports.extractResolvedServiceName = extractResolvedServiceName;
//# sourceMappingURL=data:application/json;base64,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