"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDiagram = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../aws-arch");
const cdk_graph_1 = require("../../../cdk-graph");
const uniqBy = require("lodash.uniqby"); // eslint-disable-line @typescript-eslint/no-require-imports
const Diagram = require("./entities");
const entities_1 = require("./entities");
const theme_1 = require("./theme");
const debug_1 = require("../debug");
/**
 * EdgeResolver class resolves edges within a diagram for rendering
 * @internal
 */
class EdgeResolver {
    constructor() {
        /** @internal */
        this._edges = new Map();
    }
    /** Adds diagram edge to the resolver to be resolved */
    trackEdge(edge) {
        let fromId;
        let toId;
        if (edge instanceof Diagram.Edge) {
            fromId = edge.graphEdge.source.uuid;
            toId = edge.graphEdge.target.uuid;
        }
        else {
            fromId = edge.from.graphNode.uuid;
            toId = edge.to.graphNode.uuid;
        }
        const key = fromId < toId ? `${fromId}:${toId}` : `${toId}:${fromId}`;
        const edgeSet = this._edges.get(key) || [];
        this._edges.set(key, edgeSet.concat([edge]));
    }
    /** Resolve all edges based on diagram options */
    resolveEdges(options) {
        const compact = options.preset === cdk_graph_1.FilterPreset.COMPACT;
        const resolvedEdges = [];
        for (let edges of this._edges.values()) {
            if (compact) {
                edges = edges.filter((edge) => !edge.isVerbose);
            }
            if (edges.length === 0) {
                continue;
            }
            edges.sort((a, b) => {
                const _a = getEdgeRank(a);
                const _b = getEdgeRank(b);
                if (_a === _b)
                    return 0;
                if (_a < _b)
                    return -1;
                return 1;
            });
            edges = uniqBy(edges, getEdgeRank);
            // only return highest ranked edge unless verbose
            if (compact) {
                resolvedEdges.push(edges[0]);
                continue;
            }
            for (const _edge1 of edges) {
                for (const _edge2 of edges) {
                    if (_edge1 === _edge2)
                        continue;
                    const _id1 = _edge1.attributes.get("id");
                    const _id2 = _edge2.attributes.get("id");
                    let _sameHead1 = _edge1.attributes.get("samehead");
                    _edge1.attributes.set("samehead", !_sameHead1 ? _id2 : `${_sameHead1},${_id2}`);
                    let _sameTail1 = _edge1.attributes.get("sametail");
                    _edge1.attributes.set("sametail", !_sameTail1 ? _id2 : `${_sameTail1},${_id2}`);
                    let _sameHead2 = _edge2.attributes.get("samehead");
                    _edge2.attributes.set("samehead", !_sameHead2 ? _id1 : `${_sameHead2},${_id1}`);
                    let _sameTail2 = _edge2.attributes.get("sametail");
                    _edge2.attributes.set("sametail", !_sameTail2 ? _id1 : `${_sameTail2},${_id1}`);
                }
            }
            resolvedEdges.push(...edges);
        }
        return resolvedEdges;
    }
}
/**
 * Build a {@link Diagram.Diagram Diagram} for a given {@link Graph.Store} based on {@link DiagramOptions Options}
 * @internal
 */
function buildDiagram(store, options) {
    const { title, nodePositions } = options;
    const edgeResolve = new EdgeResolver();
    theme_1.GraphTheme.init(options.theme);
    const entities = new Map();
    const diagram = new Diagram.Diagram(title, aws_arch_1.AwsArchitecture.assetDirectory);
    function visit(gNode, parent) {
        if (gNode.isDestroyed)
            return;
        let entity;
        switch (gNode.nodeType) {
            case cdk_graph_1.NodeTypeEnum.RESOURCE: {
                entity = new Diagram.ResourceNode(gNode);
                break;
            }
            case cdk_graph_1.NodeTypeEnum.CFN_RESOURCE: {
                entity = new Diagram.CfnResourceNode(gNode);
                break;
            }
            case cdk_graph_1.NodeTypeEnum.NESTED_STACK: {
                entity = new Diagram.NestedStackCluster(gNode);
                break;
            }
            case cdk_graph_1.NodeTypeEnum.STACK: {
                if (theme_1.GraphTheme.instance.rendering.stack &&
                    new RegExp(theme_1.GraphTheme.instance.rendering.stack).test(gNode.id) ===
                        false) {
                    // Ignore non-matching root stacks
                    return;
                }
                entity = new Diagram.StackCluster(gNode);
                break;
            }
            default: {
                if (gNode.isLeaf) {
                    if (gNode.hasFlag(cdk_graph_1.FlagEnum.CUSTOM_RESOURCE)) {
                        entity = new Diagram.CustomResourceNode(gNode);
                    }
                    else {
                        entity = new Diagram.Node(gNode);
                    }
                }
                else {
                    entity = new Diagram.Cluster(gNode);
                    gNode.addFlag(cdk_graph_1.FlagEnum.CLUSTER);
                }
                break;
            }
        }
        if (nodePositions && entity instanceof Diagram.Node) {
            if (entity.graphNode.id in nodePositions) {
                entity.position = nodePositions[entity.graphNode.id];
            }
        }
        if (entity instanceof entities_1.ImageNode && entity.image) {
            diagram.trackImage(entity.image);
        }
        if (parent instanceof Diagram.Container && parent.linkChildren) {
            edgeResolve.trackEdge(new entities_1.ChildLink(parent, entity));
        }
        if (gNode.isLeaf) {
            entities.set(gNode.uuid, entity);
            parent.addNode(entity);
        }
        else {
            if (entity instanceof Diagram.Node) {
                entity = asContainer(entity);
            }
            parent.addSubgraph(entity);
            entities.set(gNode.uuid, entity);
            gNode.children.forEach((child) => visit(child, entity));
        }
    }
    if (store.stages.length > 1) {
        const stageRendering = theme_1.GraphTheme.instance.rendering.stage || "last";
        let stages;
        switch (stageRendering) {
            case "all": {
                stages = store.stages;
                break;
            }
            case "first": {
                stages = store.stages.slice(0, 1);
                break;
            }
            case "last": {
                stages = store.stages.slice(-1);
                break;
            }
            default: {
                stages = store.stages.filter((stage) => stage.id.match(stageRendering));
            }
        }
        // traverse all stages
        stages.forEach((gStage) => {
            const dStage = new Diagram.StageCluster(gStage);
            diagram.addSubgraph(dStage);
            entities.set(gStage.uuid, dStage);
            gStage.children.forEach((child) => visit(child, dStage));
        });
    }
    else if (store.rootStacks.length) {
        // traverse all root stack
        store.rootStacks.forEach((gStack) => {
            const dStack = new Diagram.StackCluster(gStack);
            diagram.addSubgraph(dStack);
            entities.set(gStack.uuid, dStack);
            gStack.children.forEach((child) => visit(child, dStack));
        });
    }
    else {
        store.root.children.forEach((gChild) => {
            if (gChild.isGraphContainer) {
                gChild.children.forEach((_gChild) => {
                    visit(_gChild, diagram);
                });
            }
            else {
                visit(gChild, diagram);
            }
        });
    }
    // apply all edges
    store.edges.forEach((gEdge) => {
        if (gEdge.isDestroyed)
            return;
        const dSource = entities.get(gEdge.source.uuid);
        const dTarget = entities.get(gEdge.target.uuid);
        if (!dSource || !dTarget) {
            debug_1.IS_DEBUG &&
                console.warn("Diagram.Edge unresolved source and/or target:", `source(${gEdge.source} => ${dSource})`, `target(${gEdge.target} => ${dTarget})`);
            return;
        }
        let edge = undefined;
        switch (gEdge.edgeType) {
            case cdk_graph_1.EdgeTypeEnum.REFERENCE: {
                edge = new Diagram.ReferenceLink(gEdge, dSource, dTarget);
                break;
            }
            case cdk_graph_1.EdgeTypeEnum.DEPENDENCY: {
                edge = new Diagram.DependencyLink(gEdge, dSource, dTarget);
                break;
            }
        }
        if (edge) {
            entities.set(gEdge.uuid, edge);
            edgeResolve.trackEdge(edge);
        }
    });
    edgeResolve.resolveEdges(options).forEach((edge) => {
        diagram.addEdge(edge);
    });
    return diagram;
}
exports.buildDiagram = buildDiagram;
/**
 * Wrap a {@link Diagram.Node} with {@link Diagram.Container} to support adding child {@link Diagram.Node}s
 * @internal
 */
function asContainer(node) {
    const container = new Diagram.Container(node.graphNode);
    container.addNode(node);
    return container;
}
/**
 * Get the rank score of an {@link Diagram.BaseEdge Edge} used to sort and prioritize edges
 * @internal
 */
function getEdgeRank(edge) {
    if (edge instanceof Diagram.ChildLink) {
        return 0;
    }
    if (edge instanceof Diagram.ReferenceLink) {
        return 1;
    }
    if (edge instanceof Diagram.DependencyLink) {
        return 2;
    }
    return 3;
}
//# sourceMappingURL=data:application/json;base64,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