"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudscapeReactTsWebsiteProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const Mustache = require("mustache");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const web_1 = require("projen/lib/web");
/**
 * Synthesizes a Cloudscape React Typescript Website Project.
 *
 * @pjid cloudscape-react-ts-website
 */
class CloudscapeReactTsWebsiteProject extends web_1.ReactTypeScriptProject {
    constructor(options) {
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            name: options.name,
            sampleCode: false,
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager ?? javascript_1.NodePackageManager.PNPM,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../samples/cloudscape-react-ts-website/README.md"))
                    .toString(),
            },
            gitignore: [
                "public/runtime-config.json",
                "public/api.json",
                "public/*/api.json",
                ...(options.gitignore || []),
            ],
        });
        this.typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        this.typeSafeWebSocketApis = options.typeSafeWebSocketApis ?? [];
        this.addDeps("@aws-northstar/ui", "@cloudscape-design/components", "@cloudscape-design/board-components", "react-router-dom");
        this.testTask.reset();
        const lintTask = this.tasks.tryFind("eslint");
        lintTask && this.testTask.spawn(lintTask);
        this.testTask.exec("react-scripts test --watchAll=false --passWithNoTests");
        this.applicationName = options.applicationName ?? "Sample App";
        this.allowSignup = options.allowSignup ?? false;
        this.publicDir = options.publicDir ?? "public";
        const srcDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/src");
        const publicDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/public");
        this.typeSafeApis.forEach((typeSafeApi) => {
            const hooks = typeSafeApi.library?.typescriptReactQueryHooks;
            const libraryHooksPackage = hooks?.package?.packageName;
            const libraryHooksPackageVersion = hooks?.package?.manifest.version;
            if (!libraryHooksPackage) {
                throw new Error("Cannot pass in a Type Safe Api without React Hooks Library configured!");
            }
            this.addDeps(`${libraryHooksPackage}@${libraryHooksPackageVersion}`);
            this.setupApiExplorer(typeSafeApi);
        });
        this.typeSafeWebSocketApis.forEach((typeSafeWebSocketApi) => {
            const hooks = typeSafeWebSocketApi.library?.typescriptWebSocketHooks;
            const client = typeSafeWebSocketApi.library?.typescriptWebSocketClient;
            if (!hooks || !client) {
                throw new Error("Cannot pass in a Type Safe WebSocket Api without React Hooks Library configured");
            }
            this.addDeps(`${hooks.package.packageName}@${hooks.package.manifest.version}`, `${client.package.packageName}@${client.package.manifest.version}`);
            this.setupApiExplorer(typeSafeWebSocketApi);
        });
        const apis = this.typeSafeApis.map((tsApi, idx) => ({
            apiName: tsApi.model.apiName,
            isLast: idx === this.typeSafeApis.length - 1,
            apiNameSafe: tsApi.model.apiName
                ?.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""),
            hooksPackage: tsApi.library?.typescriptReactQueryHooks?.package?.packageName,
        }));
        const webSocketApis = this.typeSafeWebSocketApis.map((tsApi, idx) => ({
            apiName: tsApi.model.apiName,
            isLast: idx === this.typeSafeWebSocketApis.length - 1,
            apiNameSafe: tsApi.model.apiName
                ?.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""),
            hooksPackage: tsApi.library?.typescriptWebSocketHooks?.package?.packageName,
            clientPackage: tsApi.library?.typescriptWebSocketClient?.package?.packageName,
        }));
        const mustacheConfig = {
            applicationName: this.applicationName,
            hasApis: apis.length + webSocketApis.length > 0,
            typeSafeApis: apis,
            typeSafeApisReversed: [...apis].reverse(),
            typeSafeWebSocketApis: webSocketApis,
            typeSafeWebSocketApisReversed: [...webSocketApis].reverse(),
            allowSignup: options.allowSignup ?? false,
        };
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(srcDir, [], mustacheConfig)),
            },
        });
        new projen_1.SampleDir(this, this.publicDir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(publicDir, [], mustacheConfig)),
            },
        });
        // Linting is managed as part of the test task already, so disable react-scripts running eslint again
        this.tasks.addEnvironment("DISABLE_ESLINT_PLUGIN", "true");
        // Relax EsLint and TSC for dev
        this.tasks.tryFind("dev")?.env("ESLINT_NO_DEV_ERRORS", "true");
        this.tasks.tryFind("dev")?.env("TSC_COMPILE_ON_ERROR", "true");
    }
    setupApiExplorer(tsApi) {
        this.addDevDeps("@types/swagger-ui-react", "@types/uuid");
        this.addDeps("swagger-ui-react@5.5.0", "aws4fetch", "uuid", "@aws-crypto/sha256-js", "@aws-sdk/signature-v4", "@aws-sdk/protocol-http");
        const targetApiSpecFolder = `public/${tsApi.model.apiName}`;
        const targetApiSpecPath = `${targetApiSpecFolder}/api.json`;
        this.preCompileTask.exec(`rm -rf ${targetApiSpecFolder}`);
        this.preCompileTask.exec(`mkdir -p ${targetApiSpecFolder} && cp ${path.relative(this.outdir, tsApi.model.outdir)}/.api.json ${targetApiSpecPath}`);
    }
    buildSampleDirEntries(dir, pathPrefixes = [], mustacheConfig) {
        return fs
            .readdirSync(dir, { withFileTypes: true })
            .filter((f) => mustacheConfig.typeSafeApis.length +
            mustacheConfig.typeSafeWebSocketApis.length >
            0 || !`${pathPrefixes.join("/")}${f.name}`.includes("ApiExplorer"))
            .filter((f) => mustacheConfig.typeSafeApis.length > 0 ||
            !`${pathPrefixes.join("/")}${f.name}`.includes("TypeSafeApiClient"))
            .filter((f) => mustacheConfig.typeSafeWebSocketApis.length > 0 ||
            !`${pathPrefixes.join("/")}${f.name}`.includes("TypeSafeWebSocketApiClient"))
            .flatMap((f) => f.isDirectory()
            ? this.buildSampleDirEntries(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig)
            : [
                [
                    `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`,
                    Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                ],
            ]);
    }
}
exports.CloudscapeReactTsWebsiteProject = CloudscapeReactTsWebsiteProject;
_a = JSII_RTTI_SYMBOL_1;
CloudscapeReactTsWebsiteProject[_a] = { fqn: "@aws/pdk.cloudscape_react_ts_website.CloudscapeReactTsWebsiteProject", version: "0.23.72" };
//# sourceMappingURL=data:application/json;base64,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