import { Project, Task } from "projen";
import { JavaProject } from "projen/lib/java";
import { PythonProject } from "projen/lib/python";
import { TypeScriptProject } from "projen/lib/typescript";
import { Syncpack } from "./syncpack-options";
import { TypeScriptProjectOptions } from "./typescript-project-options";
import { NxConfigurator, INxProjectCore, LicenseOptions } from "../../components/nx-configurator";
import { NxWorkspace } from "../../components/nx-workspace";
import { Nx } from "../../nx-types";
/**
 * Workspace configurations.
 *
 * @see https://classic.yarnpkg.com/lang/en/docs/workspaces/
 */
export interface WorkspaceConfig {
    /**
     * Yarn workspace config
     */
    readonly yarn?: YarnWorkspaceConfig;
    /**
     * Links all local workspace project bins so they can be used for local development.
     *
     * Package bins are only linked when installed from the registry, however it is very useful
     * for monorepo development to also utilize these bin scripts. When enabled, this flag will
     * recursively link all bins from packages.json files to the root node_modules/.bin.
     */
    readonly linkLocalWorkspaceBins?: boolean;
    /**
     * List of additional package globs to include in the workspace.
     *
     * All packages which are parented by the monorepo are automatically added to the workspace, but you can use this
     * property to specify any additional paths to packages which may not be managed by projen.
     */
    readonly additionalPackages?: string[];
}
/**
 * Yarn related workspace config
 */
export interface YarnWorkspaceConfig {
    /**
     * List of package globs to exclude from hoisting in the workspace.
     *
     * @see https://classic.yarnpkg.com/blog/2018/02/15/nohoist/
     */
    readonly noHoist?: string[];
    /**
     * Disable automatically applying `noHoist` logic for all sub-project "bundledDependencies".
     *
     * @default false
     */
    readonly disableNoHoistBundled?: boolean;
}
/**
 * Configuration for Monorepo Upgrade Deps task.
 */
export interface MonorepoUpgradeDepsOptions {
    /**
     * Name of the task to create.
     *
     * @default upgrade-deps
     */
    readonly taskName?: string;
    /**
     * Syncpack configuration.
     *
     * No merging is performed and as such a complete syncpackConfig is required if supplied.
     *
     * @default Syncpack.DEFAULT_CONFIG
     */
    readonly syncpackConfig?: Syncpack.SyncpackConfig;
}
/**
 * Configuration options for the MonorepoTsProject.
 */
export interface MonorepoTsProjectOptions extends TypeScriptProjectOptions {
    /**
     * Configuration for workspace.
     */
    readonly workspaceConfig?: WorkspaceConfig;
    /**
     * Whether to include an upgrade-deps task at the root of the monorepo which will upgrade all dependencies.
     *
     * @default true
     */
    readonly monorepoUpgradeDeps?: boolean;
    /**
     * Monorepo Upgrade Deps options.
     *
     * This is only used if monorepoUpgradeDeps is true.
     *
     * @default undefined
     */
    readonly monorepoUpgradeDepsOptions?: MonorepoUpgradeDepsOptions;
    /**
     * Disable node warnings from being emitted during build tasks
     * @default false
     */
    readonly disableNodeWarnings?: boolean;
    /**
     * Default license to apply to all PDK managed packages.
     *
     * @default Apache-2.0
     */
    readonly licenseOptions?: LicenseOptions;
}
/**
 * This project type will bootstrap a monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-ts
 */
export declare class MonorepoTsProject extends TypeScriptProject implements INxProjectCore {
    /**
     * Version of projen used by the monorepo and its subprojects
     */
    private readonly projenVersion;
    private readonly workspaceConfig?;
    private readonly workspacePackages;
    readonly nxConfigurator: NxConfigurator;
    private readonly _options;
    private subNodeProjectResolves;
    constructor(options: MonorepoTsProjectOptions);
    /**
     * @inheritdoc
     */
    get nx(): NxWorkspace;
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options: Nx.RunManyOptions): string;
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options: Nx.RunManyOptions): string[];
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name: string, options: Nx.RunManyOptions): Task;
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent: Project, dependee: string | Project): void;
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent: JavaProject, dependee: JavaProject): void;
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent: PythonProject, dependee: PythonProject): void;
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs: string[]): void;
    /** Get consistently sorted list of subprojects */
    get sortedSubProjects(): Project[];
    /**
     * Create symbolic links to all local workspace bins. This enables the usage of bins the same
     * way as consumers of the packages have when installing from the registry.
     */
    protected linkLocalWorkspaceBins(): void;
    private addUpgradeDepsTask;
    preSynthesize(): void;
    /**
     * @inheritDoc
     */
    synth(): void;
    /**
     * @inheritDoc
     */
    postSynthesize(): void;
    /**
     * Resolve sub `NodePackage` dependencies.
     */
    private resolveSubNodeProjects;
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    private validateSubProjects;
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    private updateWorkspace;
    /**
     * Ensures the install task for non-node projects is executed postinstall.
     *
     * @private
     */
    private installNonNodeDependencies;
    /**
     * Suppress Node warnings from being presented in the console when running builds.
     */
    private disableNodeWarnings;
    /**
     * Resolve all local workspace dependencies to keep bun happy.
     */
    private resolveLocalBunDependencies;
}
