/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
/**
 * Types and constants for Syncpack usage in the monorepo
 * @see https://jamiemason.github.io/syncpack
 */
export declare namespace Syncpack {
    /**
     * Aliases for semver range formats supported by syncpack
     *
     * Defaults to `""` to ensure that exact dependency versions are used instead of
     * loose ranges, but this can be overridden in your config file.
     *
     * | Supported Range |   Example |
     * | --------------- | --------: |
     * | `"<"`           |  `<1.4.2` |
     * | `"<="`          | `<=1.4.2` |
     * | `""`            |   `1.4.2` |
     * | `"~"`           |  `~1.4.2` |
     * | `"^"`           |  `^1.4.2` |
     * | `">="`          | `>=1.4.2` |
     * | `">"`           |  `>1.4.2` |
     * | `"*"`           |       `*` |
     *
     * @default ""
     */
    export type SemverRange = "" | "*" | ">" | ">=" | ".x" | "<" | "<=" | "^" | "~" | "workspace:";
    export const CUSTOM_TYPES: {
        readonly dev: {
            readonly strategy: "versionsByName";
            readonly path: "devDependencies";
        };
        readonly local: {
            readonly strategy: "name~version";
            readonly namePath: "name";
            readonly path: "version";
        };
        readonly overrides: {
            readonly strategy: "versionsByName";
            readonly path: "overrides";
        };
        readonly peer: {
            readonly strategy: "versionsByName";
            readonly path: "peerDependencies";
        };
        readonly pnpmOverrides: {
            readonly strategy: "versionsByName";
            readonly path: "pnpm.overrides";
        };
        readonly prod: {
            readonly strategy: "versionsByName";
            readonly path: "dependencies";
        };
        readonly resolutions: {
            readonly strategy: "versionsByName";
            readonly path: "resolutions";
        };
    };
    type DefaultDependencyType = keyof typeof CUSTOM_TYPES;
    export type DependencyType = DefaultDependencyType | `!${DefaultDependencyType}` | (string & {});
    export type SpecifierType = string;
    export namespace BaseGroupConfig {
        interface GroupConfig {
            readonly dependencies?: string[];
            readonly dependencyTypes?: DependencyType[];
            readonly label?: string;
            readonly packages?: string[];
            readonly specifierTypes?: SpecifierType[];
        }
    }
    export namespace SemverGroupConfig {
        interface Disabled extends BaseGroupConfig.GroupConfig {
            readonly isDisabled: true;
        }
        interface Ignored extends BaseGroupConfig.GroupConfig {
            readonly isIgnored: true;
        }
        interface WithRange extends BaseGroupConfig.GroupConfig {
            readonly range: SemverRange;
        }
        type Any = Disabled | Ignored | WithRange;
    }
    export namespace VersionGroupConfig {
        interface Banned extends BaseGroupConfig.GroupConfig {
            readonly isBanned: true;
        }
        interface Ignored extends BaseGroupConfig.GroupConfig {
            readonly isIgnored: true;
        }
        interface Pinned extends BaseGroupConfig.GroupConfig {
            readonly pinVersion: string;
        }
        interface SnappedTo extends BaseGroupConfig.GroupConfig {
            readonly snapTo: string[];
        }
        interface SameRange extends BaseGroupConfig.GroupConfig {
            readonly policy: "sameRange";
        }
        interface SnappedTo extends BaseGroupConfig.GroupConfig {
            readonly snapTo: string[];
        }
        interface Standard extends BaseGroupConfig.GroupConfig {
            readonly preferVersion?: "highestSemver" | "lowestSemver";
        }
        type Any = Banned | Ignored | Pinned | SameRange | SnappedTo | Standard;
    }
    export namespace CustomTypeConfig {
        interface NameAndVersionProps {
            readonly namePath: string;
            readonly path: string;
            readonly strategy: "name~version";
        }
        interface NamedVersionString {
            readonly path: string;
            readonly strategy: "name@version";
        }
        interface UnnamedVersionString {
            readonly path: string;
            readonly strategy: "version";
        }
        interface VersionsByName {
            readonly path: string;
            readonly strategy: "versionsByName";
        }
        type Any = NameAndVersionProps | NamedVersionString | UnnamedVersionString | VersionsByName;
    }
    export interface CliConfig {
        readonly configPath?: string;
        readonly filter: string;
        readonly indent: string;
        readonly source: string[];
        readonly specs: string;
        readonly types: string;
    }
    /**
     * Configuration for Syncpack
     * @see https://jamiemason.github.io/syncpack
     */
    export interface SyncpackConfig {
        /** @see https://jamiemason.github.io/syncpack/config/custom-types */
        readonly customTypes?: Record<string, CustomTypeConfig.Any>;
        /** @see https://jamiemason.github.io/syncpack/config/dependency-types */
        readonly dependencyTypes?: DependencyType[];
        /** @see https://jamiemason.github.io/syncpack/config/filter */
        readonly filter?: string;
        /** @see https://jamiemason.github.io/syncpack/config/format-bugs */
        readonly formatBugs?: boolean;
        /** @see https://jamiemason.github.io/syncpack/config/format-repository */
        readonly formatRepository?: boolean;
        /** @see https://jamiemason.github.io/syncpack/config/indent */
        readonly indent?: string;
        /** @see https://jamiemason.github.io/syncpack/config/lint-formatting */
        readonly lintFormatting?: boolean;
        /** @see https://jamiemason.github.io/syncpack/config/lint-semver-ranges */
        readonly lintSemverRanges?: boolean;
        /** @see https://jamiemason.github.io/syncpack/config/lint-versions */
        readonly lintVersions?: boolean;
        /** @see https://jamiemason.github.io/syncpack/config/semver-groups */
        readonly semverGroups?: SemverGroupConfig.Any[];
        /** @see https://jamiemason.github.io/syncpack/config/sort-az */
        readonly sortAz?: string[];
        /** @see https://jamiemason.github.io/syncpack/config/sort-exports */
        readonly sortExports?: string[];
        /** @see https://jamiemason.github.io/syncpack/config/sort-first */
        readonly sortFirst?: string[];
        /** @see https://jamiemason.github.io/syncpack/config/sort-packages */
        readonly sortPackages?: boolean;
        /** @see https://jamiemason.github.io/syncpack/config/source */
        readonly source?: string[];
        /** @see https://jamiemason.github.io/syncpack/config/specifier-types */
        readonly specifierTypes?: SpecifierType[];
        /** @see https://jamiemason.github.io/syncpack/config/version-groups */
        readonly versionGroups?: VersionGroupConfig.Any[];
    }
    /**
     * Default monorepo configuration for Syncpack
     * @see https://jamiemason.github.io/syncpack/
     */
    export const DEFAULT_CONFIG: Syncpack.SyncpackConfig;
    export {};
}
