'use strict';

var index_js = require('../utils/index.cjs');
var index_js$1 = require('../common/index.cjs');

class Builder {
  options;
  getLocation() {
    return this.options?.locationFunction?.() ?? null;
  }
  constructor(options) {
    this.options = options;
  }
  createElement(type, props, children = []) {
    return {
      location: this.getLocation(),
      ...props,
      type,
      children,
    };
  }
}

const createElement = Builder.prototype.createElement.bind(new Builder());

const EOL = /\r?\n/;
function joinBy(sep) {
  return (value) => value.join(sep);
}
function wrapWith(wrapper) {
  return (value) => wrapper + value + wrapper;
}
function wrapByPair(l, r) {
  return (value) => l + value + r;
}
function leftPadWith(left) {
  return (value) => left + value;
}
const escape = (value) => value.replace(/\r/g, '\\r').replace(/\n/g, '\\n').replace(/"/g, '\\"');
const splitByLine = (value) => value.split(EOL);
const indent = (style, size, eol) =>
  index_js.pipe(splitByLine, index_js.map(leftPadWith(style === 'space' ? ' '.repeat(size) : '\n')), joinBy(eol));
const endOfLine = (eol) => {
  switch (eol) {
    case 'crlf':
      return '\r\n';
    case 'lf':
      return '\n';
  }
};

const AttributeListPrintPlugin = {
  match(ast) {
    return ast.type === 'AttributeList';
  },
  print(context, ast) {
    if (ast.children.length === 0) {
      return `${ast.kind.toLocaleLowerCase()} [];`;
    }
    const eol = endOfLine(context.endOfLine);
    return index_js.pipe(
      index_js.map(context.print),
      joinBy(eol),
      indent(context.indentStyle, context.indentSize, eol),
      wrapByPair(`${ast.kind.toLocaleLowerCase()} [${eol}`, `${eol}];`),
    )(ast.children);
  },
};

const AttributePrintPlugin = {
  match(ast) {
    return ast.type === 'Attribute';
  },
  print(context, ast) {
    return `${context.print(ast.key)} = ${context.print(ast.value)};`;
  },
};

const CommentPrintPlugin = {
  match(ast) {
    return ast.type === 'Comment';
  },
  print(context, ast) {
    const eol = endOfLine(context.endOfLine);
    switch (ast.kind) {
      case 'Block':
        return index_js.pipe(
          splitByLine,
          index_js.map(leftPadWith(' * ')),
          joinBy(eol),
          wrapByPair(`/**${eol}`, `${eol} */`),
        )(ast.value);
      case 'Macro':
        return index_js.pipe(splitByLine, index_js.map(leftPadWith('# ')), joinBy(eol))(ast.value);
      case 'Slash':
      default:
        return index_js.pipe(splitByLine, index_js.map(leftPadWith('// ')), joinBy(eol))(ast.value);
    }
  },
};

const DotPrintPlugin = {
  match(ast) {
    return ast.type === 'Dot';
  },
  print(context, ast) {
    return ast.children.map(context.print).join(endOfLine(context.endOfLine));
  },
};

const EdgePrintPlugin = {
  match(ast) {
    return ast.type === 'Edge';
  },
  print(context, ast) {
    const targets = index_js.pipe(index_js.map(context.print), joinBy(context.directed ? ' -> ' : ' -- '))(ast.targets);
    if (ast.children.length === 0) {
      return `${targets};`;
    }
    const eol = endOfLine(context.endOfLine);
    const contents = index_js.pipe(
      index_js.map(context.print),
      joinBy(eol),
      indent(context.indentStyle, context.indentSize, eol),
      wrapByPair(`[${eol}`, `${eol}];`),
    )(ast.children);
    return `${targets} ${contents}`;
  },
};

const GraphPrintPlugin = {
  match(ast) {
    return ast.type === 'Graph';
  },
  print(context, ast) {
    context.directed = ast.directed;
    const parts = [];
    if (ast.strict) {
      parts.push('strict');
    }
    parts.push(ast.directed ? 'digraph' : 'graph');
    if (ast.id) {
      parts.push(context.print(ast.id));
    }
    if (ast.children.length === 0) {
      return `${parts.join(' ')} {}`;
    }
    const eol = endOfLine(context.endOfLine);
    const contents = index_js.pipe(
      index_js.map(context.print),
      joinBy(eol),
      indent(context.indentStyle, context.indentSize, eol),
      wrapByPair(`{${eol}`, `${eol}}`),
    )(ast.children);
    return `${parts.join(' ')} ${contents}`;
  },
};

const quoteLiteralValue = index_js.pipe(escape, wrapWith('"'));
const quoteHTMLLikeLiteralValue = wrapByPair('<', '>');
const LiteralPrintPlugin = {
  match(ast) {
    return ast.type === 'Literal';
  },
  print(context, ast) {
    switch (ast.quoted) {
      case 'html':
        return quoteHTMLLikeLiteralValue(ast.value);
      case true:
        return quoteLiteralValue(ast.value);
      case false:
      default:
        return escape(ast.value);
    }
  },
};

const NodePrintPlugin = {
  match(ast) {
    return ast.type === 'Node';
  },
  print(context, ast) {
    const id = context.print(ast.id);
    if (ast.children.length === 0) {
      return `${id};`;
    }
    const eol = endOfLine(context.endOfLine);
    const contents = index_js.pipe(
      index_js.map(context.print),
      joinBy(eol),
      indent(context.indentStyle, context.indentSize, eol),
      wrapByPair(`[${eol}`, `${eol}];`),
    )(ast.children);
    return `${id} ${contents}`;
  },
};

const NodeRefGroupPrintPlugin = {
  match(ast) {
    return ast.type === 'NodeRefGroup';
  },
  print(context, ast) {
    return index_js.pipe(index_js.map(context.print), joinBy(' '), wrapByPair('{', '}'))(ast.children);
  },
};

const NodeRefPrintPlugin = {
  match(ast) {
    return ast.type === 'NodeRef';
  },
  print(context, ast) {
    const parts = [context.print(ast.id)];
    if (ast.port) {
      parts.push(context.print(ast.port));
    }
    if (ast.compass) {
      parts.push(context.print(ast.compass));
    }
    return parts.join(':');
  },
};

const SubgraphPrintPlugin = {
  match(ast) {
    return ast.type === 'Subgraph';
  },
  print(context, ast) {
    const parts = ['subgraph'];
    if (ast.id) {
      parts.push(context.print(ast.id));
    }
    if (ast.children.length === 0) {
      return `${parts.join(' ')} {}`;
    }
    const eol = endOfLine(context.endOfLine);
    const contents = index_js.pipe(
      index_js.map(context.print),
      joinBy(eol),
      indent(context.indentStyle, context.indentSize, eol),
      wrapByPair(`{${eol}`, `${eol}}`),
    )(ast.children);
    return `${parts.join(' ')} ${contents}`;
  },
};

const defaultPlugins$2 = [
  AttributeListPrintPlugin,
  AttributePrintPlugin,
  CommentPrintPlugin,
  DotPrintPlugin,
  EdgePrintPlugin,
  GraphPrintPlugin,
  LiteralPrintPlugin,
  NodePrintPlugin,
  NodeRefGroupPrintPlugin,
  NodeRefPrintPlugin,
  SubgraphPrintPlugin,
];

class Printer {
  options;
  #plugins = [...defaultPlugins$2];
  constructor(options = {}) {
    this.options = options;
  }
  print(ast) {
    const plugins = [...this.#plugins];
    const { indentSize = 2, indentStyle = 'space', endOfLine = 'lf' } = this.options;
    const context = {
      directed: true,
      indentSize,
      indentStyle,
      endOfLine,
      print(a) {
        for (const plugin of plugins) {
          if (plugin.match(a)) {
            return plugin.print(context, a);
          }
        }
        throw Error();
      },
    };
    return context.print(ast);
  }
}

function stringify(ast, options) {
  const result = new Printer(options).print(ast);
  if (!result) {
    throw new Error();
  }
  return result;
}

function peg$padEnd(str, targetLength, padString) {
  padString = padString || ' ';
  if (str.length > targetLength) {
    return str;
  }
  targetLength -= str.length;
  padString += padString.repeat(targetLength);
  return str + padString.slice(0, targetLength);
}
let DotSyntaxError$1 = class DotSyntaxError extends Error {
  static buildMessage(expected, found) {
    function hex(ch) {
      return ch.charCodeAt(0).toString(16).toUpperCase();
    }
    function literalEscape(s) {
      return s
        .replace(/\\/g, '\\\\')
        .replace(/"/g, '\\"')
        .replace(/\0/g, '\\0')
        .replace(/\t/g, '\\t')
        .replace(/\n/g, '\\n')
        .replace(/\r/g, '\\r')
        .replace(/[\x00-\x0F]/g, (ch) => '\\x0' + hex(ch))
        .replace(/[\x10-\x1F\x7F-\x9F]/g, (ch) => '\\x' + hex(ch));
    }
    function classEscape(s) {
      return s
        .replace(/\\/g, '\\\\')
        .replace(/\]/g, '\\]')
        .replace(/\^/g, '\\^')
        .replace(/-/g, '\\-')
        .replace(/\0/g, '\\0')
        .replace(/\t/g, '\\t')
        .replace(/\n/g, '\\n')
        .replace(/\r/g, '\\r')
        .replace(/[\x00-\x0F]/g, (ch) => '\\x0' + hex(ch))
        .replace(/[\x10-\x1F\x7F-\x9F]/g, (ch) => '\\x' + hex(ch));
    }
    function describeExpectation(expectation) {
      switch (expectation.type) {
        case 'literal':
          return '"' + literalEscape(expectation.text) + '"';
        case 'class':
          const escapedParts = expectation.parts.map((part) => {
            return Array.isArray(part) ? classEscape(part[0]) + '-' + classEscape(part[1]) : classEscape(part);
          });
          return '[' + (expectation.inverted ? '^' : '') + escapedParts + ']';
        case 'any':
          return 'any character';
        case 'end':
          return 'end of input';
        case 'other':
          return expectation.description;
      }
    }
    function describeExpected(expected1) {
      const descriptions = expected1.map(describeExpectation);
      let i;
      let j;
      descriptions.sort();
      if (descriptions.length > 0) {
        for (i = 1, j = 1; i < descriptions.length; i++) {
          if (descriptions[i - 1] !== descriptions[i]) {
            descriptions[j] = descriptions[i];
            j++;
          }
        }
        descriptions.length = j;
      }
      switch (descriptions.length) {
        case 1:
          return descriptions[0];
        case 2:
          return descriptions[0] + ' or ' + descriptions[1];
        default:
          return descriptions.slice(0, -1).join(', ') + ', or ' + descriptions[descriptions.length - 1];
      }
    }
    function describeFound(found1) {
      return found1 ? '"' + literalEscape(found1) + '"' : 'end of input';
    }
    return 'Expected ' + describeExpected(expected) + ' but ' + describeFound(found) + ' found.';
  }
  message;
  expected;
  found;
  location;
  name;
  constructor(message, expected, found, location) {
    super();
    this.message = message;
    this.expected = expected;
    this.found = found;
    this.location = location;
    this.name = 'DotSyntaxError';
    if (typeof Object.setPrototypeOf === 'function') {
      Object.setPrototypeOf(this, DotSyntaxError$1.prototype);
    } else {
      this.__proto__ = DotSyntaxError$1.prototype;
    }
    if (typeof Error.captureStackTrace === 'function') {
      Error.captureStackTrace(this, DotSyntaxError$1);
    }
  }
  format(sources) {
    let str = 'Error: ' + this.message;
    if (this.location) {
      let src = null;
      let k;
      for (k = 0; k < sources.length; k++) {
        if (sources[k].grammarSource === this.location.source) {
          src = sources[k].text.split(/\r\n|\n|\r/g);
          break;
        }
      }
      let s = this.location.start;
      let loc = this.location.source + ':' + s.line + ':' + s.column;
      if (src) {
        let e = this.location.end;
        let filler = peg$padEnd('', s.line.toString().length, ' ');
        let line = src[s.line - 1];
        let last = s.line === e.line ? e.column : line.length + 1;
        str +=
          '\n --> ' +
          loc +
          '\n' +
          filler +
          ' |\n' +
          s.line +
          ' | ' +
          line +
          '\n' +
          filler +
          ' | ' +
          peg$padEnd('', s.column - 1, ' ') +
          peg$padEnd('', last - s.column, '^');
      } else {
        str += '\n at ' + loc;
      }
    }
    return str;
  }
};
function peg$parse(input, options) {
  options = options !== undefined ? options : {};
  const peg$FAILED = {};
  const peg$source = options.grammarSource;
  const peg$startRuleFunctions = {
    Dot: peg$parseDot,
    Graph: peg$parseGraph,
    Subgraph: peg$parseSubgraph,
    Node: peg$parseNode,
    Edge: peg$parseEdge,
    AttributeList: peg$parseAttributeList,
    Attribute: peg$parseAttribute,
    ClusterStatements: peg$parseClusterStatements,
  };
  let peg$startRuleFunction = peg$parseDot;
  const peg$c0 = function (v) {
    return v;
  };
  const peg$c1 = function (c1, graph, c2) {
    return b.createElement('Dot', {}, [...c1, graph, ...c2]);
  };
  const peg$c2 = 'strict';
  const peg$c3 = peg$literalExpectation('strict', true);
  const peg$c4 = 'graph';
  const peg$c5 = peg$literalExpectation('graph', true);
  const peg$c6 = 'digraph';
  const peg$c7 = peg$literalExpectation('digraph', true);
  const peg$c8 = '{';
  const peg$c9 = peg$literalExpectation('{', false);
  const peg$c10 = '}';
  const peg$c11 = peg$literalExpectation('}', false);
  const peg$c12 = function (_strict, _kind, id, children) {
    const strict = !!_strict;
    const kind = _kind.toLowerCase();
    const directed = kind === 'digraph';
    for (const edgeop of edgeops) {
      if (directed) {
        if (edgeop.operator !== '->') {
          error(`In digraph, it's necessary to describe with "->" operator to create edge.`, edgeop.location);
        }
      } else {
        if (edgeop.operator !== '--') {
          error(`In graph, it's necessary to describe with "--" operator to create edge.`, edgeop.location);
        }
      }
    }
    return b.createElement(
      'Graph',
      id !== null
        ? {
            id,
            directed,
            strict,
          }
        : {
            directed,
            strict,
          },
      children,
    );
  };
  const peg$c13 = ';';
  const peg$c14 = peg$literalExpectation(';', false);
  const peg$c15 = function (keyValue) {
    return b.createElement(
      'Attribute',
      {
        ...keyValue,
      },
      [],
    );
  };
  const peg$c16 = 'node';
  const peg$c17 = peg$literalExpectation('node', true);
  const peg$c18 = 'edge';
  const peg$c19 = peg$literalExpectation('edge', true);
  const peg$c20 = function (_kind, children) {
    return b.createElement(
      'AttributeList',
      {
        kind: `${_kind.slice(0, 1).toUpperCase()}${_kind.slice(1).toLowerCase()}`,
      },
      children,
    );
  };
  const peg$c21 = function (id, rhs, _children) {
    return b.createElement(
      'Edge',
      {
        targets: [id, ...rhs],
      },
      _children ?? [],
    );
  };
  const peg$c22 = function (id, _children) {
    return b.createElement(
      'Node',
      {
        id,
      },
      _children ?? [],
    );
  };
  const peg$c23 = '=';
  const peg$c24 = peg$literalExpectation('=', false);
  const peg$c25 = function (key, value) {
    return { key, value };
  };
  const peg$c26 = ',';
  const peg$c27 = peg$literalExpectation(',', false);
  const peg$c28 = function (kv) {
    return b.createElement(
      'Attribute',
      {
        ...kv,
        location: location(),
      },
      [],
    );
  };
  const peg$c29 = '[';
  const peg$c30 = peg$literalExpectation('[', false);
  const peg$c31 = ']';
  const peg$c32 = peg$literalExpectation(']', false);
  const peg$c33 = function (list) {
    return list;
  };
  const peg$c34 = function (id, v) {
    return v;
  };
  const peg$c35 = function (id, rest) {
    return b.createElement('NodeRefGroup', {}, [id, ...rest]);
  };
  const peg$c36 = '->';
  const peg$c37 = peg$literalExpectation('->', false);
  const peg$c38 = '--';
  const peg$c39 = peg$literalExpectation('--', false);
  const peg$c40 = function (operator) {
    return { operator, location: location() };
  };
  const peg$c41 = function (edgeop, id, rest) {
    edgeops.push(edgeop);
    return [id].concat(rest || []);
  };
  const peg$c42 = function (id, port) {
    return b.createElement(
      'NodeRef',
      {
        id,
        ...port,
      },
      [],
    );
  };
  const peg$c43 = peg$otherExpectation('port');
  const peg$c44 = ':';
  const peg$c45 = peg$literalExpectation(':', false);
  const peg$c46 = function (port, compass) {
    return compass;
  };
  const peg$c47 = function (port, compass) {
    if (['n', 'ne', 'e', 'se', 's', 'sw', 'w', 'nw'].includes(port)) {
      return { compass: port };
    } else if (compass) {
      return { port, compass };
    }
    return { port };
  };
  const peg$c48 = 'subgraph';
  const peg$c49 = peg$literalExpectation('subgraph', true);
  const peg$c50 = function (id) {
    return id;
  };
  const peg$c51 = function (id, _children) {
    const children = _children ?? [];
    return b.createElement('Subgraph', id ? { id } : {}, children);
  };
  const peg$c52 = 'n';
  const peg$c53 = peg$literalExpectation('n', false);
  const peg$c54 = 'ne';
  const peg$c55 = peg$literalExpectation('ne', false);
  const peg$c56 = 'e';
  const peg$c57 = peg$literalExpectation('e', false);
  const peg$c58 = 'se';
  const peg$c59 = peg$literalExpectation('se', false);
  const peg$c60 = 's';
  const peg$c61 = peg$literalExpectation('s', false);
  const peg$c62 = 'sw';
  const peg$c63 = peg$literalExpectation('sw', false);
  const peg$c64 = 'w';
  const peg$c65 = peg$literalExpectation('w', false);
  const peg$c66 = 'nw';
  const peg$c67 = peg$literalExpectation('nw', false);
  const peg$c68 = function (value) {
    return { value, quoted: false };
  };
  const peg$c69 = '"';
  const peg$c70 = peg$literalExpectation('"', false);
  const peg$c71 = function (value) {
    return { value, quoted: true };
  };
  const peg$c72 = function (v) {
    return b.createElement(
      'Literal',
      {
        ...v,
      },
      [],
    );
  };
  const peg$c73 = function (value) {
    return b.createElement(
      'Literal',
      {
        value,
        quoted: false,
      },
      [],
    );
  };
  const peg$c74 = '/*';
  const peg$c75 = peg$literalExpectation('/*', false);
  const peg$c76 = '*/';
  const peg$c77 = peg$literalExpectation('*/', false);
  const peg$c78 = peg$anyExpectation();
  const peg$c79 = function (v) {
    return b.createElement(
      'Comment',
      {
        kind: 'Block',
        value: dedent(v.join('').replace(/[ \t]*\*/g, '')),
      },
      [],
    );
  };
  const peg$c80 = function (lines) {
    return b.createElement(
      'Comment',
      {
        kind: 'Slash',
        value: dedent(lines.join('\n')),
      },
      [],
    );
  };
  const peg$c81 = '//';
  const peg$c82 = peg$literalExpectation('//', false);
  const peg$c83 = function (v) {
    return v.join('');
  };
  const peg$c84 = function (lines) {
    return b.createElement(
      'Comment',
      {
        kind: 'Macro',
        value: dedent(lines.join('\n')),
      },
      [],
    );
  };
  const peg$c85 = '#';
  const peg$c86 = peg$literalExpectation('#', false);
  const peg$c87 = peg$otherExpectation('UNICODE_STRING');
  const peg$c88 = function (first, rest) {
    return first + rest.join('');
  };
  const peg$c89 = function (first, rest) {
    return first + rest;
  };
  const peg$c90 = '$';
  const peg$c91 = peg$literalExpectation('$', false);
  const peg$c92 = '_';
  const peg$c93 = peg$literalExpectation('_', false);
  const peg$c94 = peg$otherExpectation('NUMBER');
  const peg$c95 = '-';
  const peg$c96 = peg$literalExpectation('-', false);
  const peg$c97 = '.';
  const peg$c98 = peg$literalExpectation('.', false);
  const peg$c99 = /^[0-9]/;
  const peg$c100 = peg$classExpectation([['0', '9']], false, false);
  const peg$c101 = function (n) {
    return text();
  };
  const peg$c102 = function (v) {
    return b.createElement(
      'Literal',
      {
        value: v.slice(1, v.length - 1),
        quoted: 'html',
      },
      [],
    );
  };
  const peg$c103 = '<';
  const peg$c104 = peg$literalExpectation('<', false);
  const peg$c105 = '>';
  const peg$c106 = peg$literalExpectation('>', false);
  const peg$c107 = function (v) {
    return '<' + v.join('') + '>';
  };
  const peg$c108 = function (chars) {
    return b.createElement(
      'Literal',
      {
        value: chars.join(''),
        quoted: true,
      },
      [],
    );
  };
  const peg$c109 = function () {
    return text();
  };
  const peg$c110 = '\\';
  const peg$c111 = peg$literalExpectation('\\', false);
  const peg$c112 = function (v) {
    return v[1] === '"' ? '"' : v[0] + v[1];
  };
  const peg$c113 = function () {
    return '';
  };
  const peg$c114 = /^[\n\r\u2028\u2029]/;
  const peg$c115 = peg$classExpectation(['\n', '\r', '\u2028', '\u2029'], false, false);
  const peg$c116 = peg$otherExpectation('end of line');
  const peg$c117 = '\n';
  const peg$c118 = peg$literalExpectation('\n', false);
  const peg$c119 = '\r\n';
  const peg$c120 = peg$literalExpectation('\r\n', false);
  const peg$c121 = '\r';
  const peg$c122 = peg$literalExpectation('\r', false);
  const peg$c123 = '\u2028';
  const peg$c124 = peg$literalExpectation('\u2028', false);
  const peg$c125 = '\u2029';
  const peg$c126 = peg$literalExpectation('\u2029', false);
  const peg$c134 = peg$otherExpectation('whitespace');
  const peg$c135 = peg$otherExpectation('WHITESPACE');
  const peg$c136 = /^[\n\r]/;
  const peg$c137 = peg$classExpectation(['\n', '\r'], false, false);
  const peg$c138 = /^[ \t]/;
  const peg$c139 = peg$classExpectation([' ', '\t'], false, false);
  const peg$c140 =
    /^[a-z\xB5\xDF-\xF6\xF8-\xFF\u0101\u0103\u0105\u0107\u0109\u010B\u010D\u010F\u0111\u0113\u0115\u0117\u0119\u011B\u011D\u011F\u0121\u0123\u0125\u0127\u0129\u012B\u012D\u012F\u0131\u0133\u0135\u0137-\u0138\u013A\u013C\u013E\u0140\u0142\u0144\u0146\u0148-\u0149\u014B\u014D\u014F\u0151\u0153\u0155\u0157\u0159\u015B\u015D\u015F\u0161\u0163\u0165\u0167\u0169\u016B\u016D\u016F\u0171\u0173\u0175\u0177\u017A\u017C\u017E-\u0180\u0183\u0185\u0188\u018C-\u018D\u0192\u0195\u0199-\u019B\u019E\u01A1\u01A3\u01A5\u01A8\u01AA-\u01AB\u01AD\u01B0\u01B4\u01B6\u01B9-\u01BA\u01BD-\u01BF\u01C6\u01C9\u01CC\u01CE\u01D0\u01D2\u01D4\u01D6\u01D8\u01DA\u01DC-\u01DD\u01DF\u01E1\u01E3\u01E5\u01E7\u01E9\u01EB\u01ED\u01EF-\u01F0\u01F3\u01F5\u01F9\u01FB\u01FD\u01FF\u0201\u0203\u0205\u0207\u0209\u020B\u020D\u020F\u0211\u0213\u0215\u0217\u0219\u021B\u021D\u021F\u0221\u0223\u0225\u0227\u0229\u022B\u022D\u022F\u0231\u0233-\u0239\u023C\u023F-\u0240\u0242\u0247\u0249\u024B\u024D\u024F-\u0293\u0295-\u02AF\u0371\u0373\u0377\u037B-\u037D\u0390\u03AC-\u03CE\u03D0-\u03D1\u03D5-\u03D7\u03D9\u03DB\u03DD\u03DF\u03E1\u03E3\u03E5\u03E7\u03E9\u03EB\u03ED\u03EF-\u03F3\u03F5\u03F8\u03FB-\u03FC\u0430-\u045F\u0461\u0463\u0465\u0467\u0469\u046B\u046D\u046F\u0471\u0473\u0475\u0477\u0479\u047B\u047D\u047F\u0481\u048B\u048D\u048F\u0491\u0493\u0495\u0497\u0499\u049B\u049D\u049F\u04A1\u04A3\u04A5\u04A7\u04A9\u04AB\u04AD\u04AF\u04B1\u04B3\u04B5\u04B7\u04B9\u04BB\u04BD\u04BF\u04C2\u04C4\u04C6\u04C8\u04CA\u04CC\u04CE-\u04CF\u04D1\u04D3\u04D5\u04D7\u04D9\u04DB\u04DD\u04DF\u04E1\u04E3\u04E5\u04E7\u04E9\u04EB\u04ED\u04EF\u04F1\u04F3\u04F5\u04F7\u04F9\u04FB\u04FD\u04FF\u0501\u0503\u0505\u0507\u0509\u050B\u050D\u050F\u0511\u0513\u0515\u0517\u0519\u051B\u051D\u051F\u0521\u0523\u0525\u0527\u0561-\u0587\u1D00-\u1D2B\u1D6B-\u1D77\u1D79-\u1D9A\u1E01\u1E03\u1E05\u1E07\u1E09\u1E0B\u1E0D\u1E0F\u1E11\u1E13\u1E15\u1E17\u1E19\u1E1B\u1E1D\u1E1F\u1E21\u1E23\u1E25\u1E27\u1E29\u1E2B\u1E2D\u1E2F\u1E31\u1E33\u1E35\u1E37\u1E39\u1E3B\u1E3D\u1E3F\u1E41\u1E43\u1E45\u1E47\u1E49\u1E4B\u1E4D\u1E4F\u1E51\u1E53\u1E55\u1E57\u1E59\u1E5B\u1E5D\u1E5F\u1E61\u1E63\u1E65\u1E67\u1E69\u1E6B\u1E6D\u1E6F\u1E71\u1E73\u1E75\u1E77\u1E79\u1E7B\u1E7D\u1E7F\u1E81\u1E83\u1E85\u1E87\u1E89\u1E8B\u1E8D\u1E8F\u1E91\u1E93\u1E95-\u1E9D\u1E9F\u1EA1\u1EA3\u1EA5\u1EA7\u1EA9\u1EAB\u1EAD\u1EAF\u1EB1\u1EB3\u1EB5\u1EB7\u1EB9\u1EBB\u1EBD\u1EBF\u1EC1\u1EC3\u1EC5\u1EC7\u1EC9\u1ECB\u1ECD\u1ECF\u1ED1\u1ED3\u1ED5\u1ED7\u1ED9\u1EDB\u1EDD\u1EDF\u1EE1\u1EE3\u1EE5\u1EE7\u1EE9\u1EEB\u1EED\u1EEF\u1EF1\u1EF3\u1EF5\u1EF7\u1EF9\u1EFB\u1EFD\u1EFF-\u1F07\u1F10-\u1F15\u1F20-\u1F27\u1F30-\u1F37\u1F40-\u1F45\u1F50-\u1F57\u1F60-\u1F67\u1F70-\u1F7D\u1F80-\u1F87\u1F90-\u1F97\u1FA0-\u1FA7\u1FB0-\u1FB4\u1FB6-\u1FB7\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FC7\u1FD0-\u1FD3\u1FD6-\u1FD7\u1FE0-\u1FE7\u1FF2-\u1FF4\u1FF6-\u1FF7\u210A\u210E-\u210F\u2113\u212F\u2134\u2139\u213C-\u213D\u2146-\u2149\u214E\u2184\u2C30-\u2C5E\u2C61\u2C65-\u2C66\u2C68\u2C6A\u2C6C\u2C71\u2C73-\u2C74\u2C76-\u2C7B\u2C81\u2C83\u2C85\u2C87\u2C89\u2C8B\u2C8D\u2C8F\u2C91\u2C93\u2C95\u2C97\u2C99\u2C9B\u2C9D\u2C9F\u2CA1\u2CA3\u2CA5\u2CA7\u2CA9\u2CAB\u2CAD\u2CAF\u2CB1\u2CB3\u2CB5\u2CB7\u2CB9\u2CBB\u2CBD\u2CBF\u2CC1\u2CC3\u2CC5\u2CC7\u2CC9\u2CCB\u2CCD\u2CCF\u2CD1\u2CD3\u2CD5\u2CD7\u2CD9\u2CDB\u2CDD\u2CDF\u2CE1\u2CE3-\u2CE4\u2CEC\u2CEE\u2CF3\u2D00-\u2D25\u2D27\u2D2D\uA641\uA643\uA645\uA647\uA649\uA64B\uA64D\uA64F\uA651\uA653\uA655\uA657\uA659\uA65B\uA65D\uA65F\uA661\uA663\uA665\uA667\uA669\uA66B\uA66D\uA681\uA683\uA685\uA687\uA689\uA68B\uA68D\uA68F\uA691\uA693\uA695\uA697\uA723\uA725\uA727\uA729\uA72B\uA72D\uA72F-\uA731\uA733\uA735\uA737\uA739\uA73B\uA73D\uA73F\uA741\uA743\uA745\uA747\uA749\uA74B\uA74D\uA74F\uA751\uA753\uA755\uA757\uA759\uA75B\uA75D\uA75F\uA761\uA763\uA765\uA767\uA769\uA76B\uA76D\uA76F\uA771-\uA778\uA77A\uA77C\uA77F\uA781\uA783\uA785\uA787\uA78C\uA78E\uA791\uA793\uA7A1\uA7A3\uA7A5\uA7A7\uA7A9\uA7FA\uFB00-\uFB06\uFB13-\uFB17\uFF41-\uFF5A]/;
  const peg$c141 = peg$classExpectation(
    [
      ['a', 'z'],
      '\xB5',
      ['\xDF', '\xF6'],
      ['\xF8', '\xFF'],
      '\u0101',
      '\u0103',
      '\u0105',
      '\u0107',
      '\u0109',
      '\u010B',
      '\u010D',
      '\u010F',
      '\u0111',
      '\u0113',
      '\u0115',
      '\u0117',
      '\u0119',
      '\u011B',
      '\u011D',
      '\u011F',
      '\u0121',
      '\u0123',
      '\u0125',
      '\u0127',
      '\u0129',
      '\u012B',
      '\u012D',
      '\u012F',
      '\u0131',
      '\u0133',
      '\u0135',
      ['\u0137', '\u0138'],
      '\u013A',
      '\u013C',
      '\u013E',
      '\u0140',
      '\u0142',
      '\u0144',
      '\u0146',
      ['\u0148', '\u0149'],
      '\u014B',
      '\u014D',
      '\u014F',
      '\u0151',
      '\u0153',
      '\u0155',
      '\u0157',
      '\u0159',
      '\u015B',
      '\u015D',
      '\u015F',
      '\u0161',
      '\u0163',
      '\u0165',
      '\u0167',
      '\u0169',
      '\u016B',
      '\u016D',
      '\u016F',
      '\u0171',
      '\u0173',
      '\u0175',
      '\u0177',
      '\u017A',
      '\u017C',
      ['\u017E', '\u0180'],
      '\u0183',
      '\u0185',
      '\u0188',
      ['\u018C', '\u018D'],
      '\u0192',
      '\u0195',
      ['\u0199', '\u019B'],
      '\u019E',
      '\u01A1',
      '\u01A3',
      '\u01A5',
      '\u01A8',
      ['\u01AA', '\u01AB'],
      '\u01AD',
      '\u01B0',
      '\u01B4',
      '\u01B6',
      ['\u01B9', '\u01BA'],
      ['\u01BD', '\u01BF'],
      '\u01C6',
      '\u01C9',
      '\u01CC',
      '\u01CE',
      '\u01D0',
      '\u01D2',
      '\u01D4',
      '\u01D6',
      '\u01D8',
      '\u01DA',
      ['\u01DC', '\u01DD'],
      '\u01DF',
      '\u01E1',
      '\u01E3',
      '\u01E5',
      '\u01E7',
      '\u01E9',
      '\u01EB',
      '\u01ED',
      ['\u01EF', '\u01F0'],
      '\u01F3',
      '\u01F5',
      '\u01F9',
      '\u01FB',
      '\u01FD',
      '\u01FF',
      '\u0201',
      '\u0203',
      '\u0205',
      '\u0207',
      '\u0209',
      '\u020B',
      '\u020D',
      '\u020F',
      '\u0211',
      '\u0213',
      '\u0215',
      '\u0217',
      '\u0219',
      '\u021B',
      '\u021D',
      '\u021F',
      '\u0221',
      '\u0223',
      '\u0225',
      '\u0227',
      '\u0229',
      '\u022B',
      '\u022D',
      '\u022F',
      '\u0231',
      ['\u0233', '\u0239'],
      '\u023C',
      ['\u023F', '\u0240'],
      '\u0242',
      '\u0247',
      '\u0249',
      '\u024B',
      '\u024D',
      ['\u024F', '\u0293'],
      ['\u0295', '\u02AF'],
      '\u0371',
      '\u0373',
      '\u0377',
      ['\u037B', '\u037D'],
      '\u0390',
      ['\u03AC', '\u03CE'],
      ['\u03D0', '\u03D1'],
      ['\u03D5', '\u03D7'],
      '\u03D9',
      '\u03DB',
      '\u03DD',
      '\u03DF',
      '\u03E1',
      '\u03E3',
      '\u03E5',
      '\u03E7',
      '\u03E9',
      '\u03EB',
      '\u03ED',
      ['\u03EF', '\u03F3'],
      '\u03F5',
      '\u03F8',
      ['\u03FB', '\u03FC'],
      ['\u0430', '\u045F'],
      '\u0461',
      '\u0463',
      '\u0465',
      '\u0467',
      '\u0469',
      '\u046B',
      '\u046D',
      '\u046F',
      '\u0471',
      '\u0473',
      '\u0475',
      '\u0477',
      '\u0479',
      '\u047B',
      '\u047D',
      '\u047F',
      '\u0481',
      '\u048B',
      '\u048D',
      '\u048F',
      '\u0491',
      '\u0493',
      '\u0495',
      '\u0497',
      '\u0499',
      '\u049B',
      '\u049D',
      '\u049F',
      '\u04A1',
      '\u04A3',
      '\u04A5',
      '\u04A7',
      '\u04A9',
      '\u04AB',
      '\u04AD',
      '\u04AF',
      '\u04B1',
      '\u04B3',
      '\u04B5',
      '\u04B7',
      '\u04B9',
      '\u04BB',
      '\u04BD',
      '\u04BF',
      '\u04C2',
      '\u04C4',
      '\u04C6',
      '\u04C8',
      '\u04CA',
      '\u04CC',
      ['\u04CE', '\u04CF'],
      '\u04D1',
      '\u04D3',
      '\u04D5',
      '\u04D7',
      '\u04D9',
      '\u04DB',
      '\u04DD',
      '\u04DF',
      '\u04E1',
      '\u04E3',
      '\u04E5',
      '\u04E7',
      '\u04E9',
      '\u04EB',
      '\u04ED',
      '\u04EF',
      '\u04F1',
      '\u04F3',
      '\u04F5',
      '\u04F7',
      '\u04F9',
      '\u04FB',
      '\u04FD',
      '\u04FF',
      '\u0501',
      '\u0503',
      '\u0505',
      '\u0507',
      '\u0509',
      '\u050B',
      '\u050D',
      '\u050F',
      '\u0511',
      '\u0513',
      '\u0515',
      '\u0517',
      '\u0519',
      '\u051B',
      '\u051D',
      '\u051F',
      '\u0521',
      '\u0523',
      '\u0525',
      '\u0527',
      ['\u0561', '\u0587'],
      ['\u1D00', '\u1D2B'],
      ['\u1D6B', '\u1D77'],
      ['\u1D79', '\u1D9A'],
      '\u1E01',
      '\u1E03',
      '\u1E05',
      '\u1E07',
      '\u1E09',
      '\u1E0B',
      '\u1E0D',
      '\u1E0F',
      '\u1E11',
      '\u1E13',
      '\u1E15',
      '\u1E17',
      '\u1E19',
      '\u1E1B',
      '\u1E1D',
      '\u1E1F',
      '\u1E21',
      '\u1E23',
      '\u1E25',
      '\u1E27',
      '\u1E29',
      '\u1E2B',
      '\u1E2D',
      '\u1E2F',
      '\u1E31',
      '\u1E33',
      '\u1E35',
      '\u1E37',
      '\u1E39',
      '\u1E3B',
      '\u1E3D',
      '\u1E3F',
      '\u1E41',
      '\u1E43',
      '\u1E45',
      '\u1E47',
      '\u1E49',
      '\u1E4B',
      '\u1E4D',
      '\u1E4F',
      '\u1E51',
      '\u1E53',
      '\u1E55',
      '\u1E57',
      '\u1E59',
      '\u1E5B',
      '\u1E5D',
      '\u1E5F',
      '\u1E61',
      '\u1E63',
      '\u1E65',
      '\u1E67',
      '\u1E69',
      '\u1E6B',
      '\u1E6D',
      '\u1E6F',
      '\u1E71',
      '\u1E73',
      '\u1E75',
      '\u1E77',
      '\u1E79',
      '\u1E7B',
      '\u1E7D',
      '\u1E7F',
      '\u1E81',
      '\u1E83',
      '\u1E85',
      '\u1E87',
      '\u1E89',
      '\u1E8B',
      '\u1E8D',
      '\u1E8F',
      '\u1E91',
      '\u1E93',
      ['\u1E95', '\u1E9D'],
      '\u1E9F',
      '\u1EA1',
      '\u1EA3',
      '\u1EA5',
      '\u1EA7',
      '\u1EA9',
      '\u1EAB',
      '\u1EAD',
      '\u1EAF',
      '\u1EB1',
      '\u1EB3',
      '\u1EB5',
      '\u1EB7',
      '\u1EB9',
      '\u1EBB',
      '\u1EBD',
      '\u1EBF',
      '\u1EC1',
      '\u1EC3',
      '\u1EC5',
      '\u1EC7',
      '\u1EC9',
      '\u1ECB',
      '\u1ECD',
      '\u1ECF',
      '\u1ED1',
      '\u1ED3',
      '\u1ED5',
      '\u1ED7',
      '\u1ED9',
      '\u1EDB',
      '\u1EDD',
      '\u1EDF',
      '\u1EE1',
      '\u1EE3',
      '\u1EE5',
      '\u1EE7',
      '\u1EE9',
      '\u1EEB',
      '\u1EED',
      '\u1EEF',
      '\u1EF1',
      '\u1EF3',
      '\u1EF5',
      '\u1EF7',
      '\u1EF9',
      '\u1EFB',
      '\u1EFD',
      ['\u1EFF', '\u1F07'],
      ['\u1F10', '\u1F15'],
      ['\u1F20', '\u1F27'],
      ['\u1F30', '\u1F37'],
      ['\u1F40', '\u1F45'],
      ['\u1F50', '\u1F57'],
      ['\u1F60', '\u1F67'],
      ['\u1F70', '\u1F7D'],
      ['\u1F80', '\u1F87'],
      ['\u1F90', '\u1F97'],
      ['\u1FA0', '\u1FA7'],
      ['\u1FB0', '\u1FB4'],
      ['\u1FB6', '\u1FB7'],
      '\u1FBE',
      ['\u1FC2', '\u1FC4'],
      ['\u1FC6', '\u1FC7'],
      ['\u1FD0', '\u1FD3'],
      ['\u1FD6', '\u1FD7'],
      ['\u1FE0', '\u1FE7'],
      ['\u1FF2', '\u1FF4'],
      ['\u1FF6', '\u1FF7'],
      '\u210A',
      ['\u210E', '\u210F'],
      '\u2113',
      '\u212F',
      '\u2134',
      '\u2139',
      ['\u213C', '\u213D'],
      ['\u2146', '\u2149'],
      '\u214E',
      '\u2184',
      ['\u2C30', '\u2C5E'],
      '\u2C61',
      ['\u2C65', '\u2C66'],
      '\u2C68',
      '\u2C6A',
      '\u2C6C',
      '\u2C71',
      ['\u2C73', '\u2C74'],
      ['\u2C76', '\u2C7B'],
      '\u2C81',
      '\u2C83',
      '\u2C85',
      '\u2C87',
      '\u2C89',
      '\u2C8B',
      '\u2C8D',
      '\u2C8F',
      '\u2C91',
      '\u2C93',
      '\u2C95',
      '\u2C97',
      '\u2C99',
      '\u2C9B',
      '\u2C9D',
      '\u2C9F',
      '\u2CA1',
      '\u2CA3',
      '\u2CA5',
      '\u2CA7',
      '\u2CA9',
      '\u2CAB',
      '\u2CAD',
      '\u2CAF',
      '\u2CB1',
      '\u2CB3',
      '\u2CB5',
      '\u2CB7',
      '\u2CB9',
      '\u2CBB',
      '\u2CBD',
      '\u2CBF',
      '\u2CC1',
      '\u2CC3',
      '\u2CC5',
      '\u2CC7',
      '\u2CC9',
      '\u2CCB',
      '\u2CCD',
      '\u2CCF',
      '\u2CD1',
      '\u2CD3',
      '\u2CD5',
      '\u2CD7',
      '\u2CD9',
      '\u2CDB',
      '\u2CDD',
      '\u2CDF',
      '\u2CE1',
      ['\u2CE3', '\u2CE4'],
      '\u2CEC',
      '\u2CEE',
      '\u2CF3',
      ['\u2D00', '\u2D25'],
      '\u2D27',
      '\u2D2D',
      '\uA641',
      '\uA643',
      '\uA645',
      '\uA647',
      '\uA649',
      '\uA64B',
      '\uA64D',
      '\uA64F',
      '\uA651',
      '\uA653',
      '\uA655',
      '\uA657',
      '\uA659',
      '\uA65B',
      '\uA65D',
      '\uA65F',
      '\uA661',
      '\uA663',
      '\uA665',
      '\uA667',
      '\uA669',
      '\uA66B',
      '\uA66D',
      '\uA681',
      '\uA683',
      '\uA685',
      '\uA687',
      '\uA689',
      '\uA68B',
      '\uA68D',
      '\uA68F',
      '\uA691',
      '\uA693',
      '\uA695',
      '\uA697',
      '\uA723',
      '\uA725',
      '\uA727',
      '\uA729',
      '\uA72B',
      '\uA72D',
      ['\uA72F', '\uA731'],
      '\uA733',
      '\uA735',
      '\uA737',
      '\uA739',
      '\uA73B',
      '\uA73D',
      '\uA73F',
      '\uA741',
      '\uA743',
      '\uA745',
      '\uA747',
      '\uA749',
      '\uA74B',
      '\uA74D',
      '\uA74F',
      '\uA751',
      '\uA753',
      '\uA755',
      '\uA757',
      '\uA759',
      '\uA75B',
      '\uA75D',
      '\uA75F',
      '\uA761',
      '\uA763',
      '\uA765',
      '\uA767',
      '\uA769',
      '\uA76B',
      '\uA76D',
      '\uA76F',
      ['\uA771', '\uA778'],
      '\uA77A',
      '\uA77C',
      '\uA77F',
      '\uA781',
      '\uA783',
      '\uA785',
      '\uA787',
      '\uA78C',
      '\uA78E',
      '\uA791',
      '\uA793',
      '\uA7A1',
      '\uA7A3',
      '\uA7A5',
      '\uA7A7',
      '\uA7A9',
      '\uA7FA',
      ['\uFB00', '\uFB06'],
      ['\uFB13', '\uFB17'],
      ['\uFF41', '\uFF5A'],
    ],
    false,
    false,
  );
  const peg$c142 =
    /^[\u02B0-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0374\u037A\u0559\u0640\u06E5-\u06E6\u07F4-\u07F5\u07FA\u081A\u0824\u0828\u0971\u0E46\u0EC6\u10FC\u17D7\u1843\u1AA7\u1C78-\u1C7D\u1D2C-\u1D6A\u1D78\u1D9B-\u1DBF\u2071\u207F\u2090-\u209C\u2C7C-\u2C7D\u2D6F\u2E2F\u3005\u3031-\u3035\u303B\u309D-\u309E\u30FC-\u30FE\uA015\uA4F8-\uA4FD\uA60C\uA67F\uA717-\uA71F\uA770\uA788\uA7F8-\uA7F9\uA9CF\uAA70\uAADD\uAAF3-\uAAF4\uFF70\uFF9E-\uFF9F]/;
  const peg$c143 = peg$classExpectation(
    [
      ['\u02B0', '\u02C1'],
      ['\u02C6', '\u02D1'],
      ['\u02E0', '\u02E4'],
      '\u02EC',
      '\u02EE',
      '\u0374',
      '\u037A',
      '\u0559',
      '\u0640',
      ['\u06E5', '\u06E6'],
      ['\u07F4', '\u07F5'],
      '\u07FA',
      '\u081A',
      '\u0824',
      '\u0828',
      '\u0971',
      '\u0E46',
      '\u0EC6',
      '\u10FC',
      '\u17D7',
      '\u1843',
      '\u1AA7',
      ['\u1C78', '\u1C7D'],
      ['\u1D2C', '\u1D6A'],
      '\u1D78',
      ['\u1D9B', '\u1DBF'],
      '\u2071',
      '\u207F',
      ['\u2090', '\u209C'],
      ['\u2C7C', '\u2C7D'],
      '\u2D6F',
      '\u2E2F',
      '\u3005',
      ['\u3031', '\u3035'],
      '\u303B',
      ['\u309D', '\u309E'],
      ['\u30FC', '\u30FE'],
      '\uA015',
      ['\uA4F8', '\uA4FD'],
      '\uA60C',
      '\uA67F',
      ['\uA717', '\uA71F'],
      '\uA770',
      '\uA788',
      ['\uA7F8', '\uA7F9'],
      '\uA9CF',
      '\uAA70',
      '\uAADD',
      ['\uAAF3', '\uAAF4'],
      '\uFF70',
      ['\uFF9E', '\uFF9F'],
    ],
    false,
    false,
  );
  const peg$c144 =
    /^[\xAA\xBA\u01BB\u01C0-\u01C3\u0294\u05D0-\u05EA\u05F0-\u05F2\u0620-\u063F\u0641-\u064A\u066E-\u066F\u0671-\u06D3\u06D5\u06EE-\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u0800-\u0815\u0840-\u0858\u08A0\u08A2-\u08AC\u0904-\u0939\u093D\u0950\u0958-\u0961\u0972-\u0977\u0979-\u097F\u0985-\u098C\u098F-\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC-\u09DD\u09DF-\u09E1\u09F0-\u09F1\u0A05-\u0A0A\u0A0F-\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32-\u0A33\u0A35-\u0A36\u0A38-\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2-\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0-\u0AE1\u0B05-\u0B0C\u0B0F-\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32-\u0B33\u0B35-\u0B39\u0B3D\u0B5C-\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99-\u0B9A\u0B9C\u0B9E-\u0B9F\u0BA3-\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58-\u0C59\u0C60-\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0-\u0CE1\u0CF1-\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D60-\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32-\u0E33\u0E40-\u0E45\u0E81-\u0E82\u0E84\u0E87-\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA-\u0EAB\u0EAD-\u0EB0\u0EB2-\u0EB3\u0EBD\u0EC0-\u0EC4\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065-\u1066\u106E-\u1070\u1075-\u1081\u108E\u10D0-\u10FA\u10FD-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17DC\u1820-\u1842\u1844-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE-\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C77\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5-\u1CF6\u2135-\u2138\u2D30-\u2D67\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u3006\u303C\u3041-\u3096\u309F\u30A1-\u30FA\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA014\uA016-\uA48C\uA4D0-\uA4F7\uA500-\uA60B\uA610-\uA61F\uA62A-\uA62B\uA66E\uA6A0-\uA6E5\uA7FB-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA6F\uAA71-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5-\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADC\uAAE0-\uAAEA\uAAF2\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40-\uFB41\uFB43-\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF66-\uFF6F\uFF71-\uFF9D\uFFA0-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]/;
  const peg$c145 = peg$classExpectation(
    [
      '\xAA',
      '\xBA',
      '\u01BB',
      ['\u01C0', '\u01C3'],
      '\u0294',
      ['\u05D0', '\u05EA'],
      ['\u05F0', '\u05F2'],
      ['\u0620', '\u063F'],
      ['\u0641', '\u064A'],
      ['\u066E', '\u066F'],
      ['\u0671', '\u06D3'],
      '\u06D5',
      ['\u06EE', '\u06EF'],
      ['\u06FA', '\u06FC'],
      '\u06FF',
      '\u0710',
      ['\u0712', '\u072F'],
      ['\u074D', '\u07A5'],
      '\u07B1',
      ['\u07CA', '\u07EA'],
      ['\u0800', '\u0815'],
      ['\u0840', '\u0858'],
      '\u08A0',
      ['\u08A2', '\u08AC'],
      ['\u0904', '\u0939'],
      '\u093D',
      '\u0950',
      ['\u0958', '\u0961'],
      ['\u0972', '\u0977'],
      ['\u0979', '\u097F'],
      ['\u0985', '\u098C'],
      ['\u098F', '\u0990'],
      ['\u0993', '\u09A8'],
      ['\u09AA', '\u09B0'],
      '\u09B2',
      ['\u09B6', '\u09B9'],
      '\u09BD',
      '\u09CE',
      ['\u09DC', '\u09DD'],
      ['\u09DF', '\u09E1'],
      ['\u09F0', '\u09F1'],
      ['\u0A05', '\u0A0A'],
      ['\u0A0F', '\u0A10'],
      ['\u0A13', '\u0A28'],
      ['\u0A2A', '\u0A30'],
      ['\u0A32', '\u0A33'],
      ['\u0A35', '\u0A36'],
      ['\u0A38', '\u0A39'],
      ['\u0A59', '\u0A5C'],
      '\u0A5E',
      ['\u0A72', '\u0A74'],
      ['\u0A85', '\u0A8D'],
      ['\u0A8F', '\u0A91'],
      ['\u0A93', '\u0AA8'],
      ['\u0AAA', '\u0AB0'],
      ['\u0AB2', '\u0AB3'],
      ['\u0AB5', '\u0AB9'],
      '\u0ABD',
      '\u0AD0',
      ['\u0AE0', '\u0AE1'],
      ['\u0B05', '\u0B0C'],
      ['\u0B0F', '\u0B10'],
      ['\u0B13', '\u0B28'],
      ['\u0B2A', '\u0B30'],
      ['\u0B32', '\u0B33'],
      ['\u0B35', '\u0B39'],
      '\u0B3D',
      ['\u0B5C', '\u0B5D'],
      ['\u0B5F', '\u0B61'],
      '\u0B71',
      '\u0B83',
      ['\u0B85', '\u0B8A'],
      ['\u0B8E', '\u0B90'],
      ['\u0B92', '\u0B95'],
      ['\u0B99', '\u0B9A'],
      '\u0B9C',
      ['\u0B9E', '\u0B9F'],
      ['\u0BA3', '\u0BA4'],
      ['\u0BA8', '\u0BAA'],
      ['\u0BAE', '\u0BB9'],
      '\u0BD0',
      ['\u0C05', '\u0C0C'],
      ['\u0C0E', '\u0C10'],
      ['\u0C12', '\u0C28'],
      ['\u0C2A', '\u0C33'],
      ['\u0C35', '\u0C39'],
      '\u0C3D',
      ['\u0C58', '\u0C59'],
      ['\u0C60', '\u0C61'],
      ['\u0C85', '\u0C8C'],
      ['\u0C8E', '\u0C90'],
      ['\u0C92', '\u0CA8'],
      ['\u0CAA', '\u0CB3'],
      ['\u0CB5', '\u0CB9'],
      '\u0CBD',
      '\u0CDE',
      ['\u0CE0', '\u0CE1'],
      ['\u0CF1', '\u0CF2'],
      ['\u0D05', '\u0D0C'],
      ['\u0D0E', '\u0D10'],
      ['\u0D12', '\u0D3A'],
      '\u0D3D',
      '\u0D4E',
      ['\u0D60', '\u0D61'],
      ['\u0D7A', '\u0D7F'],
      ['\u0D85', '\u0D96'],
      ['\u0D9A', '\u0DB1'],
      ['\u0DB3', '\u0DBB'],
      '\u0DBD',
      ['\u0DC0', '\u0DC6'],
      ['\u0E01', '\u0E30'],
      ['\u0E32', '\u0E33'],
      ['\u0E40', '\u0E45'],
      ['\u0E81', '\u0E82'],
      '\u0E84',
      ['\u0E87', '\u0E88'],
      '\u0E8A',
      '\u0E8D',
      ['\u0E94', '\u0E97'],
      ['\u0E99', '\u0E9F'],
      ['\u0EA1', '\u0EA3'],
      '\u0EA5',
      '\u0EA7',
      ['\u0EAA', '\u0EAB'],
      ['\u0EAD', '\u0EB0'],
      ['\u0EB2', '\u0EB3'],
      '\u0EBD',
      ['\u0EC0', '\u0EC4'],
      ['\u0EDC', '\u0EDF'],
      '\u0F00',
      ['\u0F40', '\u0F47'],
      ['\u0F49', '\u0F6C'],
      ['\u0F88', '\u0F8C'],
      ['\u1000', '\u102A'],
      '\u103F',
      ['\u1050', '\u1055'],
      ['\u105A', '\u105D'],
      '\u1061',
      ['\u1065', '\u1066'],
      ['\u106E', '\u1070'],
      ['\u1075', '\u1081'],
      '\u108E',
      ['\u10D0', '\u10FA'],
      ['\u10FD', '\u1248'],
      ['\u124A', '\u124D'],
      ['\u1250', '\u1256'],
      '\u1258',
      ['\u125A', '\u125D'],
      ['\u1260', '\u1288'],
      ['\u128A', '\u128D'],
      ['\u1290', '\u12B0'],
      ['\u12B2', '\u12B5'],
      ['\u12B8', '\u12BE'],
      '\u12C0',
      ['\u12C2', '\u12C5'],
      ['\u12C8', '\u12D6'],
      ['\u12D8', '\u1310'],
      ['\u1312', '\u1315'],
      ['\u1318', '\u135A'],
      ['\u1380', '\u138F'],
      ['\u13A0', '\u13F4'],
      ['\u1401', '\u166C'],
      ['\u166F', '\u167F'],
      ['\u1681', '\u169A'],
      ['\u16A0', '\u16EA'],
      ['\u1700', '\u170C'],
      ['\u170E', '\u1711'],
      ['\u1720', '\u1731'],
      ['\u1740', '\u1751'],
      ['\u1760', '\u176C'],
      ['\u176E', '\u1770'],
      ['\u1780', '\u17B3'],
      '\u17DC',
      ['\u1820', '\u1842'],
      ['\u1844', '\u1877'],
      ['\u1880', '\u18A8'],
      '\u18AA',
      ['\u18B0', '\u18F5'],
      ['\u1900', '\u191C'],
      ['\u1950', '\u196D'],
      ['\u1970', '\u1974'],
      ['\u1980', '\u19AB'],
      ['\u19C1', '\u19C7'],
      ['\u1A00', '\u1A16'],
      ['\u1A20', '\u1A54'],
      ['\u1B05', '\u1B33'],
      ['\u1B45', '\u1B4B'],
      ['\u1B83', '\u1BA0'],
      ['\u1BAE', '\u1BAF'],
      ['\u1BBA', '\u1BE5'],
      ['\u1C00', '\u1C23'],
      ['\u1C4D', '\u1C4F'],
      ['\u1C5A', '\u1C77'],
      ['\u1CE9', '\u1CEC'],
      ['\u1CEE', '\u1CF1'],
      ['\u1CF5', '\u1CF6'],
      ['\u2135', '\u2138'],
      ['\u2D30', '\u2D67'],
      ['\u2D80', '\u2D96'],
      ['\u2DA0', '\u2DA6'],
      ['\u2DA8', '\u2DAE'],
      ['\u2DB0', '\u2DB6'],
      ['\u2DB8', '\u2DBE'],
      ['\u2DC0', '\u2DC6'],
      ['\u2DC8', '\u2DCE'],
      ['\u2DD0', '\u2DD6'],
      ['\u2DD8', '\u2DDE'],
      '\u3006',
      '\u303C',
      ['\u3041', '\u3096'],
      '\u309F',
      ['\u30A1', '\u30FA'],
      '\u30FF',
      ['\u3105', '\u312D'],
      ['\u3131', '\u318E'],
      ['\u31A0', '\u31BA'],
      ['\u31F0', '\u31FF'],
      ['\u3400', '\u4DB5'],
      ['\u4E00', '\u9FCC'],
      ['\uA000', '\uA014'],
      ['\uA016', '\uA48C'],
      ['\uA4D0', '\uA4F7'],
      ['\uA500', '\uA60B'],
      ['\uA610', '\uA61F'],
      ['\uA62A', '\uA62B'],
      '\uA66E',
      ['\uA6A0', '\uA6E5'],
      ['\uA7FB', '\uA801'],
      ['\uA803', '\uA805'],
      ['\uA807', '\uA80A'],
      ['\uA80C', '\uA822'],
      ['\uA840', '\uA873'],
      ['\uA882', '\uA8B3'],
      ['\uA8F2', '\uA8F7'],
      '\uA8FB',
      ['\uA90A', '\uA925'],
      ['\uA930', '\uA946'],
      ['\uA960', '\uA97C'],
      ['\uA984', '\uA9B2'],
      ['\uAA00', '\uAA28'],
      ['\uAA40', '\uAA42'],
      ['\uAA44', '\uAA4B'],
      ['\uAA60', '\uAA6F'],
      ['\uAA71', '\uAA76'],
      '\uAA7A',
      ['\uAA80', '\uAAAF'],
      '\uAAB1',
      ['\uAAB5', '\uAAB6'],
      ['\uAAB9', '\uAABD'],
      '\uAAC0',
      '\uAAC2',
      ['\uAADB', '\uAADC'],
      ['\uAAE0', '\uAAEA'],
      '\uAAF2',
      ['\uAB01', '\uAB06'],
      ['\uAB09', '\uAB0E'],
      ['\uAB11', '\uAB16'],
      ['\uAB20', '\uAB26'],
      ['\uAB28', '\uAB2E'],
      ['\uABC0', '\uABE2'],
      ['\uAC00', '\uD7A3'],
      ['\uD7B0', '\uD7C6'],
      ['\uD7CB', '\uD7FB'],
      ['\uF900', '\uFA6D'],
      ['\uFA70', '\uFAD9'],
      '\uFB1D',
      ['\uFB1F', '\uFB28'],
      ['\uFB2A', '\uFB36'],
      ['\uFB38', '\uFB3C'],
      '\uFB3E',
      ['\uFB40', '\uFB41'],
      ['\uFB43', '\uFB44'],
      ['\uFB46', '\uFBB1'],
      ['\uFBD3', '\uFD3D'],
      ['\uFD50', '\uFD8F'],
      ['\uFD92', '\uFDC7'],
      ['\uFDF0', '\uFDFB'],
      ['\uFE70', '\uFE74'],
      ['\uFE76', '\uFEFC'],
      ['\uFF66', '\uFF6F'],
      ['\uFF71', '\uFF9D'],
      ['\uFFA0', '\uFFBE'],
      ['\uFFC2', '\uFFC7'],
      ['\uFFCA', '\uFFCF'],
      ['\uFFD2', '\uFFD7'],
      ['\uFFDA', '\uFFDC'],
    ],
    false,
    false,
  );
  const peg$c146 = /^[\u01C5\u01C8\u01CB\u01F2\u1F88-\u1F8F\u1F98-\u1F9F\u1FA8-\u1FAF\u1FBC\u1FCC\u1FFC]/;
  const peg$c147 = peg$classExpectation(
    [
      '\u01C5',
      '\u01C8',
      '\u01CB',
      '\u01F2',
      ['\u1F88', '\u1F8F'],
      ['\u1F98', '\u1F9F'],
      ['\u1FA8', '\u1FAF'],
      '\u1FBC',
      '\u1FCC',
      '\u1FFC',
    ],
    false,
    false,
  );
  const peg$c148 =
    /^[A-Z\xC0-\xD6\xD8-\xDE\u0100\u0102\u0104\u0106\u0108\u010A\u010C\u010E\u0110\u0112\u0114\u0116\u0118\u011A\u011C\u011E\u0120\u0122\u0124\u0126\u0128\u012A\u012C\u012E\u0130\u0132\u0134\u0136\u0139\u013B\u013D\u013F\u0141\u0143\u0145\u0147\u014A\u014C\u014E\u0150\u0152\u0154\u0156\u0158\u015A\u015C\u015E\u0160\u0162\u0164\u0166\u0168\u016A\u016C\u016E\u0170\u0172\u0174\u0176\u0178-\u0179\u017B\u017D\u0181-\u0182\u0184\u0186-\u0187\u0189-\u018B\u018E-\u0191\u0193-\u0194\u0196-\u0198\u019C-\u019D\u019F-\u01A0\u01A2\u01A4\u01A6-\u01A7\u01A9\u01AC\u01AE-\u01AF\u01B1-\u01B3\u01B5\u01B7-\u01B8\u01BC\u01C4\u01C7\u01CA\u01CD\u01CF\u01D1\u01D3\u01D5\u01D7\u01D9\u01DB\u01DE\u01E0\u01E2\u01E4\u01E6\u01E8\u01EA\u01EC\u01EE\u01F1\u01F4\u01F6-\u01F8\u01FA\u01FC\u01FE\u0200\u0202\u0204\u0206\u0208\u020A\u020C\u020E\u0210\u0212\u0214\u0216\u0218\u021A\u021C\u021E\u0220\u0222\u0224\u0226\u0228\u022A\u022C\u022E\u0230\u0232\u023A-\u023B\u023D-\u023E\u0241\u0243-\u0246\u0248\u024A\u024C\u024E\u0370\u0372\u0376\u0386\u0388-\u038A\u038C\u038E-\u038F\u0391-\u03A1\u03A3-\u03AB\u03CF\u03D2-\u03D4\u03D8\u03DA\u03DC\u03DE\u03E0\u03E2\u03E4\u03E6\u03E8\u03EA\u03EC\u03EE\u03F4\u03F7\u03F9-\u03FA\u03FD-\u042F\u0460\u0462\u0464\u0466\u0468\u046A\u046C\u046E\u0470\u0472\u0474\u0476\u0478\u047A\u047C\u047E\u0480\u048A\u048C\u048E\u0490\u0492\u0494\u0496\u0498\u049A\u049C\u049E\u04A0\u04A2\u04A4\u04A6\u04A8\u04AA\u04AC\u04AE\u04B0\u04B2\u04B4\u04B6\u04B8\u04BA\u04BC\u04BE\u04C0-\u04C1\u04C3\u04C5\u04C7\u04C9\u04CB\u04CD\u04D0\u04D2\u04D4\u04D6\u04D8\u04DA\u04DC\u04DE\u04E0\u04E2\u04E4\u04E6\u04E8\u04EA\u04EC\u04EE\u04F0\u04F2\u04F4\u04F6\u04F8\u04FA\u04FC\u04FE\u0500\u0502\u0504\u0506\u0508\u050A\u050C\u050E\u0510\u0512\u0514\u0516\u0518\u051A\u051C\u051E\u0520\u0522\u0524\u0526\u0531-\u0556\u10A0-\u10C5\u10C7\u10CD\u1E00\u1E02\u1E04\u1E06\u1E08\u1E0A\u1E0C\u1E0E\u1E10\u1E12\u1E14\u1E16\u1E18\u1E1A\u1E1C\u1E1E\u1E20\u1E22\u1E24\u1E26\u1E28\u1E2A\u1E2C\u1E2E\u1E30\u1E32\u1E34\u1E36\u1E38\u1E3A\u1E3C\u1E3E\u1E40\u1E42\u1E44\u1E46\u1E48\u1E4A\u1E4C\u1E4E\u1E50\u1E52\u1E54\u1E56\u1E58\u1E5A\u1E5C\u1E5E\u1E60\u1E62\u1E64\u1E66\u1E68\u1E6A\u1E6C\u1E6E\u1E70\u1E72\u1E74\u1E76\u1E78\u1E7A\u1E7C\u1E7E\u1E80\u1E82\u1E84\u1E86\u1E88\u1E8A\u1E8C\u1E8E\u1E90\u1E92\u1E94\u1E9E\u1EA0\u1EA2\u1EA4\u1EA6\u1EA8\u1EAA\u1EAC\u1EAE\u1EB0\u1EB2\u1EB4\u1EB6\u1EB8\u1EBA\u1EBC\u1EBE\u1EC0\u1EC2\u1EC4\u1EC6\u1EC8\u1ECA\u1ECC\u1ECE\u1ED0\u1ED2\u1ED4\u1ED6\u1ED8\u1EDA\u1EDC\u1EDE\u1EE0\u1EE2\u1EE4\u1EE6\u1EE8\u1EEA\u1EEC\u1EEE\u1EF0\u1EF2\u1EF4\u1EF6\u1EF8\u1EFA\u1EFC\u1EFE\u1F08-\u1F0F\u1F18-\u1F1D\u1F28-\u1F2F\u1F38-\u1F3F\u1F48-\u1F4D\u1F59\u1F5B\u1F5D\u1F5F\u1F68-\u1F6F\u1FB8-\u1FBB\u1FC8-\u1FCB\u1FD8-\u1FDB\u1FE8-\u1FEC\u1FF8-\u1FFB\u2102\u2107\u210B-\u210D\u2110-\u2112\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u2130-\u2133\u213E-\u213F\u2145\u2183\u2C00-\u2C2E\u2C60\u2C62-\u2C64\u2C67\u2C69\u2C6B\u2C6D-\u2C70\u2C72\u2C75\u2C7E-\u2C80\u2C82\u2C84\u2C86\u2C88\u2C8A\u2C8C\u2C8E\u2C90\u2C92\u2C94\u2C96\u2C98\u2C9A\u2C9C\u2C9E\u2CA0\u2CA2\u2CA4\u2CA6\u2CA8\u2CAA\u2CAC\u2CAE\u2CB0\u2CB2\u2CB4\u2CB6\u2CB8\u2CBA\u2CBC\u2CBE\u2CC0\u2CC2\u2CC4\u2CC6\u2CC8\u2CCA\u2CCC\u2CCE\u2CD0\u2CD2\u2CD4\u2CD6\u2CD8\u2CDA\u2CDC\u2CDE\u2CE0\u2CE2\u2CEB\u2CED\u2CF2\uA640\uA642\uA644\uA646\uA648\uA64A\uA64C\uA64E\uA650\uA652\uA654\uA656\uA658\uA65A\uA65C\uA65E\uA660\uA662\uA664\uA666\uA668\uA66A\uA66C\uA680\uA682\uA684\uA686\uA688\uA68A\uA68C\uA68E\uA690\uA692\uA694\uA696\uA722\uA724\uA726\uA728\uA72A\uA72C\uA72E\uA732\uA734\uA736\uA738\uA73A\uA73C\uA73E\uA740\uA742\uA744\uA746\uA748\uA74A\uA74C\uA74E\uA750\uA752\uA754\uA756\uA758\uA75A\uA75C\uA75E\uA760\uA762\uA764\uA766\uA768\uA76A\uA76C\uA76E\uA779\uA77B\uA77D-\uA77E\uA780\uA782\uA784\uA786\uA78B\uA78D\uA790\uA792\uA7A0\uA7A2\uA7A4\uA7A6\uA7A8\uA7AA\uFF21-\uFF3A]/;
  const peg$c149 = peg$classExpectation(
    [
      ['A', 'Z'],
      ['\xC0', '\xD6'],
      ['\xD8', '\xDE'],
      '\u0100',
      '\u0102',
      '\u0104',
      '\u0106',
      '\u0108',
      '\u010A',
      '\u010C',
      '\u010E',
      '\u0110',
      '\u0112',
      '\u0114',
      '\u0116',
      '\u0118',
      '\u011A',
      '\u011C',
      '\u011E',
      '\u0120',
      '\u0122',
      '\u0124',
      '\u0126',
      '\u0128',
      '\u012A',
      '\u012C',
      '\u012E',
      '\u0130',
      '\u0132',
      '\u0134',
      '\u0136',
      '\u0139',
      '\u013B',
      '\u013D',
      '\u013F',
      '\u0141',
      '\u0143',
      '\u0145',
      '\u0147',
      '\u014A',
      '\u014C',
      '\u014E',
      '\u0150',
      '\u0152',
      '\u0154',
      '\u0156',
      '\u0158',
      '\u015A',
      '\u015C',
      '\u015E',
      '\u0160',
      '\u0162',
      '\u0164',
      '\u0166',
      '\u0168',
      '\u016A',
      '\u016C',
      '\u016E',
      '\u0170',
      '\u0172',
      '\u0174',
      '\u0176',
      ['\u0178', '\u0179'],
      '\u017B',
      '\u017D',
      ['\u0181', '\u0182'],
      '\u0184',
      ['\u0186', '\u0187'],
      ['\u0189', '\u018B'],
      ['\u018E', '\u0191'],
      ['\u0193', '\u0194'],
      ['\u0196', '\u0198'],
      ['\u019C', '\u019D'],
      ['\u019F', '\u01A0'],
      '\u01A2',
      '\u01A4',
      ['\u01A6', '\u01A7'],
      '\u01A9',
      '\u01AC',
      ['\u01AE', '\u01AF'],
      ['\u01B1', '\u01B3'],
      '\u01B5',
      ['\u01B7', '\u01B8'],
      '\u01BC',
      '\u01C4',
      '\u01C7',
      '\u01CA',
      '\u01CD',
      '\u01CF',
      '\u01D1',
      '\u01D3',
      '\u01D5',
      '\u01D7',
      '\u01D9',
      '\u01DB',
      '\u01DE',
      '\u01E0',
      '\u01E2',
      '\u01E4',
      '\u01E6',
      '\u01E8',
      '\u01EA',
      '\u01EC',
      '\u01EE',
      '\u01F1',
      '\u01F4',
      ['\u01F6', '\u01F8'],
      '\u01FA',
      '\u01FC',
      '\u01FE',
      '\u0200',
      '\u0202',
      '\u0204',
      '\u0206',
      '\u0208',
      '\u020A',
      '\u020C',
      '\u020E',
      '\u0210',
      '\u0212',
      '\u0214',
      '\u0216',
      '\u0218',
      '\u021A',
      '\u021C',
      '\u021E',
      '\u0220',
      '\u0222',
      '\u0224',
      '\u0226',
      '\u0228',
      '\u022A',
      '\u022C',
      '\u022E',
      '\u0230',
      '\u0232',
      ['\u023A', '\u023B'],
      ['\u023D', '\u023E'],
      '\u0241',
      ['\u0243', '\u0246'],
      '\u0248',
      '\u024A',
      '\u024C',
      '\u024E',
      '\u0370',
      '\u0372',
      '\u0376',
      '\u0386',
      ['\u0388', '\u038A'],
      '\u038C',
      ['\u038E', '\u038F'],
      ['\u0391', '\u03A1'],
      ['\u03A3', '\u03AB'],
      '\u03CF',
      ['\u03D2', '\u03D4'],
      '\u03D8',
      '\u03DA',
      '\u03DC',
      '\u03DE',
      '\u03E0',
      '\u03E2',
      '\u03E4',
      '\u03E6',
      '\u03E8',
      '\u03EA',
      '\u03EC',
      '\u03EE',
      '\u03F4',
      '\u03F7',
      ['\u03F9', '\u03FA'],
      ['\u03FD', '\u042F'],
      '\u0460',
      '\u0462',
      '\u0464',
      '\u0466',
      '\u0468',
      '\u046A',
      '\u046C',
      '\u046E',
      '\u0470',
      '\u0472',
      '\u0474',
      '\u0476',
      '\u0478',
      '\u047A',
      '\u047C',
      '\u047E',
      '\u0480',
      '\u048A',
      '\u048C',
      '\u048E',
      '\u0490',
      '\u0492',
      '\u0494',
      '\u0496',
      '\u0498',
      '\u049A',
      '\u049C',
      '\u049E',
      '\u04A0',
      '\u04A2',
      '\u04A4',
      '\u04A6',
      '\u04A8',
      '\u04AA',
      '\u04AC',
      '\u04AE',
      '\u04B0',
      '\u04B2',
      '\u04B4',
      '\u04B6',
      '\u04B8',
      '\u04BA',
      '\u04BC',
      '\u04BE',
      ['\u04C0', '\u04C1'],
      '\u04C3',
      '\u04C5',
      '\u04C7',
      '\u04C9',
      '\u04CB',
      '\u04CD',
      '\u04D0',
      '\u04D2',
      '\u04D4',
      '\u04D6',
      '\u04D8',
      '\u04DA',
      '\u04DC',
      '\u04DE',
      '\u04E0',
      '\u04E2',
      '\u04E4',
      '\u04E6',
      '\u04E8',
      '\u04EA',
      '\u04EC',
      '\u04EE',
      '\u04F0',
      '\u04F2',
      '\u04F4',
      '\u04F6',
      '\u04F8',
      '\u04FA',
      '\u04FC',
      '\u04FE',
      '\u0500',
      '\u0502',
      '\u0504',
      '\u0506',
      '\u0508',
      '\u050A',
      '\u050C',
      '\u050E',
      '\u0510',
      '\u0512',
      '\u0514',
      '\u0516',
      '\u0518',
      '\u051A',
      '\u051C',
      '\u051E',
      '\u0520',
      '\u0522',
      '\u0524',
      '\u0526',
      ['\u0531', '\u0556'],
      ['\u10A0', '\u10C5'],
      '\u10C7',
      '\u10CD',
      '\u1E00',
      '\u1E02',
      '\u1E04',
      '\u1E06',
      '\u1E08',
      '\u1E0A',
      '\u1E0C',
      '\u1E0E',
      '\u1E10',
      '\u1E12',
      '\u1E14',
      '\u1E16',
      '\u1E18',
      '\u1E1A',
      '\u1E1C',
      '\u1E1E',
      '\u1E20',
      '\u1E22',
      '\u1E24',
      '\u1E26',
      '\u1E28',
      '\u1E2A',
      '\u1E2C',
      '\u1E2E',
      '\u1E30',
      '\u1E32',
      '\u1E34',
      '\u1E36',
      '\u1E38',
      '\u1E3A',
      '\u1E3C',
      '\u1E3E',
      '\u1E40',
      '\u1E42',
      '\u1E44',
      '\u1E46',
      '\u1E48',
      '\u1E4A',
      '\u1E4C',
      '\u1E4E',
      '\u1E50',
      '\u1E52',
      '\u1E54',
      '\u1E56',
      '\u1E58',
      '\u1E5A',
      '\u1E5C',
      '\u1E5E',
      '\u1E60',
      '\u1E62',
      '\u1E64',
      '\u1E66',
      '\u1E68',
      '\u1E6A',
      '\u1E6C',
      '\u1E6E',
      '\u1E70',
      '\u1E72',
      '\u1E74',
      '\u1E76',
      '\u1E78',
      '\u1E7A',
      '\u1E7C',
      '\u1E7E',
      '\u1E80',
      '\u1E82',
      '\u1E84',
      '\u1E86',
      '\u1E88',
      '\u1E8A',
      '\u1E8C',
      '\u1E8E',
      '\u1E90',
      '\u1E92',
      '\u1E94',
      '\u1E9E',
      '\u1EA0',
      '\u1EA2',
      '\u1EA4',
      '\u1EA6',
      '\u1EA8',
      '\u1EAA',
      '\u1EAC',
      '\u1EAE',
      '\u1EB0',
      '\u1EB2',
      '\u1EB4',
      '\u1EB6',
      '\u1EB8',
      '\u1EBA',
      '\u1EBC',
      '\u1EBE',
      '\u1EC0',
      '\u1EC2',
      '\u1EC4',
      '\u1EC6',
      '\u1EC8',
      '\u1ECA',
      '\u1ECC',
      '\u1ECE',
      '\u1ED0',
      '\u1ED2',
      '\u1ED4',
      '\u1ED6',
      '\u1ED8',
      '\u1EDA',
      '\u1EDC',
      '\u1EDE',
      '\u1EE0',
      '\u1EE2',
      '\u1EE4',
      '\u1EE6',
      '\u1EE8',
      '\u1EEA',
      '\u1EEC',
      '\u1EEE',
      '\u1EF0',
      '\u1EF2',
      '\u1EF4',
      '\u1EF6',
      '\u1EF8',
      '\u1EFA',
      '\u1EFC',
      '\u1EFE',
      ['\u1F08', '\u1F0F'],
      ['\u1F18', '\u1F1D'],
      ['\u1F28', '\u1F2F'],
      ['\u1F38', '\u1F3F'],
      ['\u1F48', '\u1F4D'],
      '\u1F59',
      '\u1F5B',
      '\u1F5D',
      '\u1F5F',
      ['\u1F68', '\u1F6F'],
      ['\u1FB8', '\u1FBB'],
      ['\u1FC8', '\u1FCB'],
      ['\u1FD8', '\u1FDB'],
      ['\u1FE8', '\u1FEC'],
      ['\u1FF8', '\u1FFB'],
      '\u2102',
      '\u2107',
      ['\u210B', '\u210D'],
      ['\u2110', '\u2112'],
      '\u2115',
      ['\u2119', '\u211D'],
      '\u2124',
      '\u2126',
      '\u2128',
      ['\u212A', '\u212D'],
      ['\u2130', '\u2133'],
      ['\u213E', '\u213F'],
      '\u2145',
      '\u2183',
      ['\u2C00', '\u2C2E'],
      '\u2C60',
      ['\u2C62', '\u2C64'],
      '\u2C67',
      '\u2C69',
      '\u2C6B',
      ['\u2C6D', '\u2C70'],
      '\u2C72',
      '\u2C75',
      ['\u2C7E', '\u2C80'],
      '\u2C82',
      '\u2C84',
      '\u2C86',
      '\u2C88',
      '\u2C8A',
      '\u2C8C',
      '\u2C8E',
      '\u2C90',
      '\u2C92',
      '\u2C94',
      '\u2C96',
      '\u2C98',
      '\u2C9A',
      '\u2C9C',
      '\u2C9E',
      '\u2CA0',
      '\u2CA2',
      '\u2CA4',
      '\u2CA6',
      '\u2CA8',
      '\u2CAA',
      '\u2CAC',
      '\u2CAE',
      '\u2CB0',
      '\u2CB2',
      '\u2CB4',
      '\u2CB6',
      '\u2CB8',
      '\u2CBA',
      '\u2CBC',
      '\u2CBE',
      '\u2CC0',
      '\u2CC2',
      '\u2CC4',
      '\u2CC6',
      '\u2CC8',
      '\u2CCA',
      '\u2CCC',
      '\u2CCE',
      '\u2CD0',
      '\u2CD2',
      '\u2CD4',
      '\u2CD6',
      '\u2CD8',
      '\u2CDA',
      '\u2CDC',
      '\u2CDE',
      '\u2CE0',
      '\u2CE2',
      '\u2CEB',
      '\u2CED',
      '\u2CF2',
      '\uA640',
      '\uA642',
      '\uA644',
      '\uA646',
      '\uA648',
      '\uA64A',
      '\uA64C',
      '\uA64E',
      '\uA650',
      '\uA652',
      '\uA654',
      '\uA656',
      '\uA658',
      '\uA65A',
      '\uA65C',
      '\uA65E',
      '\uA660',
      '\uA662',
      '\uA664',
      '\uA666',
      '\uA668',
      '\uA66A',
      '\uA66C',
      '\uA680',
      '\uA682',
      '\uA684',
      '\uA686',
      '\uA688',
      '\uA68A',
      '\uA68C',
      '\uA68E',
      '\uA690',
      '\uA692',
      '\uA694',
      '\uA696',
      '\uA722',
      '\uA724',
      '\uA726',
      '\uA728',
      '\uA72A',
      '\uA72C',
      '\uA72E',
      '\uA732',
      '\uA734',
      '\uA736',
      '\uA738',
      '\uA73A',
      '\uA73C',
      '\uA73E',
      '\uA740',
      '\uA742',
      '\uA744',
      '\uA746',
      '\uA748',
      '\uA74A',
      '\uA74C',
      '\uA74E',
      '\uA750',
      '\uA752',
      '\uA754',
      '\uA756',
      '\uA758',
      '\uA75A',
      '\uA75C',
      '\uA75E',
      '\uA760',
      '\uA762',
      '\uA764',
      '\uA766',
      '\uA768',
      '\uA76A',
      '\uA76C',
      '\uA76E',
      '\uA779',
      '\uA77B',
      ['\uA77D', '\uA77E'],
      '\uA780',
      '\uA782',
      '\uA784',
      '\uA786',
      '\uA78B',
      '\uA78D',
      '\uA790',
      '\uA792',
      '\uA7A0',
      '\uA7A2',
      '\uA7A4',
      '\uA7A6',
      '\uA7A8',
      '\uA7AA',
      ['\uFF21', '\uFF3A'],
    ],
    false,
    false,
  );
  const peg$c150 = /^[\u16EE-\u16F0\u2160-\u2182\u2185-\u2188\u3007\u3021-\u3029\u3038-\u303A\uA6E6-\uA6EF]/;
  const peg$c151 = peg$classExpectation(
    [
      ['\u16EE', '\u16F0'],
      ['\u2160', '\u2182'],
      ['\u2185', '\u2188'],
      '\u3007',
      ['\u3021', '\u3029'],
      ['\u3038', '\u303A'],
      ['\uA6E6', '\uA6EF'],
    ],
    false,
    false,
  );
  const peg$c152 =
    /^[0-9\u0660-\u0669\u06F0-\u06F9\u07C0-\u07C9\u0966-\u096F\u09E6-\u09EF\u0A66-\u0A6F\u0AE6-\u0AEF\u0B66-\u0B6F\u0BE6-\u0BEF\u0C66-\u0C6F\u0CE6-\u0CEF\u0D66-\u0D6F\u0E50-\u0E59\u0ED0-\u0ED9\u0F20-\u0F29\u1040-\u1049\u1090-\u1099\u17E0-\u17E9\u1810-\u1819\u1946-\u194F\u19D0-\u19D9\u1A80-\u1A89\u1A90-\u1A99\u1B50-\u1B59\u1BB0-\u1BB9\u1C40-\u1C49\u1C50-\u1C59\uA620-\uA629\uA8D0-\uA8D9\uA900-\uA909\uA9D0-\uA9D9\uAA50-\uAA59\uABF0-\uABF9\uFF10-\uFF19]/;
  const peg$c153 = peg$classExpectation(
    [
      ['0', '9'],
      ['\u0660', '\u0669'],
      ['\u06F0', '\u06F9'],
      ['\u07C0', '\u07C9'],
      ['\u0966', '\u096F'],
      ['\u09E6', '\u09EF'],
      ['\u0A66', '\u0A6F'],
      ['\u0AE6', '\u0AEF'],
      ['\u0B66', '\u0B6F'],
      ['\u0BE6', '\u0BEF'],
      ['\u0C66', '\u0C6F'],
      ['\u0CE6', '\u0CEF'],
      ['\u0D66', '\u0D6F'],
      ['\u0E50', '\u0E59'],
      ['\u0ED0', '\u0ED9'],
      ['\u0F20', '\u0F29'],
      ['\u1040', '\u1049'],
      ['\u1090', '\u1099'],
      ['\u17E0', '\u17E9'],
      ['\u1810', '\u1819'],
      ['\u1946', '\u194F'],
      ['\u19D0', '\u19D9'],
      ['\u1A80', '\u1A89'],
      ['\u1A90', '\u1A99'],
      ['\u1B50', '\u1B59'],
      ['\u1BB0', '\u1BB9'],
      ['\u1C40', '\u1C49'],
      ['\u1C50', '\u1C59'],
      ['\uA620', '\uA629'],
      ['\uA8D0', '\uA8D9'],
      ['\uA900', '\uA909'],
      ['\uA9D0', '\uA9D9'],
      ['\uAA50', '\uAA59'],
      ['\uABF0', '\uABF9'],
      ['\uFF10', '\uFF19'],
    ],
    false,
    false,
  );
  let peg$currPos = 0;
  let peg$savedPos = 0;
  const peg$posDetailsCache = [{ line: 1, column: 1 }];
  let peg$maxFailPos = 0;
  let peg$maxFailExpected = [];
  let peg$silentFails = 0;
  let peg$result;
  if (options.startRule !== undefined) {
    if (!(options.startRule in peg$startRuleFunctions)) {
      throw new Error('Can\'t start parsing from rule "' + options.startRule + '".');
    }
    peg$startRuleFunction = peg$startRuleFunctions[options.startRule];
  }
  function text() {
    return input.substring(peg$savedPos, peg$currPos);
  }
  function location() {
    return peg$computeLocation(peg$savedPos, peg$currPos);
  }
  function error(message, location1) {
    location1 = location1 !== undefined ? location1 : peg$computeLocation(peg$savedPos, peg$currPos);
    throw peg$buildSimpleError(message, location1);
  }
  function peg$literalExpectation(text1, ignoreCase) {
    return { type: 'literal', text: text1, ignoreCase: ignoreCase };
  }
  function peg$classExpectation(parts, inverted, ignoreCase) {
    return { type: 'class', parts: parts, inverted: inverted, ignoreCase: ignoreCase };
  }
  function peg$anyExpectation() {
    return { type: 'any' };
  }
  function peg$endExpectation() {
    return { type: 'end' };
  }
  function peg$otherExpectation(description) {
    return { type: 'other', description: description };
  }
  function peg$computePosDetails(pos) {
    let details = peg$posDetailsCache[pos];
    let p;
    if (details) {
      return details;
    } else {
      p = pos - 1;
      while (!peg$posDetailsCache[p]) {
        p--;
      }
      details = peg$posDetailsCache[p];
      details = {
        line: details.line,
        column: details.column,
      };
      while (p < pos) {
        if (input.charCodeAt(p) === 10) {
          details.line++;
          details.column = 1;
        } else {
          details.column++;
        }
        p++;
      }
      peg$posDetailsCache[pos] = details;
      return details;
    }
  }
  function peg$computeLocation(startPos, endPos) {
    const startPosDetails = peg$computePosDetails(startPos);
    const endPosDetails = peg$computePosDetails(endPos);
    return {
      source: peg$source,
      start: {
        offset: startPos,
        line: startPosDetails.line,
        column: startPosDetails.column,
      },
      end: {
        offset: endPos,
        line: endPosDetails.line,
        column: endPosDetails.column,
      },
    };
  }
  function peg$fail(expected1) {
    if (peg$currPos < peg$maxFailPos) {
      return;
    }
    if (peg$currPos > peg$maxFailPos) {
      peg$maxFailPos = peg$currPos;
      peg$maxFailExpected = [];
    }
    peg$maxFailExpected.push(expected1);
  }
  function peg$buildSimpleError(message, location1) {
    return new DotSyntaxError$1(message, [], '', location1);
  }
  function peg$buildStructuredError(expected1, found, location1) {
    return new DotSyntaxError$1(DotSyntaxError$1.buildMessage(expected1, found), expected1, found, location1);
  }
  function peg$parseDot() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_dot();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseGraph() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_graph();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseNode() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_node();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseAttributeList() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_attributes();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseAttribute() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_attribute();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseEdge() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_edge();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseSubgraph() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_subgraph();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseNodeRef() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_node_ref();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseAttibutesItem() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_attibutes_item();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseComment() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse__();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_comment();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse__();
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c0(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseClusterStatements() {
    let s0, s1;
    s0 = [];
    s1 = peg$parseAttribute();
    if (s1 === peg$FAILED) {
      s1 = peg$parseAttributeList();
      if (s1 === peg$FAILED) {
        s1 = peg$parseEdge();
        if (s1 === peg$FAILED) {
          s1 = peg$parseSubgraph();
          if (s1 === peg$FAILED) {
            s1 = peg$parseNode();
            if (s1 === peg$FAILED) {
              s1 = peg$parseComment();
            }
          }
        }
      }
    }
    while (s1 !== peg$FAILED) {
      s0.push(s1);
      s1 = peg$parseAttribute();
      if (s1 === peg$FAILED) {
        s1 = peg$parseAttributeList();
        if (s1 === peg$FAILED) {
          s1 = peg$parseEdge();
          if (s1 === peg$FAILED) {
            s1 = peg$parseSubgraph();
            if (s1 === peg$FAILED) {
              s1 = peg$parseNode();
              if (s1 === peg$FAILED) {
                s1 = peg$parseComment();
              }
            }
          }
        }
      }
    }
    return s0;
  }
  function peg$parse_dot() {
    let s0, s1, s2, s3, s4;
    s0 = peg$currPos;
    s1 = [];
    s2 = peg$parseComment();
    while (s2 !== peg$FAILED) {
      s1.push(s2);
      s2 = peg$parseComment();
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parseGraph();
      if (s2 !== peg$FAILED) {
        s3 = [];
        s4 = peg$parseComment();
        while (s4 !== peg$FAILED) {
          s3.push(s4);
          s4 = peg$parseComment();
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c1(s1, s2, s3);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_graph() {
    let s0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10;
    s0 = peg$currPos;
    if (input.substr(peg$currPos, 6).toLowerCase() === peg$c2) {
      s1 = input.substr(peg$currPos, 6);
      peg$currPos += 6;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c3);
      }
    }
    if (s1 === peg$FAILED) {
      s1 = null;
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_();
      if (s2 !== peg$FAILED) {
        if (input.substr(peg$currPos, 5).toLowerCase() === peg$c4) {
          s3 = input.substr(peg$currPos, 5);
          peg$currPos += 5;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c5);
          }
        }
        if (s3 === peg$FAILED) {
          if (input.substr(peg$currPos, 7).toLowerCase() === peg$c6) {
            s3 = input.substr(peg$currPos, 7);
            peg$currPos += 7;
          } else {
            s3 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c7);
            }
          }
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_();
          if (s4 !== peg$FAILED) {
            s5 = peg$parse_literal();
            if (s5 === peg$FAILED) {
              s5 = null;
            }
            if (s5 !== peg$FAILED) {
              s6 = peg$parse__();
              if (s6 !== peg$FAILED) {
                if (input.charCodeAt(peg$currPos) === 123) {
                  s7 = peg$c8;
                  peg$currPos++;
                } else {
                  s7 = peg$FAILED;
                  if (peg$silentFails === 0) {
                    peg$fail(peg$c9);
                  }
                }
                if (s7 !== peg$FAILED) {
                  s8 = peg$parseClusterStatements();
                  if (s8 !== peg$FAILED) {
                    s9 = peg$parse__();
                    if (s9 !== peg$FAILED) {
                      if (input.charCodeAt(peg$currPos) === 125) {
                        s10 = peg$c10;
                        peg$currPos++;
                      } else {
                        s10 = peg$FAILED;
                        if (peg$silentFails === 0) {
                          peg$fail(peg$c11);
                        }
                      }
                      if (s10 !== peg$FAILED) {
                        peg$savedPos = s0;
                        s1 = peg$c12(s1, s3, s5, s8);
                        s0 = s1;
                      } else {
                        peg$currPos = s0;
                        s0 = peg$FAILED;
                      }
                    } else {
                      peg$currPos = s0;
                      s0 = peg$FAILED;
                    }
                  } else {
                    peg$currPos = s0;
                    s0 = peg$FAILED;
                  }
                } else {
                  peg$currPos = s0;
                  s0 = peg$FAILED;
                }
              } else {
                peg$currPos = s0;
                s0 = peg$FAILED;
              }
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_attribute() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse_key_value();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_();
      if (s2 !== peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 59) {
          s3 = peg$c13;
          peg$currPos++;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c14);
          }
        }
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c15(s1);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_attributes() {
    let s0, s1, s2, s3, s4;
    s0 = peg$currPos;
    if (input.substr(peg$currPos, 5).toLowerCase() === peg$c4) {
      s1 = input.substr(peg$currPos, 5);
      peg$currPos += 5;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c5);
      }
    }
    if (s1 === peg$FAILED) {
      if (input.substr(peg$currPos, 4).toLowerCase() === peg$c16) {
        s1 = input.substr(peg$currPos, 4);
        peg$currPos += 4;
      } else {
        s1 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c17);
        }
      }
      if (s1 === peg$FAILED) {
        if (input.substr(peg$currPos, 4).toLowerCase() === peg$c18) {
          s1 = input.substr(peg$currPos, 4);
          peg$currPos += 4;
        } else {
          s1 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c19);
          }
        }
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_attribute_list();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse_();
        if (s3 !== peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 59) {
            s4 = peg$c13;
            peg$currPos++;
          } else {
            s4 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c14);
            }
          }
          if (s4 === peg$FAILED) {
            s4 = null;
          }
          if (s4 !== peg$FAILED) {
            peg$savedPos = s0;
            s1 = peg$c20(s1, s2);
            s0 = s1;
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_edge() {
    let s0, s1, s2, s3, s4, s5;
    s0 = peg$currPos;
    s1 = peg$parse_edge_target();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_edge_rhs();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse_attribute_list();
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_();
          if (s4 !== peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 59) {
              s5 = peg$c13;
              peg$currPos++;
            } else {
              s5 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c14);
              }
            }
            if (s5 === peg$FAILED) {
              s5 = null;
            }
            if (s5 !== peg$FAILED) {
              peg$savedPos = s0;
              s1 = peg$c21(s1, s2, s3);
              s0 = s1;
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_node() {
    let s0, s1, s2, s3, s4, s5;
    s0 = peg$currPos;
    s1 = peg$parse_literal();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse_attribute_list();
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_();
          if (s4 !== peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 59) {
              s5 = peg$c13;
              peg$currPos++;
            } else {
              s5 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c14);
              }
            }
            if (s5 === peg$FAILED) {
              s5 = null;
            }
            if (s5 !== peg$FAILED) {
              peg$savedPos = s0;
              s1 = peg$c22(s1, s3);
              s0 = s1;
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_key_value() {
    let s0, s1, s2, s3, s4, s5;
    s0 = peg$currPos;
    s1 = peg$parse_literal();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_();
      if (s2 !== peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 61) {
          s3 = peg$c23;
          peg$currPos++;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c24);
          }
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_();
          if (s4 !== peg$FAILED) {
            s5 = peg$parse_literal();
            if (s5 !== peg$FAILED) {
              peg$savedPos = s0;
              s1 = peg$c25(s1, s5);
              s0 = s1;
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_attibutes_item() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$parse_key_value();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_();
      if (s2 !== peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 44) {
          s3 = peg$c26;
          peg$currPos++;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c27);
          }
        }
        if (s3 === peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 59) {
            s3 = peg$c13;
            peg$currPos++;
          } else {
            s3 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c14);
            }
          }
        }
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c28(s1);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_attribute_list() {
    let s0, s1, s2, s3, s4, s5;
    s0 = peg$currPos;
    s1 = peg$parse_();
    if (s1 !== peg$FAILED) {
      if (input.charCodeAt(peg$currPos) === 91) {
        s2 = peg$c29;
        peg$currPos++;
      } else {
        s2 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c30);
        }
      }
      if (s2 !== peg$FAILED) {
        s3 = [];
        s4 = peg$parseAttibutesItem();
        if (s4 === peg$FAILED) {
          s4 = peg$parseComment();
        }
        while (s4 !== peg$FAILED) {
          s3.push(s4);
          s4 = peg$parseAttibutesItem();
          if (s4 === peg$FAILED) {
            s4 = peg$parseComment();
          }
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse__();
          if (s4 !== peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 93) {
              s5 = peg$c31;
              peg$currPos++;
            } else {
              s5 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c32);
              }
            }
            if (s5 !== peg$FAILED) {
              peg$savedPos = s0;
              s1 = peg$c33(s3);
              s0 = s1;
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_edge_target_group() {
    let s0, s1, s2, s3, s4, s5, s6;
    s0 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 123) {
      s1 = peg$c8;
      peg$currPos++;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c9);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parseNodeRef();
      if (s2 !== peg$FAILED) {
        s3 = [];
        s4 = peg$currPos;
        if (input.charCodeAt(peg$currPos) === 44) {
          s5 = peg$c26;
          peg$currPos++;
        } else {
          s5 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c27);
          }
        }
        if (s5 === peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 59) {
            s5 = peg$c13;
            peg$currPos++;
          } else {
            s5 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c14);
            }
          }
        }
        if (s5 === peg$FAILED) {
          s5 = null;
        }
        if (s5 !== peg$FAILED) {
          s6 = peg$parseNodeRef();
          if (s6 !== peg$FAILED) {
            peg$savedPos = s4;
            s5 = peg$c34(s2, s6);
            s4 = s5;
          } else {
            peg$currPos = s4;
            s4 = peg$FAILED;
          }
        } else {
          peg$currPos = s4;
          s4 = peg$FAILED;
        }
        while (s4 !== peg$FAILED) {
          s3.push(s4);
          s4 = peg$currPos;
          if (input.charCodeAt(peg$currPos) === 44) {
            s5 = peg$c26;
            peg$currPos++;
          } else {
            s5 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c27);
            }
          }
          if (s5 === peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 59) {
              s5 = peg$c13;
              peg$currPos++;
            } else {
              s5 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c14);
              }
            }
          }
          if (s5 === peg$FAILED) {
            s5 = null;
          }
          if (s5 !== peg$FAILED) {
            s6 = peg$parseNodeRef();
            if (s6 !== peg$FAILED) {
              peg$savedPos = s4;
              s5 = peg$c34(s2, s6);
              s4 = s5;
            } else {
              peg$currPos = s4;
              s4 = peg$FAILED;
            }
          } else {
            peg$currPos = s4;
            s4 = peg$FAILED;
          }
        }
        if (s3 !== peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 44) {
            s4 = peg$c26;
            peg$currPos++;
          } else {
            s4 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c27);
            }
          }
          if (s4 === peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 59) {
              s4 = peg$c13;
              peg$currPos++;
            } else {
              s4 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c14);
              }
            }
          }
          if (s4 === peg$FAILED) {
            s4 = null;
          }
          if (s4 !== peg$FAILED) {
            s5 = peg$parse__();
            if (s5 !== peg$FAILED) {
              if (input.charCodeAt(peg$currPos) === 125) {
                s6 = peg$c10;
                peg$currPos++;
              } else {
                s6 = peg$FAILED;
                if (peg$silentFails === 0) {
                  peg$fail(peg$c11);
                }
              }
              if (s6 !== peg$FAILED) {
                peg$savedPos = s0;
                s1 = peg$c35(s2, s3);
                s0 = s1;
              } else {
                peg$currPos = s0;
                s0 = peg$FAILED;
              }
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_edge_target() {
    let s0;
    s0 = peg$parse_edge_target_group();
    if (s0 === peg$FAILED) {
      s0 = peg$parseNodeRef();
    }
    return s0;
  }
  function peg$parse_edge_operator() {
    let s0, s1;
    s0 = peg$currPos;
    if (input.substr(peg$currPos, 2) === peg$c36) {
      s1 = peg$c36;
      peg$currPos += 2;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c37);
      }
    }
    if (s1 === peg$FAILED) {
      if (input.substr(peg$currPos, 2) === peg$c38) {
        s1 = peg$c38;
        peg$currPos += 2;
      } else {
        s1 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c39);
        }
      }
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c40(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parse_edge_rhs() {
    let s0, s1, s2, s3, s4, s5, s6;
    s0 = peg$currPos;
    s1 = peg$parse_();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_edge_operator();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse_();
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_edge_target();
          if (s4 !== peg$FAILED) {
            s5 = peg$parse_();
            if (s5 !== peg$FAILED) {
              s6 = peg$parse_edge_rhs();
              if (s6 === peg$FAILED) {
                s6 = null;
              }
              if (s6 !== peg$FAILED) {
                peg$savedPos = s0;
                s1 = peg$c41(s2, s4, s6);
                s0 = s1;
              } else {
                peg$currPos = s0;
                s0 = peg$FAILED;
              }
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_node_ref() {
    let s0, s1, s2;
    s0 = peg$currPos;
    s1 = peg$parse_literal();
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_port();
      if (s2 === peg$FAILED) {
        s2 = null;
      }
      if (s2 !== peg$FAILED) {
        peg$savedPos = s0;
        s1 = peg$c42(s1, s2);
        s0 = s1;
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_port() {
    let s0, s1, s2, s3, s4, s5;
    peg$silentFails++;
    s0 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 58) {
      s1 = peg$c44;
      peg$currPos++;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c45);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_literal();
      if (s2 !== peg$FAILED) {
        s3 = peg$currPos;
        if (input.charCodeAt(peg$currPos) === 58) {
          s4 = peg$c44;
          peg$currPos++;
        } else {
          s4 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c45);
          }
        }
        if (s4 !== peg$FAILED) {
          s5 = peg$parse_compass();
          if (s5 !== peg$FAILED) {
            peg$savedPos = s3;
            s4 = peg$c46(s2, s5);
            s3 = s4;
          } else {
            peg$currPos = s3;
            s3 = peg$FAILED;
          }
        } else {
          peg$currPos = s3;
          s3 = peg$FAILED;
        }
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c47(s2, s3);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    peg$silentFails--;
    if (s0 === peg$FAILED) {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c43);
      }
    }
    return s0;
  }
  function peg$parse_subgraph_id() {
    let s0, s1, s2, s3, s4;
    s0 = peg$currPos;
    if (input.substr(peg$currPos, 8).toLowerCase() === peg$c48) {
      s1 = input.substr(peg$currPos, 8);
      peg$currPos += 8;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c49);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parse_();
      if (s2 !== peg$FAILED) {
        s3 = peg$parse_literal();
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_();
          if (s4 !== peg$FAILED) {
            peg$savedPos = s0;
            s1 = peg$c50(s3);
            s0 = s1;
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_subgraph() {
    let s0, s1, s2, s3, s4, s5;
    s0 = peg$currPos;
    s1 = peg$parse_subgraph_id();
    if (s1 === peg$FAILED) {
      s1 = null;
    }
    if (s1 !== peg$FAILED) {
      if (input.charCodeAt(peg$currPos) === 123) {
        s2 = peg$c8;
        peg$currPos++;
      } else {
        s2 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c9);
        }
      }
      if (s2 !== peg$FAILED) {
        s3 = peg$parseClusterStatements();
        if (s3 === peg$FAILED) {
          s3 = null;
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse__();
          if (s4 !== peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 125) {
              s5 = peg$c10;
              peg$currPos++;
            } else {
              s5 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c11);
              }
            }
            if (s5 !== peg$FAILED) {
              peg$savedPos = s0;
              s1 = peg$c51(s1, s3);
              s0 = s1;
            } else {
              peg$currPos = s0;
              s0 = peg$FAILED;
            }
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_compass_keyword() {
    let s0;
    if (input.charCodeAt(peg$currPos) === 110) {
      s0 = peg$c52;
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c53);
      }
    }
    if (s0 === peg$FAILED) {
      if (input.substr(peg$currPos, 2) === peg$c54) {
        s0 = peg$c54;
        peg$currPos += 2;
      } else {
        s0 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c55);
        }
      }
      if (s0 === peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 101) {
          s0 = peg$c56;
          peg$currPos++;
        } else {
          s0 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c57);
          }
        }
        if (s0 === peg$FAILED) {
          if (input.substr(peg$currPos, 2) === peg$c58) {
            s0 = peg$c58;
            peg$currPos += 2;
          } else {
            s0 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c59);
            }
          }
          if (s0 === peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 115) {
              s0 = peg$c60;
              peg$currPos++;
            } else {
              s0 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c61);
              }
            }
            if (s0 === peg$FAILED) {
              if (input.substr(peg$currPos, 2) === peg$c62) {
                s0 = peg$c62;
                peg$currPos += 2;
              } else {
                s0 = peg$FAILED;
                if (peg$silentFails === 0) {
                  peg$fail(peg$c63);
                }
              }
              if (s0 === peg$FAILED) {
                if (input.charCodeAt(peg$currPos) === 119) {
                  s0 = peg$c64;
                  peg$currPos++;
                } else {
                  s0 = peg$FAILED;
                  if (peg$silentFails === 0) {
                    peg$fail(peg$c65);
                  }
                }
                if (s0 === peg$FAILED) {
                  if (input.substr(peg$currPos, 2) === peg$c66) {
                    s0 = peg$c66;
                    peg$currPos += 2;
                  } else {
                    s0 = peg$FAILED;
                    if (peg$silentFails === 0) {
                      peg$fail(peg$c67);
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    return s0;
  }
  function peg$parse_compass() {
    let s0, s1, s2, s3, s4;
    s0 = peg$currPos;
    s1 = peg$currPos;
    s2 = peg$parse_compass_keyword();
    if (s2 !== peg$FAILED) {
      peg$savedPos = s1;
      s2 = peg$c68(s2);
    }
    s1 = s2;
    if (s1 === peg$FAILED) {
      s1 = peg$currPos;
      if (input.charCodeAt(peg$currPos) === 34) {
        s2 = peg$c69;
        peg$currPos++;
      } else {
        s2 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c70);
        }
      }
      if (s2 !== peg$FAILED) {
        s3 = peg$parse_compass_keyword();
        if (s3 !== peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 34) {
            s4 = peg$c69;
            peg$currPos++;
          } else {
            s4 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c70);
            }
          }
          if (s4 !== peg$FAILED) {
            peg$savedPos = s1;
            s2 = peg$c71(s3);
            s1 = s2;
          } else {
            peg$currPos = s1;
            s1 = peg$FAILED;
          }
        } else {
          peg$currPos = s1;
          s1 = peg$FAILED;
        }
      } else {
        peg$currPos = s1;
        s1 = peg$FAILED;
      }
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c72(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parse_literal() {
    let s0, s1;
    s0 = peg$parseQUOTED_STRING();
    if (s0 === peg$FAILED) {
      s0 = peg$parseHTML_STRING();
      if (s0 === peg$FAILED) {
        s0 = peg$currPos;
        s1 = peg$parseSTRING();
        if (s1 === peg$FAILED) {
          s1 = peg$parseNUMBER_STRING();
          if (s1 === peg$FAILED) {
            s1 = peg$parseNUMBER();
          }
        }
        if (s1 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c73(s1);
        }
        s0 = s1;
      }
    }
    return s0;
  }
  function peg$parse_comment() {
    let s0;
    s0 = peg$parse_block_comment();
    if (s0 === peg$FAILED) {
      s0 = peg$parse_slash_comment();
      if (s0 === peg$FAILED) {
        s0 = peg$parse_macro_comment();
      }
    }
    return s0;
  }
  function peg$parse_block_comment() {
    let s0, s1, s2, s3, s4, s5;
    s0 = peg$currPos;
    if (input.substr(peg$currPos, 2) === peg$c74) {
      s1 = peg$c74;
      peg$currPos += 2;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c75);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = [];
      s3 = peg$currPos;
      s4 = peg$currPos;
      peg$silentFails++;
      if (input.substr(peg$currPos, 2) === peg$c76) {
        s5 = peg$c76;
        peg$currPos += 2;
      } else {
        s5 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c77);
        }
      }
      peg$silentFails--;
      if (s5 === peg$FAILED) {
        s4 = undefined;
      } else {
        peg$currPos = s4;
        s4 = peg$FAILED;
      }
      if (s4 !== peg$FAILED) {
        if (input.length > peg$currPos) {
          s5 = input.charAt(peg$currPos);
          peg$currPos++;
        } else {
          s5 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c78);
          }
        }
        if (s5 !== peg$FAILED) {
          peg$savedPos = s3;
          s4 = peg$c0(s5);
          s3 = s4;
        } else {
          peg$currPos = s3;
          s3 = peg$FAILED;
        }
      } else {
        peg$currPos = s3;
        s3 = peg$FAILED;
      }
      while (s3 !== peg$FAILED) {
        s2.push(s3);
        s3 = peg$currPos;
        s4 = peg$currPos;
        peg$silentFails++;
        if (input.substr(peg$currPos, 2) === peg$c76) {
          s5 = peg$c76;
          peg$currPos += 2;
        } else {
          s5 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c77);
          }
        }
        peg$silentFails--;
        if (s5 === peg$FAILED) {
          s4 = undefined;
        } else {
          peg$currPos = s4;
          s4 = peg$FAILED;
        }
        if (s4 !== peg$FAILED) {
          if (input.length > peg$currPos) {
            s5 = input.charAt(peg$currPos);
            peg$currPos++;
          } else {
            s5 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c78);
            }
          }
          if (s5 !== peg$FAILED) {
            peg$savedPos = s3;
            s4 = peg$c0(s5);
            s3 = s4;
          } else {
            peg$currPos = s3;
            s3 = peg$FAILED;
          }
        } else {
          peg$currPos = s3;
          s3 = peg$FAILED;
        }
      }
      if (s2 !== peg$FAILED) {
        if (input.substr(peg$currPos, 2) === peg$c76) {
          s3 = peg$c76;
          peg$currPos += 2;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c77);
          }
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c79(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_slash_comment() {
    let s0, s1, s2;
    s0 = peg$currPos;
    s1 = [];
    s2 = peg$parse_slash_comment_line();
    if (s2 !== peg$FAILED) {
      while (s2 !== peg$FAILED) {
        s1.push(s2);
        s2 = peg$parse_slash_comment_line();
      }
    } else {
      s1 = peg$FAILED;
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c80(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parse_slash_comment_line() {
    let s0, s1, s2, s3, s4, s5, s6;
    s0 = peg$currPos;
    s1 = peg$parse_();
    if (s1 !== peg$FAILED) {
      if (input.substr(peg$currPos, 2) === peg$c81) {
        s2 = peg$c81;
        peg$currPos += 2;
      } else {
        s2 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c82);
        }
      }
      if (s2 !== peg$FAILED) {
        s3 = [];
        s4 = peg$currPos;
        s5 = peg$currPos;
        peg$silentFails++;
        s6 = peg$parse_newline();
        peg$silentFails--;
        if (s6 === peg$FAILED) {
          s5 = undefined;
        } else {
          peg$currPos = s5;
          s5 = peg$FAILED;
        }
        if (s5 !== peg$FAILED) {
          if (input.length > peg$currPos) {
            s6 = input.charAt(peg$currPos);
            peg$currPos++;
          } else {
            s6 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c78);
            }
          }
          if (s6 !== peg$FAILED) {
            peg$savedPos = s4;
            s5 = peg$c0(s6);
            s4 = s5;
          } else {
            peg$currPos = s4;
            s4 = peg$FAILED;
          }
        } else {
          peg$currPos = s4;
          s4 = peg$FAILED;
        }
        while (s4 !== peg$FAILED) {
          s3.push(s4);
          s4 = peg$currPos;
          s5 = peg$currPos;
          peg$silentFails++;
          s6 = peg$parse_newline();
          peg$silentFails--;
          if (s6 === peg$FAILED) {
            s5 = undefined;
          } else {
            peg$currPos = s5;
            s5 = peg$FAILED;
          }
          if (s5 !== peg$FAILED) {
            if (input.length > peg$currPos) {
              s6 = input.charAt(peg$currPos);
              peg$currPos++;
            } else {
              s6 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c78);
              }
            }
            if (s6 !== peg$FAILED) {
              peg$savedPos = s4;
              s5 = peg$c0(s6);
              s4 = s5;
            } else {
              peg$currPos = s4;
              s4 = peg$FAILED;
            }
          } else {
            peg$currPos = s4;
            s4 = peg$FAILED;
          }
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_newline();
          if (s4 === peg$FAILED) {
            s4 = null;
          }
          if (s4 !== peg$FAILED) {
            peg$savedPos = s0;
            s1 = peg$c83(s3);
            s0 = s1;
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parse_macro_comment() {
    let s0, s1, s2;
    s0 = peg$currPos;
    s1 = [];
    s2 = peg$parse_macro_comment_line();
    if (s2 !== peg$FAILED) {
      while (s2 !== peg$FAILED) {
        s1.push(s2);
        s2 = peg$parse_macro_comment_line();
      }
    } else {
      s1 = peg$FAILED;
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c84(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parse_macro_comment_line() {
    let s0, s1, s2, s3, s4, s5, s6;
    s0 = peg$currPos;
    s1 = peg$parse_();
    if (s1 !== peg$FAILED) {
      if (input.charCodeAt(peg$currPos) === 35) {
        s2 = peg$c85;
        peg$currPos++;
      } else {
        s2 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c86);
        }
      }
      if (s2 !== peg$FAILED) {
        s3 = [];
        s4 = peg$currPos;
        s5 = peg$currPos;
        peg$silentFails++;
        s6 = peg$parse_newline();
        peg$silentFails--;
        if (s6 === peg$FAILED) {
          s5 = undefined;
        } else {
          peg$currPos = s5;
          s5 = peg$FAILED;
        }
        if (s5 !== peg$FAILED) {
          if (input.length > peg$currPos) {
            s6 = input.charAt(peg$currPos);
            peg$currPos++;
          } else {
            s6 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c78);
            }
          }
          if (s6 !== peg$FAILED) {
            peg$savedPos = s4;
            s5 = peg$c0(s6);
            s4 = s5;
          } else {
            peg$currPos = s4;
            s4 = peg$FAILED;
          }
        } else {
          peg$currPos = s4;
          s4 = peg$FAILED;
        }
        while (s4 !== peg$FAILED) {
          s3.push(s4);
          s4 = peg$currPos;
          s5 = peg$currPos;
          peg$silentFails++;
          s6 = peg$parse_newline();
          peg$silentFails--;
          if (s6 === peg$FAILED) {
            s5 = undefined;
          } else {
            peg$currPos = s5;
            s5 = peg$FAILED;
          }
          if (s5 !== peg$FAILED) {
            if (input.length > peg$currPos) {
              s6 = input.charAt(peg$currPos);
              peg$currPos++;
            } else {
              s6 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c78);
              }
            }
            if (s6 !== peg$FAILED) {
              peg$savedPos = s4;
              s5 = peg$c0(s6);
              s4 = s5;
            } else {
              peg$currPos = s4;
              s4 = peg$FAILED;
            }
          } else {
            peg$currPos = s4;
            s4 = peg$FAILED;
          }
        }
        if (s3 !== peg$FAILED) {
          s4 = peg$parse_newline();
          if (s4 === peg$FAILED) {
            s4 = null;
          }
          if (s4 !== peg$FAILED) {
            peg$savedPos = s0;
            s1 = peg$c83(s3);
            s0 = s1;
          } else {
            peg$currPos = s0;
            s0 = peg$FAILED;
          }
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseSTRING() {
    let s0, s1, s2, s3;
    peg$silentFails++;
    s0 = peg$currPos;
    s1 = peg$parseStringStart();
    if (s1 !== peg$FAILED) {
      s2 = [];
      s3 = peg$parseStringPart();
      while (s3 !== peg$FAILED) {
        s2.push(s3);
        s3 = peg$parseStringPart();
      }
      if (s2 !== peg$FAILED) {
        peg$savedPos = s0;
        s1 = peg$c88(s1, s2);
        s0 = s1;
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    peg$silentFails--;
    if (s0 === peg$FAILED) {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c87);
      }
    }
    return s0;
  }
  function peg$parseNUMBER_STRING() {
    let s0, s1, s2;
    s0 = peg$currPos;
    s1 = peg$parseNUMBER();
    if (s1 !== peg$FAILED) {
      s2 = peg$parseSTRING();
      if (s2 !== peg$FAILED) {
        peg$savedPos = s0;
        s1 = peg$c89(s1, s2);
        s0 = s1;
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseStringStart() {
    let s0;
    s0 = peg$parseUnicodeLetter();
    if (s0 === peg$FAILED) {
      if (input.charCodeAt(peg$currPos) === 36) {
        s0 = peg$c90;
        peg$currPos++;
      } else {
        s0 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c91);
        }
      }
      if (s0 === peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 95) {
          s0 = peg$c92;
          peg$currPos++;
        } else {
          s0 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c93);
          }
        }
      }
    }
    return s0;
  }
  function peg$parseStringPart() {
    let s0;
    s0 = peg$parseStringStart();
    if (s0 === peg$FAILED) {
      s0 = peg$parseNd();
    }
    return s0;
  }
  function peg$parseNUMBER() {
    let s0, s1, s2, s3, s4, s5, s6, s7, s8;
    peg$silentFails++;
    s0 = peg$currPos;
    s1 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 45) {
      s2 = peg$c95;
      peg$currPos++;
    } else {
      s2 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c96);
      }
    }
    if (s2 === peg$FAILED) {
      s2 = null;
    }
    if (s2 !== peg$FAILED) {
      s3 = peg$currPos;
      if (input.charCodeAt(peg$currPos) === 46) {
        s4 = peg$c97;
        peg$currPos++;
      } else {
        s4 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c98);
        }
      }
      if (s4 !== peg$FAILED) {
        s5 = [];
        if (peg$c99.test(input.charAt(peg$currPos))) {
          s6 = input.charAt(peg$currPos);
          peg$currPos++;
        } else {
          s6 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c100);
          }
        }
        if (s6 !== peg$FAILED) {
          while (s6 !== peg$FAILED) {
            s5.push(s6);
            if (peg$c99.test(input.charAt(peg$currPos))) {
              s6 = input.charAt(peg$currPos);
              peg$currPos++;
            } else {
              s6 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c100);
              }
            }
          }
        } else {
          s5 = peg$FAILED;
        }
        if (s5 !== peg$FAILED) {
          s4 = [s4, s5];
          s3 = s4;
        } else {
          peg$currPos = s3;
          s3 = peg$FAILED;
        }
      } else {
        peg$currPos = s3;
        s3 = peg$FAILED;
      }
      if (s3 === peg$FAILED) {
        s3 = peg$currPos;
        s4 = [];
        if (peg$c99.test(input.charAt(peg$currPos))) {
          s5 = input.charAt(peg$currPos);
          peg$currPos++;
        } else {
          s5 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c100);
          }
        }
        if (s5 !== peg$FAILED) {
          while (s5 !== peg$FAILED) {
            s4.push(s5);
            if (peg$c99.test(input.charAt(peg$currPos))) {
              s5 = input.charAt(peg$currPos);
              peg$currPos++;
            } else {
              s5 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c100);
              }
            }
          }
        } else {
          s4 = peg$FAILED;
        }
        if (s4 !== peg$FAILED) {
          s5 = peg$currPos;
          if (input.charCodeAt(peg$currPos) === 46) {
            s6 = peg$c97;
            peg$currPos++;
          } else {
            s6 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c98);
            }
          }
          if (s6 !== peg$FAILED) {
            s7 = [];
            if (peg$c99.test(input.charAt(peg$currPos))) {
              s8 = input.charAt(peg$currPos);
              peg$currPos++;
            } else {
              s8 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c100);
              }
            }
            while (s8 !== peg$FAILED) {
              s7.push(s8);
              if (peg$c99.test(input.charAt(peg$currPos))) {
                s8 = input.charAt(peg$currPos);
                peg$currPos++;
              } else {
                s8 = peg$FAILED;
                if (peg$silentFails === 0) {
                  peg$fail(peg$c100);
                }
              }
            }
            if (s7 !== peg$FAILED) {
              s6 = [s6, s7];
              s5 = s6;
            } else {
              peg$currPos = s5;
              s5 = peg$FAILED;
            }
          } else {
            peg$currPos = s5;
            s5 = peg$FAILED;
          }
          if (s5 === peg$FAILED) {
            s5 = null;
          }
          if (s5 !== peg$FAILED) {
            s4 = [s4, s5];
            s3 = s4;
          } else {
            peg$currPos = s3;
            s3 = peg$FAILED;
          }
        } else {
          peg$currPos = s3;
          s3 = peg$FAILED;
        }
      }
      if (s3 !== peg$FAILED) {
        s2 = [s2, s3];
        s1 = s2;
      } else {
        peg$currPos = s1;
        s1 = peg$FAILED;
      }
    } else {
      peg$currPos = s1;
      s1 = peg$FAILED;
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c101();
    }
    s0 = s1;
    peg$silentFails--;
    if (s0 === peg$FAILED) {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c94);
      }
    }
    return s0;
  }
  function peg$parseHTML_STRING() {
    let s0, s1;
    s0 = peg$currPos;
    s1 = peg$parsehtml_raw_string();
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c102(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parsehtml_raw_string() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 60) {
      s1 = peg$c103;
      peg$currPos++;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c104);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = [];
      s3 = peg$parsehtml_char();
      if (s3 === peg$FAILED) {
        s3 = peg$parsehtml_raw_string();
      }
      while (s3 !== peg$FAILED) {
        s2.push(s3);
        s3 = peg$parsehtml_char();
        if (s3 === peg$FAILED) {
          s3 = peg$parsehtml_raw_string();
        }
      }
      if (s2 !== peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 62) {
          s3 = peg$c105;
          peg$currPos++;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c106);
          }
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c107(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parsehtml_char() {
    let s0, s1, s2, s3, s4;
    s0 = peg$currPos;
    s1 = [];
    s2 = peg$currPos;
    s3 = peg$currPos;
    peg$silentFails++;
    if (input.charCodeAt(peg$currPos) === 62) {
      s4 = peg$c105;
      peg$currPos++;
    } else {
      s4 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c106);
      }
    }
    if (s4 === peg$FAILED) {
      if (input.charCodeAt(peg$currPos) === 60) {
        s4 = peg$c103;
        peg$currPos++;
      } else {
        s4 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c104);
        }
      }
    }
    peg$silentFails--;
    if (s4 === peg$FAILED) {
      s3 = undefined;
    } else {
      peg$currPos = s3;
      s3 = peg$FAILED;
    }
    if (s3 !== peg$FAILED) {
      if (input.length > peg$currPos) {
        s4 = input.charAt(peg$currPos);
        peg$currPos++;
      } else {
        s4 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c78);
        }
      }
      if (s4 !== peg$FAILED) {
        peg$savedPos = s2;
        s3 = peg$c0(s4);
        s2 = s3;
      } else {
        peg$currPos = s2;
        s2 = peg$FAILED;
      }
    } else {
      peg$currPos = s2;
      s2 = peg$FAILED;
    }
    if (s2 !== peg$FAILED) {
      while (s2 !== peg$FAILED) {
        s1.push(s2);
        s2 = peg$currPos;
        s3 = peg$currPos;
        peg$silentFails++;
        if (input.charCodeAt(peg$currPos) === 62) {
          s4 = peg$c105;
          peg$currPos++;
        } else {
          s4 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c106);
          }
        }
        if (s4 === peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 60) {
            s4 = peg$c103;
            peg$currPos++;
          } else {
            s4 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c104);
            }
          }
        }
        peg$silentFails--;
        if (s4 === peg$FAILED) {
          s3 = undefined;
        } else {
          peg$currPos = s3;
          s3 = peg$FAILED;
        }
        if (s3 !== peg$FAILED) {
          if (input.length > peg$currPos) {
            s4 = input.charAt(peg$currPos);
            peg$currPos++;
          } else {
            s4 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c78);
            }
          }
          if (s4 !== peg$FAILED) {
            peg$savedPos = s2;
            s3 = peg$c0(s4);
            s2 = s3;
          } else {
            peg$currPos = s2;
            s2 = peg$FAILED;
          }
        } else {
          peg$currPos = s2;
          s2 = peg$FAILED;
        }
      }
    } else {
      s1 = peg$FAILED;
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c83(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parseQUOTED_STRING() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 34) {
      s1 = peg$c69;
      peg$currPos++;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c70);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = [];
      s3 = peg$parseDoubleStringCharacter();
      while (s3 !== peg$FAILED) {
        s2.push(s3);
        s3 = peg$parseDoubleStringCharacter();
      }
      if (s2 !== peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 34) {
          s3 = peg$c69;
          peg$currPos++;
        } else {
          s3 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c70);
          }
        }
        if (s3 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c108(s2);
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseDoubleStringCharacter() {
    let s0, s1, s2;
    s0 = peg$parseQuoteEscape();
    if (s0 === peg$FAILED) {
      s0 = peg$currPos;
      s1 = peg$currPos;
      peg$silentFails++;
      if (input.charCodeAt(peg$currPos) === 34) {
        s2 = peg$c69;
        peg$currPos++;
      } else {
        s2 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c70);
        }
      }
      if (s2 === peg$FAILED) {
        s2 = peg$parseLineTerminator();
      }
      peg$silentFails--;
      if (s2 === peg$FAILED) {
        s1 = undefined;
      } else {
        peg$currPos = s1;
        s1 = peg$FAILED;
      }
      if (s1 !== peg$FAILED) {
        s2 = peg$parseSourceCharacter();
        if (s2 !== peg$FAILED) {
          peg$savedPos = s0;
          s1 = peg$c109();
          s0 = s1;
        } else {
          peg$currPos = s0;
          s0 = peg$FAILED;
        }
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
      if (s0 === peg$FAILED) {
        s0 = peg$parseLineContinuation();
      }
    }
    return s0;
  }
  function peg$parseQuoteEscape() {
    let s0, s1, s2, s3;
    s0 = peg$currPos;
    s1 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 92) {
      s2 = peg$c110;
      peg$currPos++;
    } else {
      s2 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c111);
      }
    }
    if (s2 !== peg$FAILED) {
      if (input.length > peg$currPos) {
        s3 = input.charAt(peg$currPos);
        peg$currPos++;
      } else {
        s3 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c78);
        }
      }
      if (s3 !== peg$FAILED) {
        s2 = [s2, s3];
        s1 = s2;
      } else {
        peg$currPos = s1;
        s1 = peg$FAILED;
      }
    } else {
      peg$currPos = s1;
      s1 = peg$FAILED;
    }
    if (s1 !== peg$FAILED) {
      peg$savedPos = s0;
      s1 = peg$c112(s1);
    }
    s0 = s1;
    return s0;
  }
  function peg$parseLineContinuation() {
    let s0, s1, s2;
    s0 = peg$currPos;
    if (input.charCodeAt(peg$currPos) === 92) {
      s1 = peg$c110;
      peg$currPos++;
    } else {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c111);
      }
    }
    if (s1 !== peg$FAILED) {
      s2 = peg$parseLineTerminatorSequence();
      if (s2 !== peg$FAILED) {
        peg$savedPos = s0;
        s1 = peg$c113();
        s0 = s1;
      } else {
        peg$currPos = s0;
        s0 = peg$FAILED;
      }
    } else {
      peg$currPos = s0;
      s0 = peg$FAILED;
    }
    return s0;
  }
  function peg$parseLineTerminator() {
    let s0;
    if (peg$c114.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c115);
      }
    }
    return s0;
  }
  function peg$parseLineTerminatorSequence() {
    let s0;
    peg$silentFails++;
    if (input.charCodeAt(peg$currPos) === 10) {
      s0 = peg$c117;
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c118);
      }
    }
    if (s0 === peg$FAILED) {
      if (input.substr(peg$currPos, 2) === peg$c119) {
        s0 = peg$c119;
        peg$currPos += 2;
      } else {
        s0 = peg$FAILED;
        if (peg$silentFails === 0) {
          peg$fail(peg$c120);
        }
      }
      if (s0 === peg$FAILED) {
        if (input.charCodeAt(peg$currPos) === 13) {
          s0 = peg$c121;
          peg$currPos++;
        } else {
          s0 = peg$FAILED;
          if (peg$silentFails === 0) {
            peg$fail(peg$c122);
          }
        }
        if (s0 === peg$FAILED) {
          if (input.charCodeAt(peg$currPos) === 8232) {
            s0 = peg$c123;
            peg$currPos++;
          } else {
            s0 = peg$FAILED;
            if (peg$silentFails === 0) {
              peg$fail(peg$c124);
            }
          }
          if (s0 === peg$FAILED) {
            if (input.charCodeAt(peg$currPos) === 8233) {
              s0 = peg$c125;
              peg$currPos++;
            } else {
              s0 = peg$FAILED;
              if (peg$silentFails === 0) {
                peg$fail(peg$c126);
              }
            }
          }
        }
      }
    }
    peg$silentFails--;
    if (s0 === peg$FAILED) {
      if (peg$silentFails === 0) {
        peg$fail(peg$c116);
      }
    }
    return s0;
  }
  function peg$parseSourceCharacter() {
    let s0;
    if (input.length > peg$currPos) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c78);
      }
    }
    return s0;
  }
  function peg$parse_() {
    let s0, s1;
    peg$silentFails++;
    s0 = [];
    s1 = peg$parse_whitespace();
    while (s1 !== peg$FAILED) {
      s0.push(s1);
      s1 = peg$parse_whitespace();
    }
    peg$silentFails--;
    if (s0 === peg$FAILED) {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c134);
      }
    }
    return s0;
  }
  function peg$parse__() {
    let s0, s1;
    peg$silentFails++;
    s0 = [];
    s1 = peg$parse_whitespace();
    if (s1 === peg$FAILED) {
      s1 = peg$parse_newline();
    }
    while (s1 !== peg$FAILED) {
      s0.push(s1);
      s1 = peg$parse_whitespace();
      if (s1 === peg$FAILED) {
        s1 = peg$parse_newline();
      }
    }
    peg$silentFails--;
    if (s0 === peg$FAILED) {
      s1 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c135);
      }
    }
    return s0;
  }
  function peg$parse_newline() {
    let s0;
    if (peg$c136.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c137);
      }
    }
    return s0;
  }
  function peg$parse_whitespace() {
    let s0;
    if (peg$c138.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c139);
      }
    }
    return s0;
  }
  function peg$parseUnicodeLetter() {
    let s0;
    s0 = peg$parseLu();
    if (s0 === peg$FAILED) {
      s0 = peg$parseLl();
      if (s0 === peg$FAILED) {
        s0 = peg$parseLt();
        if (s0 === peg$FAILED) {
          s0 = peg$parseLm();
          if (s0 === peg$FAILED) {
            s0 = peg$parseLo();
            if (s0 === peg$FAILED) {
              s0 = peg$parseNl();
            }
          }
        }
      }
    }
    return s0;
  }
  function peg$parseLl() {
    let s0;
    if (peg$c140.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c141);
      }
    }
    return s0;
  }
  function peg$parseLm() {
    let s0;
    if (peg$c142.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c143);
      }
    }
    return s0;
  }
  function peg$parseLo() {
    let s0;
    if (peg$c144.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c145);
      }
    }
    return s0;
  }
  function peg$parseLt() {
    let s0;
    if (peg$c146.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c147);
      }
    }
    return s0;
  }
  function peg$parseLu() {
    let s0;
    if (peg$c148.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c149);
      }
    }
    return s0;
  }
  function peg$parseNl() {
    let s0;
    if (peg$c150.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c151);
      }
    }
    return s0;
  }
  function peg$parseNd() {
    let s0;
    if (peg$c152.test(input.charAt(peg$currPos))) {
      s0 = input.charAt(peg$currPos);
      peg$currPos++;
    } else {
      s0 = peg$FAILED;
      if (peg$silentFails === 0) {
        peg$fail(peg$c153);
      }
    }
    return s0;
  }
  function dedent(value) {
    const str = value.trim();
    const matches = str.match(/\n([\t ]+|(?!\s).)/g);
    if (matches) {
      const indentLengths = matches.map((match) => match.match(/[\t ]/g)?.length ?? 0);
      const pattern = new RegExp(`\n[\t ]{${Math.min(...indentLengths)}}`, 'g');
      return str.replace(pattern, '\n');
    }
    return str;
  }
  const edgeops = [];
  const b = new Builder({
    locationFunction: location,
  });
  peg$result = peg$startRuleFunction();
  if (peg$result !== peg$FAILED && peg$currPos === input.length) {
    return peg$result;
  } else {
    if (peg$result !== peg$FAILED && peg$currPos < input.length) {
      peg$fail(peg$endExpectation());
    }
    throw peg$buildStructuredError(
      peg$maxFailExpected,
      peg$maxFailPos < input.length ? input.charAt(peg$maxFailPos) : null,
      peg$maxFailPos < input.length
        ? peg$computeLocation(peg$maxFailPos, peg$maxFailPos + 1)
        : peg$computeLocation(peg$maxFailPos, peg$maxFailPos),
    );
  }
}
const parse$1 = peg$parse;

function parse(input, options) {
  return parse$1(input, options);
}
const DotSyntaxError = DotSyntaxError$1;

function convertAttribute(key, value) {
  if (typeof value === 'string') {
    const trimmed = value.trim();
    const isHTMLLike = /^<.+>$/ms.test(trimmed);
    if (isHTMLLike) {
      return createElement(
        'Attribute',
        {
          key: createElement('Literal', { value: key, quoted: false }, []),
          value: createElement('Literal', { value: trimmed.slice(1, trimmed.length - 1), quoted: 'html' }, []),
        },
        [],
      );
    } else {
      return createElement(
        'Attribute',
        {
          key: createElement('Literal', { value: key, quoted: false }, []),
          value: createElement('Literal', { value: value, quoted: true }, []),
        },
        [],
      );
    }
  }
  return createElement(
    'Attribute',
    {
      key: createElement('Literal', { value: key, quoted: false }, []),
      value: createElement('Literal', { value: String(value), quoted: false }, []),
    },
    [],
  );
}

function convertComment(value, kind) {
  return createElement(
    'Comment',
    {
      kind: kind,
      value: value,
    },
    [],
  );
}

function convertClusterChildren(context, model) {
  return Array.from(
    (function* () {
      for (const [key, value] of model.values) {
        yield convertAttribute(key, value);
      }
      for (const attrs of Object.values(model.attributes)) {
        if (attrs.size > 0) {
          if (attrs.comment) {
            yield convertComment(attrs.comment, context.commentKind);
          }
          yield context.convert(attrs);
        }
      }
      for (const node of model.nodes) {
        if (node.comment) {
          yield convertComment(node.comment, context.commentKind);
        }
        yield context.convert(node);
      }
      for (const subgraph of model.subgraphs) {
        if (subgraph.comment) {
          yield convertComment(subgraph.comment, context.commentKind);
        }
        yield context.convert(subgraph);
      }
      for (const edge of model.edges) {
        if (edge.comment) {
          yield convertComment(edge.comment, context.commentKind);
        }
        yield context.convert(edge);
      }
    })(),
  );
}

const AttributeListPlugin = {
  match(model) {
    return model.$$type === 'AttributeList';
  },
  convert(context, model) {
    return createElement(
      'AttributeList',
      {
        kind: model.$$kind,
      },
      model.values.map(([key, value]) => convertAttribute(key, value)),
    );
  },
};

const EdgePlugin$1 = {
  match(model) {
    return model.$$type === 'Edge';
  },
  convert(context, model) {
    return createElement(
      'Edge',
      {
        targets: model.targets.map((target) => {
          if (index_js$1.isNodeModel(target)) {
            return createElement(
              'NodeRef',
              {
                id: createElement(
                  'Literal',
                  {
                    value: target.id,
                    quoted: true,
                  },
                  [],
                ),
              },
              [],
            );
          } else if (index_js$1.isForwardRefNode(target)) {
            return createElement(
              'NodeRef',
              {
                id: createElement(
                  'Literal',
                  {
                    value: target.id,
                    quoted: true,
                  },
                  [],
                ),
                port: target.port
                  ? createElement(
                      'Literal',
                      {
                        value: target.port,
                        quoted: true,
                      },
                      [],
                    )
                  : undefined,
                compass: target.compass
                  ? createElement(
                      'Literal',
                      {
                        value: target.compass,
                        quoted: true,
                      },
                      [],
                    )
                  : undefined,
              },
              [],
            );
          } else {
            return createElement(
              'NodeRefGroup',
              {},
              target.map((n) => {
                if (index_js$1.isNodeModel(n)) {
                  return createElement(
                    'NodeRef',
                    {
                      id: createElement(
                        'Literal',
                        {
                          value: n.id,
                          quoted: true,
                        },
                        [],
                      ),
                    },
                    [],
                  );
                }
                return createElement(
                  'NodeRef',
                  {
                    id: createElement(
                      'Literal',
                      {
                        value: n.id,
                        quoted: true,
                      },
                      [],
                    ),
                    port: n.port
                      ? createElement(
                          'Literal',
                          {
                            value: n.port,
                            quoted: true,
                          },
                          [],
                        )
                      : undefined,
                    compass: n.compass
                      ? createElement(
                          'Literal',
                          {
                            value: n.compass,
                            quoted: true,
                          },
                          [],
                        )
                      : undefined,
                  },
                  [],
                );
              }),
            );
          }
        }),
      },
      [
        ...(model.attributes.comment ? [convertComment(model.attributes.comment, context.commentKind)] : []),
        ...model.attributes.values.map(([key, value]) => convertAttribute(key, value)),
      ],
    );
  },
};

const GraphPlugin$1 = {
  match(model) {
    return model.$$type === 'Graph';
  },
  convert(context, model) {
    return createElement('Dot', {}, [
      ...(model.comment ? [convertComment(model.comment, context.commentKind)] : []),
      createElement(
        'Graph',
        {
          directed: model.directed,
          strict: model.strict,
          id: model.id
            ? createElement(
                'Literal',
                {
                  value: model.id,
                  quoted: true,
                },
                [],
              )
            : undefined,
        },
        convertClusterChildren(context, model),
      ),
    ]);
  },
};

const NodePlugin$1 = {
  match(model) {
    return model.$$type === 'Node';
  },
  convert(context, model) {
    return createElement(
      'Node',
      {
        id: createElement(
          'Literal',
          {
            value: model.id,
            quoted: true,
          },
          [],
        ),
      },
      [
        ...(model.attributes.comment ? [convertComment(model.attributes.comment, context.commentKind)] : []),
        ...model.attributes.values.map(([key, value]) => convertAttribute(key, value)),
      ],
    );
  },
};

const SubgraphPlugin$1 = {
  match(model) {
    return model.$$type === 'Subgraph';
  },
  convert(context, model) {
    return createElement(
      'Subgraph',
      {
        id: model.id
          ? createElement(
              'Literal',
              {
                value: model.id,
                quoted: true,
              },
              [],
            )
          : undefined,
      },
      convertClusterChildren(context, model),
    );
  },
};

const defaultPlugins$1 = [AttributeListPlugin, EdgePlugin$1, NodePlugin$1, GraphPlugin$1, SubgraphPlugin$1];

class FromModelConverter {
  options;
  #plugins = [...defaultPlugins$1];
  constructor(options = {}) {
    this.options = options;
  }
  convert(model) {
    const plugins = [...this.#plugins];
    const { commentKind = 'Slash' } = this.options;
    const context = {
      commentKind,
      convert(m) {
        for (const plugin of plugins) {
          if (plugin.match(m)) {
            return plugin.convert(context, m);
          }
        }
        throw Error();
      },
    };
    return context.convert(model);
  }
}

function fromModel(model, options) {
  return new FromModelConverter(options).convert(model);
}

class CommentHolder {
  comment = null;
  set(comment) {
    this.comment = comment;
  }
  reset() {
    this.comment = null;
  }
  apply(model, location) {
    if (location && this.comment?.location) {
      if (this.comment?.kind === 'Block') {
        if (this.comment.location.end.line === location.start.line - 1) {
          model.comment = this.comment.value;
        }
      } else {
        if (this.comment.location.end.line === location.start.line) {
          model.comment = this.comment.value;
        }
      }
    } else {
      model.comment = this.comment?.value;
    }
    this.reset();
  }
}

const DotPlugin = {
  match(ast) {
    return ast.type === 'Dot';
  },
  convert(context, ast) {
    const commentHolder = new CommentHolder();
    for (const stmt of ast.children) {
      switch (stmt.type) {
        case 'Comment':
          commentHolder.set(stmt);
          break;
        case 'Graph':
          const graph = context.convert(stmt);
          commentHolder.apply(graph, stmt.location);
          return graph;
      }
    }
    throw Error();
  },
};

function convertToEdgeTargetTuple(edge) {
  return edge.targets.map((t) => {
    switch (t.type) {
      case 'NodeRef':
        return { id: t.id.value, port: t.port?.value, compass: t.compass?.value };
      case 'NodeRefGroup':
        return t.children.map((t) => ({ id: t.id.value, port: t.port?.value, compass: t.compass?.value }));
    }
  });
}

const EdgePlugin = {
  match(ast) {
    return ast.type === 'Edge';
  },
  convert(context, ast) {
    const edge = new context.models.Edge(
      convertToEdgeTargetTuple(ast),
      ast.children
        .filter((v) => v.type === 'Attribute')
        .reduce((prev, curr) => ({ ...prev, [curr.key.value]: curr.value.value }), {}),
    );
    return edge;
  },
};

function applyStatements(graph, statements) {
  const commentHolder = new CommentHolder();
  for (const stmt of statements) {
    switch (stmt.type) {
      case 'Subgraph':
        const subgraph = stmt.id ? graph.subgraph(stmt.id.value) : graph.subgraph();
        applyStatements(subgraph, stmt.children);
        commentHolder.apply(subgraph, stmt.location);
        break;
      case 'Attribute':
        graph.set(stmt.key.value, stmt.value.value);
        commentHolder.reset();
        break;
      case 'Node':
        commentHolder.apply(
          graph.node(
            stmt.id.value,
            stmt.children
              .filter((v) => v.type === 'Attribute')
              .reduce((prev, curr) => ({ ...prev, [curr.key.value]: curr.value.value }), {}),
          ),
          stmt.location,
        );
        break;
      case 'Edge':
        commentHolder.apply(
          graph.edge(
            convertToEdgeTargetTuple(stmt),
            stmt.children
              .filter((v) => v.type === 'Attribute')
              .reduce((prev, curr) => ({ ...prev, [curr.key.value]: curr.value.value }), {}),
          ),
          stmt.location,
        );
        break;
      case 'AttributeList':
        const attrs = stmt.children
          .filter((v) => v.type === 'Attribute')
          .reduce((prev, curr) => ({ ...prev, [curr.key.value]: curr.value.value }), {});
        switch (stmt.kind) {
          case 'Edge':
            graph.edge(attrs);
            break;
          case 'Node':
            graph.node(attrs);
            break;
          case 'Graph':
            graph.graph(attrs);
            break;
        }
        commentHolder.reset();
        break;
      case 'Comment':
        commentHolder.set(stmt);
    }
  }
}

const GraphPlugin = {
  match(ast) {
    return ast.type === 'Graph';
  },
  convert(context, ast) {
    const G = ast.directed ? context.models.Digraph : context.models.Graph;
    const graph = new G(ast.id?.value, ast.strict);
    applyStatements(graph, ast.children);
    return graph;
  },
};

const SubgraphPlugin = {
  match(ast) {
    return ast.type === 'Subgraph';
  },
  convert(context, ast) {
    const subgraph = new context.models.Subgraph(ast.id?.value);
    applyStatements(subgraph, ast.children);
    return subgraph;
  },
};

const NodePlugin = {
  match(ast) {
    return ast.type === 'Node';
  },
  convert(context, ast) {
    const node = new context.models.Node(
      ast.id.value,
      ast.children
        .filter((v) => v.type === 'Attribute')
        .reduce((prev, curr) => ({ ...prev, [curr.key.value]: curr.value.value }), {}),
    );
    return node;
  },
};

const defaultPlugins = [NodePlugin, EdgePlugin, SubgraphPlugin, GraphPlugin, DotPlugin];

class ToModelConverter {
  options;
  plugins = [...defaultPlugins];
  constructor(options = {}) {
    this.options = options;
  }
  convert(ast) {
    const plugins = [...this.plugins];
    const context = {
      models: index_js$1.createModelsContext(this.options.models ?? {}),
      convert(m) {
        for (const plugin of plugins) {
          if (plugin.match(m)) {
            return plugin.convert(context, m);
          }
        }
        throw Error();
      },
    };
    return context.convert(ast);
  }
}

function toModel(ast, options) {
  return new ToModelConverter(options).convert(ast);
}

exports.Builder = Builder;
exports.DotSyntaxError = DotSyntaxError;
exports.FromModelConverter = FromModelConverter;
exports.Printer = Printer;
exports.ToModelConverter = ToModelConverter;
exports.createElement = createElement;
exports.fromModel = fromModel;
exports.parse = parse;
exports.stringify = stringify;
exports.toModel = toModel;
