import { RemovalPolicy, Stage } from "aws-cdk-lib";
import { IRepository } from "aws-cdk-lib/aws-codecommit";
import { AddStageOpts, CodePipeline, ShellStepProps, StageDeployment } from "aws-cdk-lib/pipelines";
import { Construct, Node } from "constructs";
import { SonarCodeScannerConfig } from "./code_scanner/sonar-code-scanner";
import { CodePipelineProps as _CodePipelineProps } from "./codepipeline-props";
export * from "./code_scanner/sonar-code-scanner";
export declare const DEFAULT_BRANCH_NAME = "mainline";
/**
 * Properties to configure the PDKPipeline.
 *
 * Note: Due to limitations with JSII and generic support it should be noted that
 * the synth, synthShellStepPartialProps.input and
 * synthShellStepPartialProps.primaryOutputDirectory properties will be ignored
 * if passed in to this construct.
 *
 * synthShellStepPartialProps.commands is marked as a required field, however
 * if you pass in [] the default commands of this construct will be retained.
 */
export interface PDKPipelineProps extends _CodePipelineProps {
    /**
     * Name of the CodeCommit repository to create.
     */
    readonly repositoryName: string;
    /**
     * Output directory for cdk synthesized artifacts i.e: packages/infra/cdk.out.
     */
    readonly primarySynthDirectory: string;
    /**
     * PDKPipeline by default assumes a NX Monorepo structure for it's codebase and
     * uses sane defaults for the install and run commands. To override these defaults
     * and/or provide additional inputs, specify env settings, etc you can provide
     * a partial ShellStepProps.
     */
    readonly synthShellStepPartialProps?: ShellStepProps;
    /**
     * Branch to trigger the pipeline execution.
     *
     * @default mainline
     */
    readonly defaultBranchName?: string;
    /**
     * Configuration for enabling Sonarqube code scanning on a successful synth.
     *
     * @default undefined
     */
    readonly sonarCodeScannerConfig?: SonarCodeScannerConfig;
    /**
     * Possible values for a resource's Removal Policy
     * The removal policy controls what happens to the resource if it stops being managed by CloudFormation.
     */
    readonly codeCommitRemovalPolicy?: RemovalPolicy;
    /**
     * Branch name prefixes
     * Any branches created matching this list of prefixes will create a new pipeline and stack.
     *
     * @example
     * // Creates a new pipeline and stack for any branch
     * new PDKPipeline(this, 'PDKPipeline', {
     *   repositoryName: 'my-repo',
     *   branchNamePrefixes: PDKPipeline.ALL_BRANCHES,
     * }
     * @example
     * // Creates a new pipeline and stack for any branch starting with 'feature/' or 'fix/'
     * new PDKPipeline(this, 'PDKPipeline', {
     *   repositoryName: 'my-repo',
     *   branchNamePrefixes: ['feature/', 'fix/'],
     * }
     * @example
     * // Disables feature branches (default)
     * new PDKPipeline(this, 'PDKPipeline', {
     *   repositoryName: 'my-repo',
     *   branchNamePrefixes: [], // or simply exclude this line
     * }
     * @default undefined
     */
    readonly branchNamePrefixes?: string[];
    /**
     * The directory with `cdk.json` to run cdk synth from. Set this if you enabled
     * feature branches and `cdk.json` is not located in the parent directory of
     * `primarySynthDirectory`.
     *
     * @default The parent directory of `primarySynthDirectory`
     */
    readonly cdkSrcDir?: string;
    /**
     * CDK command. Override the command used to call cdk for synth and deploy.
     *
     * @default 'npx cdk'
     */
    readonly cdkCommand?: string;
}
/**
 * Properties to help the isDefaultBranch function determine the default branch name.
 */
export interface IsDefaultBranchProps {
    /**
     * The current node to fetch defaultBranchName from context.
     */
    readonly node?: Node;
    /**
     * Specify the default branch name without context.
     */
    readonly defaultBranchName?: string;
}
/**
 * An extension to CodePipeline which configures sane defaults for a NX Monorepo
 * codebase. In addition to this, it also creates a CodeCommit repository with
 * automated PR builds and approvals.
 */
export declare class PDKPipeline extends Construct {
    static readonly ALL_BRANCHES: string[];
    static readonly defaultBranchName = "mainline";
    /**
     * A helper function to normalize the branch name with only alphanumeric characters and hypens ('-').
     * @param branchName The name of the branch to normalize.
     * @returns The normalized branch name.
     */
    static normalizeBranchName(branchName: string): string;
    /**
     * A helper function to determine if the current branch is the default branch.
     *
     * If there is no BRANCH environment variable, then assume this is the default
     * branch. Otherwise, check that BRANCH matches the default branch name.
     *
     * The default branch name is determined in the following priority:
     *
     * 1. defaultBranchName property
     * 2. defaultBranchName context
     * 3. PDKPipeline.defaultBranchName constant
     *
     * @param props? {
     *    defaultBranchName? Specify the default branch name without context.
     *    node? The current app to fetch defaultBranchName from context.
     * }
     * @returns True if the current branch is the default branch.
     */
    static isDefaultBranch(props?: IsDefaultBranchProps): boolean;
    /**
     * A helper function to create a branch prefix. The prefix is empty on the default branch.
     * @param props? {
     *    defaultBranchName? Specify the default branch name without context.
     *    node? The current app to fetch defaultBranchName from context.
     * }
     * @returns The branch prefix.
     */
    static getBranchPrefix(props?: IsDefaultBranchProps): string;
    readonly codePipeline: CodePipeline;
    readonly codeRepository: IRepository;
    private readonly sonarCodeScannerConfig?;
    private readonly branchNamePrefixes?;
    private readonly defaultBranchName?;
    private readonly repositoryName;
    constructor(scope: Construct, id: string, props: PDKPipelineProps);
    /**
     * @inheritDoc
     */
    addStage(stage: Stage, options?: AddStageOpts): StageDeployment;
    buildPipeline(): void;
    suppressCDKViolations(): void;
    private suppressRules;
}
