"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const crypto = require("crypto");
const client_s3_1 = require("@aws-sdk/client-s3");
const prepare_spec_1 = require("./prepare-spec");
const s3 = new client_s3_1.S3Client({
    customUserAgent: `aws-pdk/type-safe-api/prepare-spec`,
});
/**
 * Prepare the api spec for API Gateway
 * @param inputSpecLocation location of the specification to prepare
 * @param outputSpecLocation location to write the prepared spec to
 * @param options integrations, authorizers etc to apply
 * @return the output location of the prepared spec
 */
const prepare = async ({ inputSpecLocation, outputSpecLocation, ...options }) => {
    // Read the spec from the s3 input location
    const inputSpec = JSON.parse(await (await s3.send(new client_s3_1.GetObjectCommand({
        Bucket: inputSpecLocation.bucket,
        Key: inputSpecLocation.key,
    }))).Body.transformToString("utf-8"));
    // Prepare the spec
    const preparedSpec = (0, prepare_spec_1.prepareApiSpec)(inputSpec, options);
    const preparedSpecHash = crypto
        .createHash("sha256")
        .update(JSON.stringify(preparedSpec))
        .digest("hex");
    const outputLocation = {
        bucket: outputSpecLocation.bucket,
        key: `${outputSpecLocation.key}/${preparedSpecHash}.json`,
    };
    // Write the spec to the s3 output location
    await s3.send(new client_s3_1.PutObjectCommand({
        Bucket: outputLocation.bucket,
        Key: outputLocation.key,
        Body: JSON.stringify(preparedSpec),
    }));
    return outputLocation;
};
exports.handler = async (event) => {
    switch (event.RequestType) {
        case "Create":
        case "Update":
            // Prepare the spec on create
            const outputLocation = await prepare(event.ResourceProperties);
            return {
                PhysicalResourceId: outputLocation.key,
                Status: "SUCCESS",
                Data: {
                    outputSpecKey: outputLocation.key,
                },
            };
        case "Delete":
        // Nothing to do for delete
        default:
            break;
    }
    return {
        PhysicalResourceId: event.PhysicalResourceId,
        Status: "SUCCESS",
    };
};
//# sourceMappingURL=data:application/json;base64,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