"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeWebsocketApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const pdk_nag_1 = require("../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigatewayv2_1 = require("aws-cdk-lib/aws-apigatewayv2");
const aws_apigatewayv2_integrations_1 = require("aws-cdk-lib/aws-apigatewayv2-integrations");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const websocket_schema_1 = require("./prepare-spec-event-handler/websocket-schema");
/**
 * A construct for creating a websocket API, based on the provided spec and integrations
 */
class TypeSafeWebsocketApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._props = props;
        // Create the WebSocket API
        this.api = new aws_apigatewayv2_1.WebSocketApi(this, id, {
            ...props,
            routeSelectionExpression: "$request.body.route",
        });
        // Add the connect/disconnect routes
        this.addRoute("$connect", {
            integration: props.connect?.integration ??
                new aws_apigatewayv2_integrations_1.WebSocketMockIntegration("ConnectIntegration"),
            authorizer: props.authorizer,
        });
        const disconnectRoute = this.addRoute("$disconnect", {
            integration: props.connect?.integration ??
                new aws_apigatewayv2_integrations_1.WebSocketMockIntegration("DisconnectIntegration"),
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(disconnectRoute, ["AwsPrototyping-APIGWAuthorization", "AwsSolutions-APIG4"].map((ruleId) => ({
            id: ruleId,
            reason: `Authorizers only apply to the $connect route`,
        })), true);
        // Create a default stage
        this.defaultStage = new aws_apigatewayv2_1.WebSocketStage(this, "default", {
            webSocketApi: this.api,
            autoDeploy: true,
            stageName: "default",
            ...props.stageProps,
        });
        // Enable execution logs by default
        this.defaultStage.node.defaultChild.defaultRouteSettings = {
            loggingLevel: "INFO",
            dataTraceEnabled: false,
        };
        // Enable access logging by default
        if (!props.disableAccessLogging) {
            const logGroup = new aws_logs_1.LogGroup(this, `AccessLogs`);
            this.defaultStage.node.defaultChild.accessLogSettings = {
                destinationArn: logGroup.logGroupArn,
                format: `$context.identity.sourceIp - - [$context.requestTime] "$context.httpMethod $context.routeKey $context.protocol" $context.status $context.responseLength $context.requestId`,
            };
        }
        const lambdaHandlers = [
            ...Object.values(props.integrations),
            props.connect,
            props.disconnect,
        ].flatMap((integration) => integration?.integration instanceof aws_apigatewayv2_integrations_1.WebSocketLambdaIntegration &&
            integration.integration.handler?.grantPrincipal
            ? [integration.integration.handler]
            : []);
        const stack = aws_cdk_lib_1.Stack.of(this);
        // By default, grant lambda handlers access to the management api
        if (!props.disableGrantManagementAccessToLambdas) {
            lambdaHandlers.forEach((fn) => {
                this.defaultStage.grantManagementApiAccess(fn);
                cdk_nag_1.NagSuppressions.addResourceSuppressions(fn, ["AwsPrototyping-IAMNoWildcardPermissions", "AwsSolutions-IAM5"].map((ruleId) => ({
                    id: ruleId,
                    reason: "WebSocket handlers are granted permissions to manage arbitrary connections",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:execute-api:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:.*\\/${this.defaultStage.stageName}\\/\\*\\/@connections\\/\\*$/g`,
                        },
                    ],
                })), true);
            });
        }
        // Where the same function is used for multiple routes, grant permission for API gateway to invoke
        // the lambda for all routes
        const uniqueLambdaHandlers = new Set();
        const duplicateLambdaHandlers = new Set();
        lambdaHandlers.forEach((fn) => {
            if (uniqueLambdaHandlers.has(fn)) {
                duplicateLambdaHandlers.add(fn);
            }
            uniqueLambdaHandlers.add(fn);
        });
        [...duplicateLambdaHandlers].forEach((fn, i) => {
            new aws_lambda_1.CfnPermission(this, `GrantRouteInvoke${i}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: fn.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.apiId,
                    resourceName: "*",
                }),
            });
        });
        // Read and parse the spec
        const spec = JSON.parse(fs.readFileSync(props.specPath, "utf-8"));
        // Map of route key to paths
        const serverOperationPaths = Object.fromEntries(Object.values(props.operationLookup).map((details) => [
            details.path.replace(/\//g, ""),
            details.path,
        ]));
        // Locally check that we can extract the schema for every operation
        const schemas = (0, websocket_schema_1.extractWebSocketSchemas)(Object.keys(serverOperationPaths), serverOperationPaths, spec);
        // Check that every operation has an integration
        const missingIntegrations = Object.keys(props.operationLookup).filter((operationId) => !props.integrations[operationId]);
        if (missingIntegrations.length > 0) {
            throw new Error(`Missing integrations for operations ${missingIntegrations.join(", ")}`);
        }
        // Create an asset for the spec, which we'll read from the custom resource
        const inputSpec = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: props.specPath,
        });
        // Function for managing schemas/models associated with routes
        const schemaHandler = new aws_lambda_1.Function(this, "SchemaHandler", {
            handler: "websocket-schema-handler.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "./prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.minutes(1),
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(schemaHandler, ["AwsPrototyping-IAMNoManagedPolicies", "AwsSolutions-IAM4"].map((ruleId) => ({
            id: ruleId,
            reason: `AWSLambdaBasicExecutionRole grants minimal permissions required for lambda execution`,
        })), true);
        schemaHandler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject"],
            resources: [inputSpec.bucket.arnForObjects(inputSpec.s3ObjectKey)],
        }));
        schemaHandler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "apigateway:DELETE",
                "apigateway:PATCH",
                "apigateway:POST",
                "apigateway:GET",
            ],
            resources: [
                stack.formatArn({
                    service: "apigateway",
                    account: "",
                    resource: `/apis/${this.api.apiId}/models`,
                }),
                stack.formatArn({
                    service: "apigateway",
                    account: "",
                    resource: `/apis/${this.api.apiId}/models/*`,
                }),
            ],
        }));
        schemaHandler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["apigateway:PATCH", "apigateway:GET"],
            resources: [
                stack.formatArn({
                    service: "apigateway",
                    account: "",
                    resource: `/apis/${this.api.apiId}/routes`,
                }),
                stack.formatArn({
                    service: "apigateway",
                    account: "",
                    resource: `/apis/${this.api.apiId}/routes/*`,
                }),
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(schemaHandler, ["AwsPrototyping-IAMNoWildcardPermissions", "AwsSolutions-IAM5"].map((ruleId) => ({
            id: ruleId,
            reason: `Schema custom resource manages all routes and models`,
        })), true);
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "SchemaProvider", {
            onEventHandler: schemaHandler,
            role: providerRole,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, ["AwsPrototyping-IAMNoWildcardPermissions", "AwsSolutions-IAM5"].map((ruleId) => ({
            id: ruleId,
            reason: `Custom resource provider may invoke arbitrary lambda versions`,
        })), true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, ["AwsPrototyping-LambdaLatestVersion", "AwsSolutions-L1"].map((ruleId) => ({
            id: ruleId,
            reason: `Provider framework lambda is managed by CDK`,
        })), true);
        const schemaCustomResourceProperties = {
            apiId: this.api.apiId,
            inputSpecLocation: {
                bucket: inputSpec.s3BucketName,
                key: inputSpec.s3ObjectKey,
            },
            serverOperationPaths,
        };
        const schemaCustomResource = new aws_cdk_lib_1.CustomResource(this, "SchemaCustomResource", {
            serviceToken: provider.serviceToken,
            properties: schemaCustomResourceProperties,
        });
        // Add a route for every integration
        Object.entries(props.integrations).forEach(([operationId, integration]) => {
            const op = props.operationLookup[operationId];
            if (!op) {
                throw new Error(`Integration not found in operation lookup for operation ${operationId}`);
            }
            // Add the route
            const routeKey = op.path.replace(/\//g, "");
            const route = this.addRoute(routeKey, {
                integration: integration.integration,
            });
            cdk_nag_1.NagSuppressions.addResourceSuppressions(route, ["AwsPrototyping-APIGWAuthorization", "AwsSolutions-APIG4"].map((ruleId) => ({
                id: ruleId,
                reason: `Authorizers only apply to the $connect route`,
            })), true);
            // Associate the route with its corresponding schema (which is created by the custom resource)
            if (schemas[routeKey]) {
                route.node.defaultChild.requestModels = {
                    model: routeKey,
                };
                route.node.defaultChild.modelSelectionExpression =
                    "model";
            }
            route.node.addDependency(schemaCustomResource);
        });
    }
    /**
     * Add a route to the websocket api
     */
    addRoute(routeKey, options) {
        // Unless disableMockIntegrationResponses is true, we automatically configure the integration requests and responses
        // required to successfully mock the route, when the integration is a mock integration
        const shouldAddMockResponse = !this._props.disableMockIntegrationResponses &&
            options.integration instanceof aws_apigatewayv2_integrations_1.WebSocketMockIntegration;
        const route = this.api.addRoute(routeKey, {
            ...options,
            returnResponse: shouldAddMockResponse,
        });
        if (shouldAddMockResponse &&
            options.integration?.integration
                ?.integrationId) {
            const integration = options.integration
                ?.integration;
            integration.node.defaultChild.requestTemplates = {
                "application/json": '{"statusCode":200}',
            };
            new aws_apigatewayv2_1.CfnIntegrationResponse(this, `${routeKey}IntegRes`, {
                apiId: this.api.apiId,
                integrationId: integration.integrationId,
                integrationResponseKey: "/2\\d\\d/",
                templateSelectionExpression: "/2\\d\\d/",
                responseTemplates: {
                    "200": '{"statusCode":200}',
                },
            });
        }
        return route;
    }
}
exports.TypeSafeWebsocketApi = TypeSafeWebsocketApi;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeWebsocketApi[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeWebsocketApi", version: "0.23.72" };
//# sourceMappingURL=data:application/json;base64,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