"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeApiCommandEnvironment = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
const typescript_1 = require("projen/lib/typescript");
const utils_1 = require("./utils");
/**
 * Component for setting up the environemnt for type safe api commands
 */
class TypeSafeApiCommandEnvironment extends projen_1.Component {
    /**
     * Retrieves an instance of TypeSafeApiCommandEnvironment if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => monorepo_1.ProjectUtils.isNamedInstanceOf(c, TypeSafeApiCommandEnvironment));
    }
    /**
     * Retrieves an instance of TypeSafeApiCommandEnvironment if one is associated to the given project,
     * otherwise creates a TypeSafeApiCommandEnvironment instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project) {
        return (TypeSafeApiCommandEnvironment.of(project) ||
            new TypeSafeApiCommandEnvironment(project));
    }
    constructor(project) {
        super(project);
        // Add all environment variables to the task environment
        Object.entries((0, utils_1.getTypeSafeApiTaskEnvironment)()).forEach(([key, value]) => project.tasks.addEnvironment(key, value));
        // TypeScript projects need a dev dependency on PDK to ensure npx resolves to the correct pdk
        if (monorepo_1.ProjectUtils.isNamedInstanceOf(project, typescript_1.TypeScriptProject)) {
            project.addDevDeps("@aws/pdk@^0");
        }
    }
}
exports.TypeSafeApiCommandEnvironment = TypeSafeApiCommandEnvironment;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZS1zYWZlLWFwaS1jb21tYW5kLWVudmlyb25tZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidHlwZS1zYWZlLWFwaS1jb21tYW5kLWVudmlyb25tZW50LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBO3NDQUNzQztBQUN0Qyw0Q0FBNkM7QUFDN0MsbUNBQTRDO0FBQzVDLHNEQUEwRDtBQUMxRCxtQ0FBd0Q7QUFFeEQ7O0dBRUc7QUFDSCxNQUFhLDZCQUE4QixTQUFRLGtCQUFTO0lBQzFEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsRUFBRSxDQUFDLE9BQWdCO1FBQ3hCLE9BQU8sT0FBTyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUNuQyx1QkFBWSxDQUFDLGlCQUFpQixDQUFDLENBQUMsRUFBRSw2QkFBNkIsQ0FBQyxDQUNwQixDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7T0FLRztJQUNILE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBZ0I7UUFDNUIsT0FBTyxDQUNMLDZCQUE2QixDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUM7WUFDekMsSUFBSSw2QkFBNkIsQ0FBQyxPQUFPLENBQUMsQ0FDM0MsQ0FBQztJQUNKLENBQUM7SUFFRCxZQUFZLE9BQWdCO1FBQzFCLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVmLHdEQUF3RDtRQUN4RCxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUEscUNBQTZCLEdBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FDdkUsT0FBTyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUN6QyxDQUFDO1FBRUYsNkZBQTZGO1FBQzdGLElBQUksdUJBQVksQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLEVBQUUsOEJBQWlCLENBQUMsRUFBRSxDQUFDO1lBQy9ELE9BQU8sQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDcEMsQ0FBQztJQUNILENBQUM7Q0FDRjtBQXRDRCxzRUFzQ0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCB7IFByb2plY3RVdGlscyB9IGZyb20gXCJAYXdzL21vbm9yZXBvXCI7XG5pbXBvcnQgeyBDb21wb25lbnQsIFByb2plY3QgfSBmcm9tIFwicHJvamVuXCI7XG5pbXBvcnQgeyBUeXBlU2NyaXB0UHJvamVjdCB9IGZyb20gXCJwcm9qZW4vbGliL3R5cGVzY3JpcHRcIjtcbmltcG9ydCB7IGdldFR5cGVTYWZlQXBpVGFza0Vudmlyb25tZW50IH0gZnJvbSBcIi4vdXRpbHNcIjtcblxuLyoqXG4gKiBDb21wb25lbnQgZm9yIHNldHRpbmcgdXAgdGhlIGVudmlyb25lbW50IGZvciB0eXBlIHNhZmUgYXBpIGNvbW1hbmRzXG4gKi9cbmV4cG9ydCBjbGFzcyBUeXBlU2FmZUFwaUNvbW1hbmRFbnZpcm9ubWVudCBleHRlbmRzIENvbXBvbmVudCB7XG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYW4gaW5zdGFuY2Ugb2YgVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQgaWYgb25lIGlzIGFzc29jaWF0ZWQgdG8gdGhlIGdpdmVuIHByb2plY3QuXG4gICAqXG4gICAqIEBwYXJhbSBwcm9qZWN0IHByb2plY3QgaW5zdGFuY2UuXG4gICAqL1xuICBzdGF0aWMgb2YocHJvamVjdDogUHJvamVjdCk6IFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50IHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gcHJvamVjdC5jb21wb25lbnRzLmZpbmQoKGMpID0+XG4gICAgICBQcm9qZWN0VXRpbHMuaXNOYW1lZEluc3RhbmNlT2YoYywgVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQpXG4gICAgKSBhcyBUeXBlU2FmZUFwaUNvbW1hbmRFbnZpcm9ubWVudCB8IHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYW4gaW5zdGFuY2Ugb2YgVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQgaWYgb25lIGlzIGFzc29jaWF0ZWQgdG8gdGhlIGdpdmVuIHByb2plY3QsXG4gICAqIG90aGVyd2lzZSBjcmVhdGVzIGEgVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQgaW5zdGFuY2UgZm9yIHRoZSBwcm9qZWN0LlxuICAgKlxuICAgKiBAcGFyYW0gcHJvamVjdCBwcm9qZWN0IGluc3RhbmNlLlxuICAgKi9cbiAgc3RhdGljIGVuc3VyZShwcm9qZWN0OiBQcm9qZWN0KTogVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQge1xuICAgIHJldHVybiAoXG4gICAgICBUeXBlU2FmZUFwaUNvbW1hbmRFbnZpcm9ubWVudC5vZihwcm9qZWN0KSB8fFxuICAgICAgbmV3IFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50KHByb2plY3QpXG4gICAgKTtcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHByb2plY3Q6IFByb2plY3QpIHtcbiAgICBzdXBlcihwcm9qZWN0KTtcblxuICAgIC8vIEFkZCBhbGwgZW52aXJvbm1lbnQgdmFyaWFibGVzIHRvIHRoZSB0YXNrIGVudmlyb25tZW50XG4gICAgT2JqZWN0LmVudHJpZXMoZ2V0VHlwZVNhZmVBcGlUYXNrRW52aXJvbm1lbnQoKSkuZm9yRWFjaCgoW2tleSwgdmFsdWVdKSA9PlxuICAgICAgcHJvamVjdC50YXNrcy5hZGRFbnZpcm9ubWVudChrZXksIHZhbHVlKVxuICAgICk7XG5cbiAgICAvLyBUeXBlU2NyaXB0IHByb2plY3RzIG5lZWQgYSBkZXYgZGVwZW5kZW5jeSBvbiBQREsgdG8gZW5zdXJlIG5weCByZXNvbHZlcyB0byB0aGUgY29ycmVjdCBwZGtcbiAgICBpZiAoUHJvamVjdFV0aWxzLmlzTmFtZWRJbnN0YW5jZU9mKHByb2plY3QsIFR5cGVTY3JpcHRQcm9qZWN0KSkge1xuICAgICAgcHJvamVjdC5hZGREZXZEZXBzKFwiQGF3cy9wZGtAXjBcIik7XG4gICAgfVxuICB9XG59XG4iXX0=