"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHandlersProjectVendorExtensions = exports.buildInvokeMockDataGeneratorCommand = exports.buildCleanOpenApiGeneratedCodeCommand = exports.buildInvokeOpenApiGeneratorCommandArgs = exports.buildCodegenCommandArgs = exports.buildTypeSafeApiExecCommand = exports.getTypeSafeApiTaskEnvironment = exports.TypeSafeApiScript = exports.OtherGenerators = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../../../monorepo");
const runtime_version_utils_1 = require("../runtime-version-utils");
/**
 * Enum for generator directories for non-runtime generators
 */
var OtherGenerators;
(function (OtherGenerators) {
    OtherGenerators["DOCS"] = "docs";
    // Infrastructure
    OtherGenerators["TYPESCRIPT_CDK_INFRASTRUCTURE"] = "typescript-cdk-infrastructure";
    OtherGenerators["PYTHON_CDK_INFRASTRUCTURE"] = "python-cdk-infrastructure";
    OtherGenerators["JAVA_CDK_INFRASTRUCTURE"] = "java-cdk-infrastructure";
    OtherGenerators["TYPESCRIPT_ASYNC_CDK_INFRASTRUCTURE"] = "typescript-async-cdk-infrastructure";
    OtherGenerators["PYTHON_ASYNC_CDK_INFRASTRUCTURE"] = "python-async-cdk-infrastructure";
    OtherGenerators["JAVA_ASYNC_CDK_INFRASTRUCTURE"] = "java-async-cdk-infrastructure";
    // Handlers
    OtherGenerators["TYPESCRIPT_LAMBDA_HANDLERS"] = "typescript-lambda-handlers";
    OtherGenerators["PYTHON_LAMBDA_HANDLERS"] = "python-lambda-handlers";
    OtherGenerators["JAVA_LAMBDA_HANDLERS"] = "java-lambda-handlers";
    OtherGenerators["TYPESCRIPT_ASYNC_LAMBDA_HANDLERS"] = "typescript-async-lambda-handlers";
    OtherGenerators["PYTHON_ASYNC_LAMBDA_HANDLERS"] = "python-async-lambda-handlers";
    OtherGenerators["JAVA_ASYNC_LAMBDA_HANDLERS"] = "java-async-lambda-handlers";
    // Async runtime
    OtherGenerators["TYPESCRIPT_ASYNC_RUNTIME"] = "typescript-async-runtime";
    OtherGenerators["PYTHON_ASYNC_RUNTIME"] = "python-async-runtime";
    OtherGenerators["JAVA_ASYNC_RUNTIME"] = "java-async-runtime";
})(OtherGenerators || (exports.OtherGenerators = OtherGenerators = {}));
/**
 * Built in scripts.
 * If adding a script here, ensure you map it in TypeSafeApiProject (in /projenrc/projects)
 */
var TypeSafeApiScript;
(function (TypeSafeApiScript) {
    TypeSafeApiScript["PARSE_OPENAPI_SPEC"] = "type-safe-api parse-openapi-spec";
    TypeSafeApiScript["GENERATE"] = "type-safe-api.generate";
    TypeSafeApiScript["GENERATE_NEXT"] = "type-safe-api generate";
    TypeSafeApiScript["GENERATE_MOCK_DATA"] = "type-safe-api generate-mock-data";
    TypeSafeApiScript["CLEAN_OPENAPI_GENERATED_CODE"] = "type-safe-api.clean-openapi-generated-code";
    TypeSafeApiScript["COPY_GRADLE_WRAPPER"] = "type-safe-api copy-gradle-wrapper";
    TypeSafeApiScript["COPY_ASYNC_SMITHY_TRANSFORMER"] = "type-safe-api copy-async-smithy-transformer";
    TypeSafeApiScript["GENERATE_ASYNCAPI_SPEC"] = "type-safe-api generate-asyncapi-spec";
})(TypeSafeApiScript || (exports.TypeSafeApiScript = TypeSafeApiScript = {}));
/**
 * Return the environment that should be used for executing type safe api commands
 */
const getTypeSafeApiTaskEnvironment = () => ({
    AWS_PDK_VERSION: monorepo_1.ProjectUtils.getPdkVersion(),
});
exports.getTypeSafeApiTaskEnvironment = getTypeSafeApiTaskEnvironment;
/**
 * Build a command for running a script from this project's bin
 */
const buildTypeSafeApiExecCommand = (script, args) => {
    return `npx --yes -p @aws/pdk@$AWS_PDK_VERSION ${script}${args ? ` ${args}` : ""}`;
};
exports.buildTypeSafeApiExecCommand = buildTypeSafeApiExecCommand;
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
const buildCodegenCommandArgs = (options) => {
    const metadata = options.metadata
        ? ` --metadata '${JSON.stringify(options.metadata)}'`
        : "";
    return `--specPath ${options.specPath} --outputPath . --templateDirs ${options.templateDirs
        .map((t) => `"${t}"`)
        .join(" ")}${metadata}`;
};
exports.buildCodegenCommandArgs = buildCodegenCommandArgs;
/**
 * Generate code or docs by invoking the root generate script
 */
const buildInvokeOpenApiGeneratorCommandArgs = (options) => {
    const srcDir = options.srcDir ?? "src";
    const tstDir = options.tstDir ?? "test";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    const normalizers = options.normalizers
        ? ` --openapi-normalizer "${serializeProperties(Object.fromEntries(Object.entries(options.normalizers).map(([k, v]) => [k, `${v}`])))}"`
        : "";
    const extensions = options.extraVendorExtensions
        ? ` --extra-vendor-extensions '${JSON.stringify(options.extraVendorExtensions)}'`
        : "";
    const generateAliasAsModel = options.generateAliasAsModel ?? true ? " --generate-alias-as-model" : "";
    const specPath = options.specPath;
    const outputPath = ".";
    return `--generator ${options.generator} --spec-path ${specPath} --output-path ${outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir} --tst-dir ${tstDir}${additionalProperties}${normalizers}${extensions}${generateAliasAsModel}`;
};
exports.buildInvokeOpenApiGeneratorCommandArgs = buildInvokeOpenApiGeneratorCommandArgs;
/**
 * Builds a command to clean up files which were previously generated by openapi generator
 */
const buildCleanOpenApiGeneratedCodeCommand = () => {
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.CLEAN_OPENAPI_GENERATED_CODE, `--code-path .`);
};
exports.buildCleanOpenApiGeneratedCodeCommand = buildCleanOpenApiGeneratedCodeCommand;
/**
 * Invoke the mock data generator script
 */
const buildInvokeMockDataGeneratorCommand = (options) => {
    const outputPath = options.outputSubDir ?? ".";
    const locale = options.locale ? ` --locale ${options.locale}` : "";
    const maxArrayLength = options.maxArrayLength !== undefined
        ? ` --maxArrayLength ${options.maxArrayLength}`
        : "";
    const seed = options.seed !== undefined ? ` --seed ${options.seed}` : "";
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.GENERATE_MOCK_DATA, `--specPath ${options.specPath} --outputPath ${outputPath}${locale}${maxArrayLength}${seed}`);
};
exports.buildInvokeMockDataGeneratorCommand = buildInvokeMockDataGeneratorCommand;
/**
 * Return vendor extensions containing details about the handler projects
 */
const getHandlersProjectVendorExtensions = (targetProject, { java, python, typescript }) => ({
    "x-handlers-python-module": python?.moduleName ?? "",
    "x-handlers-java-package": java?.packageName ?? "",
    "x-handlers-typescript-asset-path": typescript
        ? path.join(path.relative(targetProject.outdir, typescript.outdir), "dist", "lambda")
        : "",
    "x-handlers-python-asset-path": python
        ? path.join(path.relative(targetProject.outdir, python.outdir), "dist", "lambda")
        : "",
    "x-handlers-java-asset-path": java
        ? path.join(path.relative(targetProject.outdir, java.outdir), java.distdir, ...java.pom.groupId.split("."), java.pom.artifactId, java.pom.version, `${java.pom.artifactId}-${java.pom.version}.jar`)
        : "",
    "x-handlers-node-lambda-runtime-version": typescript
        ? runtime_version_utils_1.RuntimeVersionUtils.NODE.getLambdaRuntime(typescript.runtimeVersion)
        : "",
    "x-handlers-python-lambda-runtime-version": python
        ? runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getLambdaRuntime(python.runtimeVersion)
        : "",
    "x-handlers-java-lambda-runtime-version": java
        ? runtime_version_utils_1.RuntimeVersionUtils.JAVA.getLambdaRuntime(java.runtimeVersion)
        : "",
});
exports.getHandlersProjectVendorExtensions = getHandlersProjectVendorExtensions;
//# sourceMappingURL=data:application/json;base64,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