"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateDocsProjects = exports.generateAsyncLibraryProjects = exports.generateLibraryProjects = exports.generateAsyncRuntimeProjects = exports.generateRuntimeProjects = exports.generateAsyncHandlersProjects = exports.generateHandlersProjects = exports._generateInfraProject = exports.generateAsyncInfraProject = exports.generateInfraProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const generated_asyncapi_html_documentation_project_1 = require("./documentation/generated-asyncapi-html-documentation-project");
const generated_asyncapi_markdown_documentation_project_1 = require("./documentation/generated-asyncapi-markdown-documentation-project");
const generated_html_redoc_documentation_project_1 = require("./documentation/generated-html-redoc-documentation-project");
const generated_html2_documentation_project_1 = require("./documentation/generated-html2-documentation-project");
const generated_markdown_documentation_project_1 = require("./documentation/generated-markdown-documentation-project");
const generated_plantuml_documentation_project_1 = require("./documentation/generated-plantuml-documentation-project");
const generated_java_async_handlers_project_1 = require("./handlers/generated-java-async-handlers-project");
const generated_java_handlers_project_1 = require("./handlers/generated-java-handlers-project");
const generated_python_async_handlers_project_1 = require("./handlers/generated-python-async-handlers-project");
const generated_python_handlers_project_1 = require("./handlers/generated-python-handlers-project");
const generated_typescript_async_handlers_project_1 = require("./handlers/generated-typescript-async-handlers-project");
const generated_typescript_handlers_project_1 = require("./handlers/generated-typescript-handlers-project");
const generated_java_async_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-java-async-cdk-infrastructure-project");
const generated_java_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-java-cdk-infrastructure-project");
const generated_python_async_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-python-async-cdk-infrastructure-project");
const generated_python_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-python-cdk-infrastructure-project");
const generated_typescript_async_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-typescript-async-cdk-infrastructure-project");
const generated_typescript_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-typescript-cdk-infrastructure-project");
const typescript_react_query_hooks_library_1 = require("./library/typescript-react-query-hooks-library");
const typescript_websocket_client_library_1 = require("./library/typescript-websocket-client-library");
const typescript_websocket_hooks_library_1 = require("./library/typescript-websocket-hooks-library");
const generated_java_async_runtime_project_1 = require("./runtime/generated-java-async-runtime-project");
const generated_java_runtime_project_1 = require("./runtime/generated-java-runtime-project");
const generated_python_async_runtime_project_1 = require("./runtime/generated-python-async-runtime-project");
const generated_python_runtime_project_1 = require("./runtime/generated-python-runtime-project");
const generated_typescript_async_runtime_project_1 = require("./runtime/generated-typescript-async-runtime-project");
const generated_typescript_runtime_project_1 = require("./runtime/generated-typescript-runtime-project");
const languages_1 = require("../languages");
const logger = (0, log4js_1.getLogger)();
// No dashes or underscores since this is used in the package name in imports
const sanitiseJavaProjectName = (name) => name.replace(/@/g, "").replace(/[\-/_]/g, "");
// kebab-case for java artifact ids
const sanitiseJavaArtifactId = (name) => name.replace(/@/g, "").replace(/[/_]/g, "-");
// kebab-case for typescript packages
const sanitiseTypescriptPackageName = (name) => name.replace(/_/g, "-");
// snake_case for python modules
const sanitisePythonModuleName = (name) => name.replace(/@/g, "").replace(/[\-/]/g, "_");
// Use dashes in project name since distributable's PKG-INFO always converts _ to -
// https://stackoverflow.com/questions/36300788/python-package-wheel-pkg-info-name
const sanitisePythonPackageName = (name) => name.replace(/@/g, "").replace(/[_/]/g, "-");
const generateLanguageProject = (suffix, language, projectConstructors, options) => {
    const packageName = `${options.parentPackageName}-${language}-${suffix}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, language),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (language) {
        case languages_1.Language.TYPESCRIPT: {
            logger.trace(`Attempting to generate TYPESCRIPT ${suffix} project.`);
            return new projectConstructors[language]({
                ...options,
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.Language.PYTHON: {
            logger.trace(`Attempting to generate PYTHON ${suffix} project.`);
            return new projectConstructors[language]({
                ...options,
                ...commonOptions,
                name: sanitisePythonPackageName(packageName),
                moduleName: sanitisePythonModuleName(packageName),
                ...options.pythonOptions,
            });
        }
        case languages_1.Language.JAVA: {
            logger.trace(`Attempting to generate JAVA ${suffix} project.`);
            return new projectConstructors[language]({
                ...options,
                ...commonOptions,
                name: sanitiseJavaProjectName(packageName),
                artifactId: sanitiseJavaArtifactId(packageName),
                groupId: "com.generated.api",
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown ${suffix} language ${language}`);
    }
};
const generateLanguageProjects = (suffix, languages, projectConstructors, options) => {
    const projects = {};
    languages.forEach((language) => {
        const project = generateLanguageProject(suffix, language, projectConstructors, options);
        if (project != null) {
            projects[language] = project;
        }
    });
    return projects;
};
/**
 * Returns a generated infrastructure project for the given language
 */
const generateInfraProject = (language, options) => {
    return (0, exports._generateInfraProject)(language, {
        [languages_1.Language.JAVA]: generated_java_cdk_infrastructure_project_1.GeneratedJavaCdkInfrastructureProject,
        [languages_1.Language.PYTHON]: generated_python_cdk_infrastructure_project_1.GeneratedPythonCdkInfrastructureProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_cdk_infrastructure_project_1.GeneratedTypescriptCdkInfrastructureProject,
    }, options);
};
exports.generateInfraProject = generateInfraProject;
/**
 * Returns a generated async infrastructure project for the given language
 */
const generateAsyncInfraProject = (language, options) => {
    return (0, exports._generateInfraProject)(language, {
        [languages_1.Language.JAVA]: generated_java_async_cdk_infrastructure_project_1.GeneratedJavaAsyncCdkInfrastructureProject,
        [languages_1.Language.PYTHON]: generated_python_async_cdk_infrastructure_project_1.GeneratedPythonAsyncCdkInfrastructureProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_async_cdk_infrastructure_project_1.GeneratedTypescriptAsyncCdkInfrastructureProject,
    }, options);
};
exports.generateAsyncInfraProject = generateAsyncInfraProject;
const _generateInfraProject = (language, projectConstructors, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Infrastructure",
            "",
            "This directory contains a generated type-safe CDK construct which can provision the API gateway infrastructure for an API based on your model.",
        ],
        readonly: true,
    });
    return generateLanguageProject("infra", language, projectConstructors, {
        ...options,
        typescriptOptions: {
            ...options.typescriptOptions,
            generatedTypescriptTypes: options.generatedRuntimes.typescript,
        },
        pythonOptions: {
            ...options.pythonOptions,
            generatedPythonTypes: options.generatedRuntimes.python,
        },
        javaOptions: {
            ...options.javaOptions,
            generatedJavaTypes: options.generatedRuntimes.java,
        },
    });
};
exports._generateInfraProject = _generateInfraProject;
/**
 * Create handlers projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateHandlersProjects = (languages, options) => {
    return _generateHandlersProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_handlers_project_1.GeneratedJavaHandlersProject,
        [languages_1.Language.PYTHON]: generated_python_handlers_project_1.GeneratedPythonHandlersProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_handlers_project_1.GeneratedTypescriptHandlersProject,
    }, options);
};
exports.generateHandlersProjects = generateHandlersProjects;
/**
 * Create async handlers projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateAsyncHandlersProjects = (languages, options) => {
    return _generateHandlersProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_async_handlers_project_1.GeneratedJavaAsyncHandlersProject,
        [languages_1.Language.PYTHON]: generated_python_async_handlers_project_1.GeneratedPythonAsyncHandlersProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_async_handlers_project_1.GeneratedTypescriptAsyncHandlersProject,
    }, options);
};
exports.generateAsyncHandlersProjects = generateAsyncHandlersProjects;
const _generateHandlersProjects = (languages, projectConstructors, options) => {
    if (languages.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Handlers",
                "",
                "This directory contains lambda handlers for implementing your API.",
                "",
                "Whenever an operation is annotated with the `@handler` trait in Smithy (or the `x-handler` vendor extension in OpenAPI), a stub handler implementation will be generated for you, which you are free to modify.",
            ],
            readonly: true,
        });
    }
    return generateLanguageProjects("handlers", languages, projectConstructors, {
        ...options,
        typescriptOptions: {
            ...options.typescriptOptions,
            generatedTypescriptTypes: options.generatedRuntimes.typescript,
        },
        pythonOptions: {
            ...options.pythonOptions,
            generatedPythonTypes: options.generatedRuntimes.python,
        },
        javaOptions: {
            ...options.javaOptions,
            generatedJavaTypes: options.generatedRuntimes.java,
        },
    });
};
/**
 * Create runtime projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateRuntimeProjects = (languages, options) => {
    return _generateRuntimeProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_runtime_project_1.GeneratedJavaRuntimeProject,
        [languages_1.Language.PYTHON]: generated_python_runtime_project_1.GeneratedPythonRuntimeProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_runtime_project_1.GeneratedTypescriptRuntimeProject,
    }, options);
};
exports.generateRuntimeProjects = generateRuntimeProjects;
/**
 * Create async runtime projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateAsyncRuntimeProjects = (languages, options) => {
    return _generateRuntimeProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_async_runtime_project_1.GeneratedJavaAsyncRuntimeProject,
        [languages_1.Language.PYTHON]: generated_python_async_runtime_project_1.GeneratedPythonAsyncRuntimeProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_async_runtime_project_1.GeneratedTypescriptAsyncRuntimeProject,
    }, options);
};
exports.generateAsyncRuntimeProjects = generateAsyncRuntimeProjects;
const _generateRuntimeProjects = (languages, projectConstructors, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Runtimes",
            "",
            "This directory contains generated runtime projects based on your API model.",
            "",
            "Each runtime project includes types from your API model, as well as type-safe client and server code.",
        ],
        readonly: true,
    });
    return generateLanguageProjects("runtime", languages, projectConstructors, options);
};
/**
 * Returns a generated client project for the given language
 */
const generateLibraryProject = (library, options) => {
    const packageName = `${options.parentPackageName}-${library}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, library),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (library) {
        case languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS: {
            return new typescript_react_query_hooks_library_1.TypescriptReactQueryHooksLibrary({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptReactQueryHooksOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        default:
            throw new Error(`Unknown library ${library}`);
    }
};
/**
 * Create library projects
 * @param libraries the libraries to generate for
 * @param options options for the projects to be created
 */
const generateLibraryProjects = (libraries, options) => {
    if (libraries.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Libraries",
                "",
                "This directory contains generated libraries based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedLibraries = {};
    libraries.forEach((library) => {
        const project = generateLibraryProject(library, options);
        if (project != null) {
            generatedLibraries[library] = project;
        }
    });
    return generatedLibraries;
};
exports.generateLibraryProjects = generateLibraryProjects;
/**
 * Returns a generated client project for the given language
 */
const generateAsyncLibraryProject = (library, options) => {
    const packageName = `${options.parentPackageName}-${library}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, library),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (library) {
        case languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT: {
            return new typescript_websocket_client_library_1.TypescriptWebsocketClientLibrary({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptWebSocketClientOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS: {
            return new typescript_websocket_hooks_library_1.TypescriptWebsocketHooksLibrary({
                ...commonOptions,
                clientPackageName: options.typescriptWebSocketHooksOptions.clientPackageName ??
                    sanitiseTypescriptPackageName(`${options.parentPackageName}-${languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT}`),
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptWebSocketClientOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        default:
            throw new Error(`Unknown library ${library}`);
    }
};
/**
 * Create library projects
 * @param libraries the libraries to generate for
 * @param options options for the projects to be created
 */
const generateAsyncLibraryProjects = (libraries, options) => {
    if (libraries.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Libraries",
                "",
                "This directory contains generated libraries based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedLibraries = {};
    libraries.forEach((library) => {
        const project = generateAsyncLibraryProject(library, options);
        if (project != null) {
            generatedLibraries[library] = project;
        }
    });
    return generatedLibraries;
};
exports.generateAsyncLibraryProjects = generateAsyncLibraryProjects;
const generateDocsProject = (format, options) => {
    const commonProps = {
        name: `${options.parentPackageName}-documentation-${format.replace(/_/g, "-")}`,
        parent: options.parent,
        outdir: path.join(options.generatedDocsDir, format),
        specPath: options.parsedSpecPath,
    };
    switch (format) {
        case languages_1.DocumentationFormat.HTML2: {
            return new generated_html2_documentation_project_1.GeneratedHtml2DocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.html2,
            });
        }
        case languages_1.DocumentationFormat.HTML_REDOC: {
            return new generated_html_redoc_documentation_project_1.GeneratedHtmlRedocDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.htmlRedoc,
            });
        }
        case languages_1.DocumentationFormat.MARKDOWN: {
            return new generated_markdown_documentation_project_1.GeneratedMarkdownDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.markdown,
            });
        }
        case languages_1.DocumentationFormat.PLANTUML: {
            return new generated_plantuml_documentation_project_1.GeneratedPlantumlDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.plantuml,
            });
        }
        case languages_1.WebSocketDocumentationFormat.HTML: {
            return new generated_asyncapi_html_documentation_project_1.GeneratedAsyncApiHtmlDocumentationProject({
                ...commonProps,
                ...options.asyncDocumentationOptions?.html,
            });
        }
        case languages_1.WebSocketDocumentationFormat.MARKDOWN: {
            return new generated_asyncapi_markdown_documentation_project_1.GeneratedAsyncApiMarkdownDocumentationProject({
                ...commonProps,
                ...options.asyncDocumentationOptions?.markdown,
            });
        }
        default:
            throw new Error(`Unknown documentation format ${format}`);
    }
};
const generateDocsProjects = (formats, options) => {
    if (formats.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedDocsDir, "README.md"), {
            lines: [
                "## Generated Documentation",
                "",
                "This directory contains generated documentation based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedClients = {};
    formats.forEach((format) => {
        const project = generateDocsProject(format, options);
        if (project != null) {
            generatedClients[format] = project;
        }
    });
    return generatedClients;
};
exports.generateDocsProjects = generateDocsProjects;
//# sourceMappingURL=data:application/json;base64,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