"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptCdkInfrastructureBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedTypescriptCdkInfrastructureBaseProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            jest: false,
            eslint: false,
            prettier: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        /**
         * Path to the packaged copy of the openapi specification
         */
        this.packagedSpecPath = "assets/api.json";
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.addDevDeps("@types/aws-lambda");
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the types can be
            // resolved
            options.isWithinMonorepo
                ? `${options.generatedTypescriptTypes.package.packageName}@${options.generatedTypescriptTypes.package.manifest.version}`
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        const devAndPeerDeps = ["@aws/pdk", "constructs", "aws-cdk-lib", "cdk-nag"];
        this.addDevDeps(...devAndPeerDeps.filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.BUILD)));
        this.addPeerDeps(...devAndPeerDeps.filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.PEER)));
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        const generateTask = this.addTask("generate");
        this.generateTask = generateTask;
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE_NEXT, this.buildGenerateCommandArgs()));
        // Copy the api spec to within the package
        generateTask.exec(`mkdir -p ${path.dirname(this.packagedSpecPath)}`);
        generateTask.exec(`cp -f ${this.options.specPath} ${this.packagedSpecPath}`);
        if (!options.commitGeneratedCode) {
            this.gitignore.addPatterns(`/${this.packagedSpecPath}`);
        }
        this.preCompileTask.spawn(generateTask);
        if (!options.commitGeneratedCode) {
            // Ignore the generated code
            this.gitignore.addPatterns(this.srcdir, "mocks");
        }
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-manifest");
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptCdkInfrastructureBaseProject = GeneratedTypescriptCdkInfrastructureBaseProject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2VuZXJhdGVkLXR5cGVzY3JpcHQtY2RrLWluZnJhc3RydWN0dXJlLWJhc2UtcHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdlbmVyYXRlZC10eXBlc2NyaXB0LWNkay1pbmZyYXN0cnVjdHVyZS1iYXNlLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7c0NBQ3NDO0FBQ3RDLDZCQUE2QjtBQUM3QixtQ0FBMEQ7QUFDMUQsc0RBQTJEO0FBQzNELHNEQUcrQjtBQUsvQiwwR0FBbUc7QUFDbkcsa0RBS2dDO0FBMkJoQyxNQUFzQiwrQ0FBZ0QsU0FBUSw4QkFBaUI7SUFhN0YsWUFBWSxPQUErRDtRQUN6RSxLQUFLLENBQUM7WUFDSixHQUFJLE9BQWU7WUFDbkIsVUFBVSxFQUFFLEtBQUs7WUFDakIsSUFBSSxFQUFFLEtBQUs7WUFDWCxNQUFNLEVBQUUsS0FBSztZQUNiLFFBQVEsRUFBRSxLQUFLO1lBQ2YsUUFBUSxFQUFFO2dCQUNSLGVBQWUsRUFBRTtvQkFDZixHQUFHLEVBQUUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDO29CQUN0QixvRUFBb0U7b0JBQ3BFLGNBQWMsRUFBRSxLQUFLO29CQUNyQixrQkFBa0IsRUFBRSxLQUFLO29CQUN6QixZQUFZLEVBQUUsSUFBSTtvQkFDbEIsR0FBRyxPQUFPLEVBQUUsUUFBUSxFQUFFLGVBQWU7aUJBQ3RDO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRSxLQUFLO1NBQ3hCLENBQUMsQ0FBQztRQXpCTDs7V0FFRztRQUNnQixxQkFBZ0IsR0FBRyxpQkFBaUIsQ0FBQztRQStHakQsNkJBQXdCLEdBQUcsR0FBRyxFQUFFO1lBQ3JDLE9BQU8sSUFBQSwrQkFBdUIsRUFBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBQzdELENBQUMsQ0FBQztRQTFGQSxpRUFBNkIsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFFdkIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxPQUFPLENBQ1YsR0FBRztZQUNELDhHQUE4RztZQUM5RyxXQUFXO1lBQ1gsT0FBTyxDQUFDLGdCQUFnQjtnQkFDdEIsQ0FBQyxDQUFDLEdBQUcsT0FBTyxDQUFDLHdCQUF3QixDQUFDLE9BQU8sQ0FBQyxXQUFXLElBQUksT0FBTyxDQUFDLHdCQUF3QixDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFO2dCQUN4SCxDQUFDLENBQUMsR0FDRSxPQUFPLENBQUMsd0JBQXdCLENBQUMsT0FBTyxDQUFDLFdBQzNDLFNBQVMsSUFBSSxDQUFDLFFBQVEsQ0FDcEIsSUFBSSxDQUFDLE1BQU0sRUFDWCxPQUFPLENBQUMsd0JBQXdCLENBQUMsTUFBTSxDQUN4QyxFQUFFO1NBQ1IsQ0FBQyxNQUFNLENBQ04sQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsdUJBQWMsQ0FBQyxPQUFPLENBQUMsQ0FDbEUsQ0FDRixDQUFDO1FBRUYsTUFBTSxjQUFjLEdBQUcsQ0FBQyxVQUFVLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM1RSxJQUFJLENBQUMsVUFBVSxDQUNiLEdBQUcsY0FBYyxDQUFDLE1BQU0sQ0FDdEIsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsdUJBQWMsQ0FBQyxLQUFLLENBQUMsQ0FDaEUsQ0FDRixDQUFDO1FBQ0YsSUFBSSxDQUFDLFdBQVcsQ0FDZCxHQUFHLGNBQWMsQ0FBQyxNQUFNLENBQ3RCLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLHVCQUFjLENBQUMsSUFBSSxDQUFDLENBQy9ELENBQ0YsQ0FBQztRQUVGLDBEQUEwRDtRQUMxRCxNQUFNLFNBQVMsR0FBRyxJQUFJLG1CQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ3JELFNBQVMsQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUVwRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO1FBQ2pDLFlBQVksQ0FBQyxJQUFJLENBQ2YsSUFBQSxtQ0FBMkIsRUFDekIseUJBQWlCLENBQUMsYUFBYSxFQUMvQixJQUFJLENBQUMsd0JBQXdCLEVBQUUsQ0FDaEMsQ0FDRixDQUFDO1FBRUYsMENBQTBDO1FBQzFDLFlBQVksQ0FBQyxJQUFJLENBQUMsWUFBWSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUNyRSxZQUFZLENBQUMsSUFBSSxDQUNmLFNBQVMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQzFELENBQUM7UUFDRixJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLENBQUM7WUFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQzFELENBQUM7UUFFRCxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUV4QyxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLENBQUM7WUFDakMsNEJBQTRCO1lBQzVCLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDbkQsQ0FBQztRQUVELElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLG9CQUFvQixFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFFcEUsaUhBQWlIO1FBQ2pILElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztZQUM5QixRQUFRLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLENBQUM7Z0JBQ3BDLEtBQUssK0JBQWtCLENBQUMsSUFBSTtvQkFDMUIsSUFBSSxDQUFDLEtBQUs7eUJBQ1AsT0FBTyxDQUFDLFNBQVMsQ0FBQzt3QkFDbkIsRUFBRSxXQUFXLENBQ1gsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsVUFBVSxJQUFJLENBQUMsUUFBUSxDQUNuRCxJQUFJLENBQUMsTUFBTSxFQUNYLElBQUksQ0FBQyxPQUFPLENBQUMsd0JBQXdCLENBQUMsTUFBTSxDQUM3QyxFQUFFLENBQ0osQ0FBQztvQkFDSixNQUFNO2dCQUNSO29CQUNFLElBQUksQ0FBQyxLQUFLO3lCQUNQLE9BQU8sQ0FBQyxTQUFTLENBQUM7d0JBQ25CLEVBQUUsV0FBVyxDQUNYLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLFNBQVMsSUFBSSxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLENBQ25HLENBQUM7b0JBQ0osTUFBTTtZQUNWLENBQUM7UUFDSCxDQUFDO0lBQ0gsQ0FBQztDQU9GO0FBN0hELDBHQTZIQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgRGVwZW5kZW5jeVR5cGUsIElnbm9yZUZpbGUsIFRhc2sgfSBmcm9tIFwicHJvamVuXCI7XG5pbXBvcnQgeyBOb2RlUGFja2FnZU1hbmFnZXIgfSBmcm9tIFwicHJvamVuL2xpYi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQge1xuICBUeXBlU2NyaXB0UHJvamVjdCxcbiAgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zLFxufSBmcm9tIFwicHJvamVuL2xpYi90eXBlc2NyaXB0XCI7XG5pbXBvcnQge1xuICBDb2RlR2VuZXJhdGlvblNvdXJjZU9wdGlvbnMsXG4gIEdlbmVyYXRlZFdpdGhPcGVuQXBpR2VuZXJhdG9yT3B0aW9ucyxcbn0gZnJvbSBcIi4uLy4uLy4uL3R5cGVzXCI7XG5pbXBvcnQgeyBUeXBlU2FmZUFwaUNvbW1hbmRFbnZpcm9ubWVudCB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL3R5cGUtc2FmZS1hcGktY29tbWFuZC1lbnZpcm9ubWVudFwiO1xuaW1wb3J0IHtcbiAgYnVpbGRDb2RlZ2VuQ29tbWFuZEFyZ3MsXG4gIGJ1aWxkVHlwZVNhZmVBcGlFeGVjQ29tbWFuZCxcbiAgQ29kZWdlbk9wdGlvbnMsXG4gIFR5cGVTYWZlQXBpU2NyaXB0LFxufSBmcm9tIFwiLi4vLi4vY29tcG9uZW50cy91dGlsc1wiO1xuaW1wb3J0IHsgR2VuZXJhdGVkSGFuZGxlcnNQcm9qZWN0cyB9IGZyb20gXCIuLi8uLi9nZW5lcmF0ZVwiO1xuaW1wb3J0IHsgR2VuZXJhdGVkVHlwZXNjcmlwdFJ1bnRpbWVCYXNlUHJvamVjdCB9IGZyb20gXCIuLi8uLi9ydW50aW1lL2dlbmVyYXRlZC10eXBlc2NyaXB0LXJ1bnRpbWUtYmFzZS1wcm9qZWN0XCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgR2VuZXJhdGVkVHlwZXNjcmlwdEluZnJhc3RydWN0dXJlQmFzZU9wdGlvbnNcbiAgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMsXG4gICAgR2VuZXJhdGVkV2l0aE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zIHt9XG5cbmV4cG9ydCBpbnRlcmZhY2UgR2VuZXJhdGVkVHlwZXNjcmlwdENka0luZnJhc3RydWN0dXJlQmFzZVByb2plY3RPcHRpb25zXG4gIGV4dGVuZHMgR2VuZXJhdGVkVHlwZXNjcmlwdEluZnJhc3RydWN0dXJlQmFzZU9wdGlvbnMsXG4gICAgQ29kZUdlbmVyYXRpb25Tb3VyY2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIEdlbmVyYXRlZCB0eXBlc2NyaXB0IHR5cGVzIHByb2plY3RcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRlZFR5cGVzY3JpcHRUeXBlczogR2VuZXJhdGVkVHlwZXNjcmlwdFJ1bnRpbWVCYXNlUHJvamVjdDtcblxuICAvKipcbiAgICogR2VuZXJhdGVkIGhhbmRsZXJzIHByb2plY3RzXG4gICAqL1xuICByZWFkb25seSBnZW5lcmF0ZWRIYW5kbGVyczogR2VuZXJhdGVkSGFuZGxlcnNQcm9qZWN0cztcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgaW5mcmFzdHJ1Y3R1cmUgYW5kIGNsaWVudCBwcm9qZWN0cyBhcmUgcGFyZW50ZWQgYnkgYW4gbW9ub3JlcG8gb3Igbm90XG4gICAqL1xuICByZWFkb25seSBpc1dpdGhpbk1vbm9yZXBvPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEdlbmVyYXRlZFR5cGVzY3JpcHRDZGtJbmZyYXN0cnVjdHVyZUJhc2VQcm9qZWN0IGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3Qge1xuICAvKipcbiAgICogT3B0aW9ucyBjb25maWd1cmVkIGZvciB0aGUgcHJvamVjdFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IG9wdGlvbnM6IEdlbmVyYXRlZFR5cGVzY3JpcHRDZGtJbmZyYXN0cnVjdHVyZUJhc2VQcm9qZWN0T3B0aW9ucztcblxuICAvKipcbiAgICogUGF0aCB0byB0aGUgcGFja2FnZWQgY29weSBvZiB0aGUgb3BlbmFwaSBzcGVjaWZpY2F0aW9uXG4gICAqL1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgcGFja2FnZWRTcGVjUGF0aCA9IFwiYXNzZXRzL2FwaS5qc29uXCI7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGdlbmVyYXRlVGFzazogVGFzaztcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBHZW5lcmF0ZWRUeXBlc2NyaXB0Q2RrSW5mcmFzdHJ1Y3R1cmVCYXNlUHJvamVjdE9wdGlvbnMpIHtcbiAgICBzdXBlcih7XG4gICAgICAuLi4ob3B0aW9ucyBhcyBhbnkpLFxuICAgICAgc2FtcGxlQ29kZTogZmFsc2UsXG4gICAgICBqZXN0OiBmYWxzZSxcbiAgICAgIGVzbGludDogZmFsc2UsXG4gICAgICBwcmV0dGllcjogZmFsc2UsXG4gICAgICB0c2NvbmZpZzoge1xuICAgICAgICBjb21waWxlck9wdGlvbnM6IHtcbiAgICAgICAgICBsaWI6IFtcImRvbVwiLCBcImVzMjAxOVwiXSxcbiAgICAgICAgICAvLyBHZW5lcmF0ZWQgY29kZSBpbXBvcnRzIGFsbCBtb2RlbHMsIGFuZCBtYXkgbm90IHJlZmVyZW5jZSB0aGVtIGFsbFxuICAgICAgICAgIG5vVW51c2VkTG9jYWxzOiBmYWxzZSxcbiAgICAgICAgICBub1VudXNlZFBhcmFtZXRlcnM6IGZhbHNlLFxuICAgICAgICAgIHNraXBMaWJDaGVjazogdHJ1ZSxcbiAgICAgICAgICAuLi5vcHRpb25zPy50c2NvbmZpZz8uY29tcGlsZXJPcHRpb25zLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIG5wbWlnbm9yZUVuYWJsZWQ6IGZhbHNlLFxuICAgIH0pO1xuICAgIFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50LmVuc3VyZSh0aGlzKTtcbiAgICB0aGlzLm9wdGlvbnMgPSBvcHRpb25zO1xuXG4gICAgdGhpcy5hZGREZXZEZXBzKFwiQHR5cGVzL2F3cy1sYW1iZGFcIik7XG4gICAgdGhpcy5hZGREZXBzKFxuICAgICAgLi4uW1xuICAgICAgICAvLyBJZiB3aXRoaW4gYSBtb25vcmVwbywgYWRkIGEgcmVndWxhciBkZXBlbmRlbmN5LiBPdGhlcndpc2UsIHVzZSBhIGZpbGUgZGVwZW5kZW5jeSB0byBlbnN1cmUgdGhlIHR5cGVzIGNhbiBiZVxuICAgICAgICAvLyByZXNvbHZlZFxuICAgICAgICBvcHRpb25zLmlzV2l0aGluTW9ub3JlcG9cbiAgICAgICAgICA/IGAke29wdGlvbnMuZ2VuZXJhdGVkVHlwZXNjcmlwdFR5cGVzLnBhY2thZ2UucGFja2FnZU5hbWV9QCR7b3B0aW9ucy5nZW5lcmF0ZWRUeXBlc2NyaXB0VHlwZXMucGFja2FnZS5tYW5pZmVzdC52ZXJzaW9ufWBcbiAgICAgICAgICA6IGAke1xuICAgICAgICAgICAgICBvcHRpb25zLmdlbmVyYXRlZFR5cGVzY3JpcHRUeXBlcy5wYWNrYWdlLnBhY2thZ2VOYW1lXG4gICAgICAgICAgICB9QGZpbGU6JHtwYXRoLnJlbGF0aXZlKFxuICAgICAgICAgICAgICB0aGlzLm91dGRpcixcbiAgICAgICAgICAgICAgb3B0aW9ucy5nZW5lcmF0ZWRUeXBlc2NyaXB0VHlwZXMub3V0ZGlyXG4gICAgICAgICAgICApfWAsXG4gICAgICBdLmZpbHRlcihcbiAgICAgICAgKGRlcCkgPT4gIXRoaXMuZGVwcy50cnlHZXREZXBlbmRlbmN5KGRlcCwgRGVwZW5kZW5jeVR5cGUuUlVOVElNRSlcbiAgICAgIClcbiAgICApO1xuXG4gICAgY29uc3QgZGV2QW5kUGVlckRlcHMgPSBbXCJAYXdzL3Bka1wiLCBcImNvbnN0cnVjdHNcIiwgXCJhd3MtY2RrLWxpYlwiLCBcImNkay1uYWdcIl07XG4gICAgdGhpcy5hZGREZXZEZXBzKFxuICAgICAgLi4uZGV2QW5kUGVlckRlcHMuZmlsdGVyKFxuICAgICAgICAoZGVwKSA9PiAhdGhpcy5kZXBzLnRyeUdldERlcGVuZGVuY3koZGVwLCBEZXBlbmRlbmN5VHlwZS5CVUlMRClcbiAgICAgIClcbiAgICApO1xuICAgIHRoaXMuYWRkUGVlckRlcHMoXG4gICAgICAuLi5kZXZBbmRQZWVyRGVwcy5maWx0ZXIoXG4gICAgICAgIChkZXApID0+ICF0aGlzLmRlcHMudHJ5R2V0RGVwZW5kZW5jeShkZXAsIERlcGVuZGVuY3lUeXBlLlBFRVIpXG4gICAgICApXG4gICAgKTtcblxuICAgIC8vIE1pbmltYWwgLm5wbWlnbm9yZSB0byBhdm9pZCBpbXBhY3RpbmcgT3BlbkFQSSBHZW5lcmF0b3JcbiAgICBjb25zdCBucG1pZ25vcmUgPSBuZXcgSWdub3JlRmlsZSh0aGlzLCBcIi5ucG1pZ25vcmVcIik7XG4gICAgbnBtaWdub3JlLmFkZFBhdHRlcm5zKFwiLy5wcm9qZW4vXCIsIFwiL3NyY1wiLCBcIi9kaXN0XCIpO1xuXG4gICAgY29uc3QgZ2VuZXJhdGVUYXNrID0gdGhpcy5hZGRUYXNrKFwiZ2VuZXJhdGVcIik7XG4gICAgdGhpcy5nZW5lcmF0ZVRhc2sgPSBnZW5lcmF0ZVRhc2s7XG4gICAgZ2VuZXJhdGVUYXNrLmV4ZWMoXG4gICAgICBidWlsZFR5cGVTYWZlQXBpRXhlY0NvbW1hbmQoXG4gICAgICAgIFR5cGVTYWZlQXBpU2NyaXB0LkdFTkVSQVRFX05FWFQsXG4gICAgICAgIHRoaXMuYnVpbGRHZW5lcmF0ZUNvbW1hbmRBcmdzKClcbiAgICAgIClcbiAgICApO1xuXG4gICAgLy8gQ29weSB0aGUgYXBpIHNwZWMgdG8gd2l0aGluIHRoZSBwYWNrYWdlXG4gICAgZ2VuZXJhdGVUYXNrLmV4ZWMoYG1rZGlyIC1wICR7cGF0aC5kaXJuYW1lKHRoaXMucGFja2FnZWRTcGVjUGF0aCl9YCk7XG4gICAgZ2VuZXJhdGVUYXNrLmV4ZWMoXG4gICAgICBgY3AgLWYgJHt0aGlzLm9wdGlvbnMuc3BlY1BhdGh9ICR7dGhpcy5wYWNrYWdlZFNwZWNQYXRofWBcbiAgICApO1xuICAgIGlmICghb3B0aW9ucy5jb21taXRHZW5lcmF0ZWRDb2RlKSB7XG4gICAgICB0aGlzLmdpdGlnbm9yZS5hZGRQYXR0ZXJucyhgLyR7dGhpcy5wYWNrYWdlZFNwZWNQYXRofWApO1xuICAgIH1cblxuICAgIHRoaXMucHJlQ29tcGlsZVRhc2suc3Bhd24oZ2VuZXJhdGVUYXNrKTtcblxuICAgIGlmICghb3B0aW9ucy5jb21taXRHZW5lcmF0ZWRDb2RlKSB7XG4gICAgICAvLyBJZ25vcmUgdGhlIGdlbmVyYXRlZCBjb2RlXG4gICAgICB0aGlzLmdpdGlnbm9yZS5hZGRQYXR0ZXJucyh0aGlzLnNyY2RpciwgXCJtb2Nrc1wiKTtcbiAgICB9XG5cbiAgICB0aGlzLmdpdGlnbm9yZS5hZGRQYXR0ZXJucyhcIi5vcGVuYXBpLWdlbmVyYXRvclwiLCBcIi50c2FwaS1tYW5pZmVzdFwiKTtcblxuICAgIC8vIElmIHdlJ3JlIG5vdCBpbiBhIG1vbm9yZXBvLCB3ZSBuZWVkIHRvIGxpbmsgdGhlIGdlbmVyYXRlZCB0eXBlcyBzdWNoIHRoYXQgdGhlIGxvY2FsIGRlcGVuZGVuY3kgY2FuIGJlIHJlc29sdmVkXG4gICAgaWYgKCFvcHRpb25zLmlzV2l0aGluTW9ub3JlcG8pIHtcbiAgICAgIHN3aXRjaCAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyKSB7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLlBOUE06XG4gICAgICAgICAgdGhpcy50YXNrc1xuICAgICAgICAgICAgLnRyeUZpbmQoXCJpbnN0YWxsXCIpXG4gICAgICAgICAgICA/LnByZXBlbmRFeGVjKFxuICAgICAgICAgICAgICBgJHt0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJ9IGxpbmsgLyR7cGF0aC5yZWxhdGl2ZShcbiAgICAgICAgICAgICAgICB0aGlzLm91dGRpcixcbiAgICAgICAgICAgICAgICB0aGlzLm9wdGlvbnMuZ2VuZXJhdGVkVHlwZXNjcmlwdFR5cGVzLm91dGRpclxuICAgICAgICAgICAgICApfWBcbiAgICAgICAgICAgICk7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgdGhpcy50YXNrc1xuICAgICAgICAgICAgLnRyeUZpbmQoXCJpbnN0YWxsXCIpXG4gICAgICAgICAgICA/LnByZXBlbmRFeGVjKFxuICAgICAgICAgICAgICBgJHt0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJ9IGxpbmsgJHt0aGlzLm9wdGlvbnMuZ2VuZXJhdGVkVHlwZXNjcmlwdFR5cGVzLnBhY2thZ2UucGFja2FnZU5hbWV9YFxuICAgICAgICAgICAgKTtcbiAgICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYnVpbGRHZW5lcmF0ZUNvbW1hbmRBcmdzID0gKCkgPT4ge1xuICAgIHJldHVybiBidWlsZENvZGVnZW5Db21tYW5kQXJncyh0aGlzLmJ1aWxkQ29kZWdlbk9wdGlvbnMoKSk7XG4gIH07XG5cbiAgcHJvdGVjdGVkIGFic3RyYWN0IGJ1aWxkQ29kZWdlbk9wdGlvbnMoKTogQ29kZWdlbk9wdGlvbnM7XG59XG4iXX0=