"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonRuntimeBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const python_1 = require("projen/lib/python");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Python project containing types generated using OpenAPI Generator CLI
 */
class GeneratedPythonRuntimeBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Add dependencies required by the client
        [
            "python-dateutil@~2.8.2",
            "pydantic@^2.5.2",
            "aenum@^3.1.11",
            "urllib3@~1.26.7",
            `aws-lambda-powertools@{extras=["tracer", "aws-sdk"],version="^2.28.0"}`,
            "python@^3.9",
        ].forEach((dep) => this.addDependency(dep));
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE_NEXT, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        if (!this.options.commitGeneratedCode) {
            // Ignore all the generated code
            this.gitignore.addPatterns(this.moduleName, "docs", "README.md");
        }
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-manifest");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py and README.md as our first install step to keep poetry happy until the
        // generator overwrites it.
        ["install", "install:ci"].forEach((t) => this.tasks
            .tryFind(t)
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py README.md`));
    }
}
exports.GeneratedPythonRuntimeBaseProject = GeneratedPythonRuntimeBaseProject;
//# sourceMappingURL=data:application/json;base64,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